
/*#if defined(NDEBUG)*/
/* #undef NDEBUG*/
/*#endif*/

#include "rt_alloc.h"
#include <OS.h>
#include <stdio.h>
#include <stdlib.h>
#include <Debug.h>

rtm_pool * _rtm_pool;
#if defined(__cplusplus)
extern "C" {
#endif
void rtm_dump_block(rtm_pool * pool, void * block);
#if defined(__cplusplus)
}
#endif
 

#if !NDEBUG
#define assert(x,y) do { if (!(x)) { fprintf(stderr, "%s:%d: assertion failed: %s\n", __FILE__, __LINE__, #x); printf y ; exit(1); } } while (0)
#else
#define assert(x,y) (void)0
#endif


typedef struct rtm_block rtm_block;

typedef struct rtm_free_list rtm_free_list;

struct rtm_free_list {
	rtm_free_list *
					next;
	rtm_free_list *
					prev;
};


struct rtm_pool {
	rtm_pool *	next;
	int32			lock_count;
	sem_id			lock_sem;
	area_id			area;
	int32			total_size;
	int32			free_size;
	int32			total_blocks;
	int32			free_blocks;
	rtm_free_list
					free_list;
	int32			min_free_size;
	int32			_reserved[5];
};

struct rtm_block {
	rtm_pool *	pool;
	rtm_block *
					prev;
	int32			phys_size;
	int32			log_size;	/* negative if free */
};
#define FREE_BLOCK -1
/* How much can we waste in a free block when allocating before we split into two blocks? */
#define PADDING_OK 48
/* Logical block sizes are rounded up to a multiple of PADSIZE (power of 2) */
#define PADSIZE 16


void * rtm_internal(rtm_pool * pool, size_t size);

static void rtm_validate_pool(rtm_pool * pool)
{
	area_info info;
	status_t err;
	rtm_block * bl, * en, * pr;
	rtm_free_list * fl, * fp;
	int total_free = 0;
	int total_blocks = 0;
	int free_size = 0;

	/* validate the pool */
	assert(pool != NULL,("NULL"));
	assert(pool->lock_count < 0,("%d\n", pool->lock_count));
	assert((err = get_area_info(pool->area, &info)) >= B_OK,("0x%x\n", err));
	assert(info.address == (void *)pool,("0x%x 0x%x\n", info.address, pool));
	assert(info.size == pool->total_size+sizeof(rtm_pool),("0x%x 0x%x", info.size, pool->total_size+sizeof(rtm_pool)));
	assert(pool->free_list.prev == NULL,("0x%x\n", pool->free_list.prev));
	assert(pool->free_blocks < pool->total_blocks || (pool->free_blocks == 1 && pool->total_blocks == 1), \
		("0x%x 0x%x\n", pool->free_blocks, pool->total_blocks));
	assert(pool->free_size <= pool->total_size-sizeof(rtm_block)*pool->free_blocks, \
		("0x%x 0x%x\n", pool->free_size, pool->total_size-sizeof(rtm_block)*pool->free_blocks));
	assert(pool->min_free_size <= pool->free_size,("0x%x, 0x%x\n", pool->min_free_size, pool->free_size));

	/* set up heap limits */
	bl = (rtm_block *)&pool[1];
	en = (rtm_block *)((char *)bl + pool->total_size);
	/* validate the free list */
	fl = &pool->free_list;
	fp = NULL;
	while (fl->next != NULL) {
		total_free++;
		if (total_free > pool->free_blocks+10) {
			break;	/* avoid infinite loops */
		}
		assert((char*)fl->next >= (char *)bl, ("0x%x 0x%x\n", fl->next, bl));
		assert((char*)fl->next < (char *)en, ("0x%x 0x%x\n", fl->next, en));
		assert(fl->next->prev == fl,("%d: 0x%x 0x%x 0x%x\n", total_free, fl, fl->next, fl->next->prev));
		assert(fl->prev == fp,("0x%x 0x%x 0x%x\n", fl, fl->prev, fp));
		pr = &((rtm_block *)fl->next)[-1];
		assert(pr->log_size == FREE_BLOCK,("0x%x 0x%x\n", pr, pr->log_size));
		free_size += pr->phys_size;
		fp = fl;
		fl = fl->next;
	}
	assert(total_free == pool->free_blocks,("%d %d\n", total_free, pool->free_blocks));
	assert(free_size == pool->free_size,("0x%x 0x%x\n", free_size, pool->free_size));

	/* validate the linear block count */
	assert(en > bl,("0x%x 0x%x\n", en, bl));
	pr = NULL;
	while (bl < en) {
		total_blocks++;
		assert(bl->phys_size >= sizeof(rtm_free_list),("0x%x 0x%x\n", bl, bl->phys_size));
		assert(!(bl->phys_size&(PADSIZE-1)), ("0x%x 0x%x\n", bl, bl->phys_size));
		assert(bl->log_size == FREE_BLOCK || bl->phys_size <= bl->log_size+PADSIZE+PADDING_OK,
			("0x%x 0x%x 0x%x\n", bl, bl->log_size, bl->phys_size));
		assert(bl->log_size >= FREE_BLOCK,("0x%x 0x%x\n", bl, bl->log_size));
		assert(bl->prev == pr, ("0x%x 0x%x\n", bl, bl->prev, pr));
		assert(bl->pool == pool, ("0x%x 0x%x 0x%x\n", bl, bl->pool, pool));
		pr = bl;
		bl = (rtm_block *)((char *)&bl[1]+bl->phys_size);
	}
	assert(bl == en,("0x%x 0x%x\n", bl, en));
	assert(total_blocks == pool->total_blocks,("%d %d\n", total_blocks, pool->total_blocks));
}


status_t rtm_create_pool(rtm_pool ** out_pool, size_t total_size)
{
	rtm_pool p;
	rtm_block b;
	sem_id sem = -1;
	area_id area = -1;
	void * start = NULL;

	if (out_pool == NULL)
	{
		out_pool = &_rtm_pool;
		if (_rtm_pool != NULL)
			return EALREADY;
	}

	sem = create_sem(0, "_rtm_pool_area_");
	if (sem < 0) {
		goto bail;
	}
	total_size = (total_size+sizeof(rtm_pool)+sizeof(rtm_block)+
		(B_PAGE_SIZE-1)) & ~(B_PAGE_SIZE-1);
	area = create_area("_rtm_pool_area_", &start, B_ANY_ADDRESS, total_size,
		B_FULL_LOCK, B_READ_AREA | B_WRITE_AREA);
	if (area < 0 || !start) goto bail;

	p.next = 0;
	p.lock_count = 0;
	p.lock_sem = sem;
	p.area = area;
	p.total_size = total_size-sizeof(rtm_pool);
	p.free_size = p.total_size-sizeof(rtm_block);
	p.min_free_size = p.free_size;
	p.total_blocks = 1;
	p.free_blocks = 1;
	*out_pool = (rtm_pool *)start;
	/*	User data (and free list data) live after the block headers.	*/
	p.free_list.next = (rtm_free_list *)&((rtm_block *)&(*out_pool)[1])[1];
	p.free_list.prev = NULL;
	p.free_list.next->next = NULL;
	p.free_list.next->prev = &(**out_pool).free_list;
	**out_pool = p;
	b.pool = *out_pool;
	b.prev = NULL;
	b.phys_size = p.free_size;	/* size of user block */
	b.log_size = FREE_BLOCK;
	*((rtm_block *)&(*out_pool)[1]) = b;
	return B_OK;

bail:
	if (sem >= 0) delete_sem(sem);
	if (area >= 0) delete_sem(area);
	return (sem < 0) ? sem : (area < 0) ? area : B_ERROR;
}

status_t rtm_delete_pool(rtm_pool * pool)
{
	if (!pool) return B_BAD_VALUE;
	
	while (pool) {
		status_t err;
		rtm_pool *next_pool = pool->next;
		
		assert(pool->area > 0,("0x%x", pool->area));
		if (getenv("MEDIA_LOCKED_MEMORY_DEBUG") != 0) {
			rtm_dump_block(pool, NULL);
		}
	
		err = delete_area(pool->area);
		if (err)
			return err;
			
		pool = next_pool;
	}
	
	return B_OK;
}

void * rtm_alloc(rtm_pool * pool, size_t size)
{
	const char *ptr;
	int max_pools;
	int pool_size;
	rtm_pool *try_pool = pool;
	int num_pools = 1;
	
	if (pool == NULL)
	{
		try_pool = _rtm_pool;
		pool = _rtm_pool;
		if (pool == NULL)
			return NULL;
	}

	assert(try_pool != NULL, ("NULL\n"));
	while (true) {
		void *ret = rtm_internal(try_pool, size);
		if (ret)
			return ret;

		if (!try_pool->next)
			break;
			
		try_pool = try_pool->next;			
		num_pools++;
	}
	
	
	/* Couldn't allocate from existing pools */
	PRINT(("rtm_alloc couldn't find memory in allocated pools\n"));
	ptr = getenv("MEDIA_MAX_ALLOCATOR_POOLS");
	max_pools = ptr ? atoi(ptr) : 5;
	if (max_pools > 20)
		max_pools = 20;

	if (num_pools >= max_pools) {
		PRINT(("max_pools allocated... fail\n"));
		return 0;
	}
	
	pool_size = try_pool->total_size-sizeof(rtm_block);
	if (pool_size < 128000) pool_size = 128000;
	if (pool_size > 3*1024*1024) pool_size = 3*1024*1024;
	PRINT(("allocate new pool size %d\n", pool_size));
	if (size > pool_size-sizeof(rtm_block)) {
		if (getenv("MEDIA_LOCKED_MEMORY_DEBUG") != 0) {
			fprintf(stderr, "Tried to allocate %d bytes from pool size %d\n", size, pool_size);
		}
		return 0;
	}
	rtm_create_pool(&try_pool->next, pool_size);
	return rtm_internal(try_pool->next, size);
}


void * rtm_internal(rtm_pool * pool, size_t size)
{
	rtm_free_list * list = NULL;
	rtm_block * block = NULL;
	size_t padded_size = (((size > 0) ? size : 1) + PADSIZE-1) & ~(PADSIZE-1);

	assert(pool != NULL,("NULL\n"));
	if (!pool) {
		return NULL;
	}

	if (atomic_add(&pool->lock_count, -1) < 0) {
		acquire_sem(pool->lock_sem);
	}
	list = &pool->free_list;
	if (!list->next || (pool->free_size < size)) {
		if (atomic_add(&pool->lock_count, 1) < -1) {
			release_sem(pool->lock_sem);
		}
		return NULL;	/*	guaranteed nothing big enough	*/
	}
	while (list->next && (block = &((rtm_block *)list->next)[-1])->phys_size < size) {
		list = list->next;
	}
	if (!list->next) {
		if (atomic_add(&pool->lock_count, 1) < -1) {
			release_sem(pool->lock_sem);
		}
		return NULL;	/*	nothing big enough	*/
	}
	assert(block->log_size == FREE_BLOCK,("0x%x 0x%x\n", block, block->log_size));
	assert(block->pool == pool,("0x%x 0x%x 0x%x\n", block, block->pool, pool));
	assert(list->next->prev == list,("0x%x 0x%x 0x%x\n", list, list->next, list->next->prev));
	assert(block->phys_size >= padded_size,("0x%x 0x%x 0x%x\n", block, block->phys_size, padded_size));
	/*	split the block?	*/
	if (block->phys_size > padded_size+PADDING_OK) {
		size_t o_phys = block->phys_size;
		rtm_free_list * next_free = NULL;
		rtm_block * next = (rtm_block *)((char *)&block[1] + padded_size);
		rtm_block * after;
		/*	set up split-off block	*/
		next->phys_size = o_phys-padded_size-sizeof(rtm_block);
		next->log_size = FREE_BLOCK;
		next->prev = block;
		next->pool = pool;
		next_free = (rtm_free_list *)&next[1];
		next_free->next = list->next->next;
		if (next_free->next) {
			next_free->next->prev = next_free;
		}
		next_free->prev = list;
		list->next = next_free;
		/* set up block after that */
		after = (rtm_block *)((char *)&next[1]+next->phys_size);
		if (after < (rtm_block *)((char *)&pool[1] + pool->total_size)) {
			assert(after->log_size >= 0,("0x%x 0x%x\n", after, after->log_size));
			after->prev = next;
		}
		/*	set up current block	*/
		block->phys_size = padded_size;
		block->log_size = size;
		/*	update pool	*/
		pool->total_blocks++;
		pool->free_size -= padded_size+sizeof(rtm_block);
		if (pool->free_size < pool->min_free_size) {
			pool->min_free_size = pool->free_size;
//			fprintf(stderr, "min_free_size %x %x\n", pool->min_free_size, pool->free_size);
		}
//		fprintf(stderr, "TRACE: split new block at 0x%x\n", &next[1]);
	}
	else {	/*	just claim the block	*/
		/*	unlink from free list	*/
		if (list->next->next) {
			list->next->next->prev = list;
		}
		list->next = list->next->next;
		/*	set up current block	*/
		block->log_size = size;
		/*	update pool	*/
		pool->free_blocks--;
		pool->free_size -= block->phys_size;
		if (pool->free_size < pool->min_free_size) {
			pool->min_free_size = pool->free_size;
//			fprintf(stderr, "min_free_size %x %x\n", pool->min_free_size, pool->free_size);
		}
	}

//	fprintf(stderr, "TRACE: alloc(0x%x)\n", &block[1]);
#if !NDEBUG
	rtm_validate_pool(pool);
#endif

	if (atomic_add(&pool->lock_count, 1) < -1) {
		release_sem(pool->lock_sem);
	}

	return &block[1];
}

status_t rtm_free(void * data)
{
	rtm_pool * pool = NULL;
	rtm_block * block = NULL, *next = NULL, *prev = NULL;
	rtm_free_list * free_list = NULL, *next_free = NULL;

	if (data == NULL) {
		return B_BAD_VALUE;	/*	idiotic, but useful	*/
	}
	block = &((rtm_block *)data)[-1];
	pool = block->pool;

	assert(pool != NULL,("NULL\n"));
	assert(block->phys_size >= block->log_size,("0x%x 0x%x 0x%x\n", block, block->phys_size, block->log_size));
	assert(block->phys_size <= pool->total_size,("0x%x 0x%x 0x%x 0x%x\n", block, block->phys_size, pool, pool->total_size));

	if (atomic_add(&pool->lock_count, -1) < 0) {
		acquire_sem(pool->lock_sem);
	}

//	fprintf(stderr, "TRACE: free(0x%x)\n", data);

	/*	we're freeing at least this amount	*/
	pool->free_size += block->phys_size;

	/*	is there a next block?	*/
	next = (rtm_block *)((char *)data+block->phys_size);
	if (next >= (rtm_block *)((char *)&pool[1]+pool->total_size)) {
		next = NULL;
	}
	/*	merge with next block?	*/
	if (next && next->log_size == FREE_BLOCK) {
		/*	fix up the free list	*/
		next_free = (rtm_free_list *)&next[1];
		assert(next_free->prev != NULL,("0x%x 0x%x\n", next_free, next_free->prev));
		/*	unlink next block from free list	*/
		if (next_free->next) {
			assert(next_free->next->prev == next_free,("0x%x 0x%x 0x%x\n", next_free, next_free->next, next_free->next->prev));
			next_free->next->prev = next_free->prev;
		}
		assert(next_free->prev->next == next_free,("0x%x 0x%x 0x%x\n", next_free, next_free->prev, next_free->prev->next));
		next_free->prev->next = next_free->next;
		/*	merge this block, freeing up a block header's worth of bytes	*/
		block->phys_size += next->phys_size + sizeof(rtm_block);
		pool->free_blocks--;	/* Because we unlinked it	*/
		pool->total_blocks--;
		pool->free_size += sizeof(rtm_block);
		/*	fix up next linear block's back pointer	*/
		next = (rtm_block *)((char *)&next[1] + next->phys_size);
		if (next < (rtm_block *)((char *)&pool[1]+pool->total_size)) {
			next->prev = block;
		}
		else {
			next = NULL;
		}

//		fprintf(stderr, "TRACE: merge next free block @ %x\n", &next[1]);
	}
	/*	merge with previous block?	*/
	prev = block->prev;
	if (prev && prev->log_size == FREE_BLOCK) {
		/*	we just stay in the free list	*/
		assert((rtm_block *)((char *)&prev[1]+prev->phys_size) == block,("0x%x 0x%x 0x%x\n", &prev[1], prev->phys_size, block));
		prev->phys_size += block->phys_size+sizeof(rtm_block);
		if (next) {
			next->prev = prev;
		}
		pool->total_blocks--;
		pool->free_size += sizeof(rtm_block);

//		fprintf(stderr, "TRACE: merge previous free block @ %x\n", &prev[1]);
	}
	else {
		/*	we need to link into the free list	*/
		free_list = (rtm_free_list *)&block[1];
		free_list->next = pool->free_list.next;
		free_list->prev = &pool->free_list;
		if (pool->free_list.next) {
			assert(pool->free_list.next->prev == &pool->free_list,("0x%x 0x%x 0x%x\n", &pool->free_list, pool->free_list.next, pool->free_list.next->prev));
			pool->free_list.next->prev = free_list;
		}
		pool->free_list.next = free_list;
		pool->free_blocks++;
		block->log_size = FREE_BLOCK;
	}

#if !NDEBUG
	rtm_validate_pool(pool);
#endif

	if (atomic_add(&pool->lock_count, 1) < -1) {
		release_sem(pool->lock_sem);
	}
	return B_OK;
}

/*
status_t rtm_realloc(void ** data, size_t new_size)
{
}
/* Not implemented yet. */

status_t rtm_size_for(void * data)
{
	rtm_pool * pool = NULL;
	rtm_block * block = NULL;
	size_t size = 0;

	if (data == NULL) {
		return B_BAD_VALUE;	/*	idiotic, but useful	*/
	}
	block = &((rtm_block *)data)[-1];
	pool = block->pool;

	assert(pool != NULL,("NULL\n"));
	assert(block->phys_size >= block->log_size,("0x%x 0x%x 0x%x\n", block, block->phys_size, block->log_size));
	assert(block->phys_size <= pool->total_size,("0x%x 0x%x 0x%x 0x%x\n", block, block->phys_size, pool, pool->total_size));

	if (atomic_add(&pool->lock_count, -1) < 0) {
		acquire_sem(pool->lock_sem);
	}

	size = block->log_size;

	if (atomic_add(&pool->lock_count, 1) < -1) {
		release_sem(pool->lock_sem);
	}

	return size;
}

void rtm_dump_block(rtm_pool * pool, void * data)
{
	assert(pool != NULL,("NULL"));

	if (atomic_add(&pool->lock_count, -1) < 0) {
		/*	some bozo may have died holding the lock	*/
		acquire_sem_etc(pool->lock_sem, 1, B_TIMEOUT, 1000000);
	}
	fprintf(stderr, "--------------------------------------------------------------\n");
	if (data) {
		rtm_block * block = &((rtm_block *)data)[-1];
		assert(block->pool == pool,("0x%x 0x%x 0x%x\n", block, block->pool, pool));
		fprintf(stderr, "pool: 0x%x through 0x%x\n", pool, pool->total_size+sizeof(rtm_pool));
		fprintf(stderr, "block: 0x%x phys 0x%x log 0x%x prev 0x%x pool 0x%x\n", block,
			block->phys_size, block->log_size, block->prev, block->pool);
		if (block->log_size < 0) {
			rtm_free_list * fl = (rtm_free_list *)&block[1];
			fprintf(stderr, "   free_next: 0x%x free_prev: 0x%x\n", fl->next, fl->prev);
		}
	}
	else {
		rtm_block * block = (rtm_block *)&pool[1];
		fprintf(stderr, "pool: 0x%x first block: 0x%x end: 0x%x\n", pool,
			&((rtm_block*)&pool[1])[1], (char *)&pool[1] + pool->total_size);
		fprintf(stderr, "first free block: 0x%x\n", pool->free_list.next);
		fprintf(stderr, "total size: %x  free size: %x  min free: %x  total blocks: %x  free blocks: %x\n",
			pool->total_size, pool->free_size, pool->min_free_size, pool->total_blocks, pool->free_blocks);
		while (block < (rtm_block *)((char *)&pool[1] + pool->total_size)) {
			fprintf(stderr, "block: 0x%x phys 0x%x log 0x%x prev 0x%x pool 0x%x\n", block,
				block->phys_size, block->log_size, block->prev, block->pool);
			if (block->pool != pool) {
				fprintf(stderr, "\n### Damaged header; cannot continue!\n");
				break;
			}
			if (block->log_size < 0) {
				rtm_free_list * fl = (rtm_free_list *)&block[1];
				fprintf(stderr, "   free_next: 0x%x free_prev: 0x%x\n", fl->next, fl->prev);
			}
			block = (rtm_block *)((char *)&block[1] + block->phys_size);
		}
	}
	if (atomic_add(&pool->lock_count, 1) < -1) {
		release_sem(pool->lock_sem);
	}
}


rtm_pool *
rtm_default_pool()
{
	return _rtm_pool;
}
