
#include "ChromeVideo.h"

#include "TimedEventQueue.h"

#include <BufferGroup.h>
#include <Buffer.h>
#include <GraphicsDefs.h>
#include <TimeSource.h>

#include <stdio.h>

#define	CALL		//printf
#define	TIMING		//printf
#define	INFO		//printf
#define ERROR		//printf

struct pixel {
	uint8 b;
	uint8 g;
	uint8 r;
	uint8 a;
};

ChromeVideo::ChromeVideo(BMediaAddOn *addOn) :
	BBufferFilter("ChromeVideo", B_MEDIA_RAW_VIDEO, B_MEDIA_RAW_VIDEO, addOn),
	BMediaNode("ChromeVideo"),
	fInBufferSize(0),
	fOutBufferSize(0)
{
}


ChromeVideo::~ChromeVideo()
{
}

void 
ChromeVideo::Init()
{
	BBufferFilter::Init();
}

void 
ChromeVideo::CleanUp()
{
	BBufferFilter::CleanUp();
}

void 
ChromeVideo::PublishLatency(bigtime_t *latency)
{
	// a bogus value until we can find something a little
	// more accurate
	*latency = 10000;
}

status_t 
ChromeVideo::StartNow()
{
	return B_OK;
}

status_t 
ChromeVideo::StopNow()
{
	return B_OK;
}

status_t 
ChromeVideo::SeekNow(bigtime_t seekTo)
{
	return B_OK;
}

status_t 
ChromeVideo::WarpNow(bigtime_t warpTo)
{
	return B_OK;
}

void 
ChromeVideo::HandleEvent(const bigtime_t time, const int32 what, const void *pointer, const uint32 flags, const int64 data)
{
	BBufferFilter::HandleEvent(time, what, pointer, flags, data);
}

void 
ChromeVideo::FilterBuffer(BBuffer *inBuffer)
{
	if (!inBuffer) return;
	/* here is where we do all of the real work */
	CALL("FilterBuffer now: %Ld\n", TimeSource()->Now());

	BBuffer *outBuffer = fOut.buffers->RequestBuffer(fOutBufferSize);
	if (!outBuffer)
		inBuffer->Recycle();
	
	CALL("ChromeVideo::FilterBuffer: inBuffer: 0x%x outBuffer: 0x%x\n", inBuffer, outBuffer);

	media_header *inHeader = inBuffer->Header();

	CALL("now: %Ld start_time: %Ld\n", TimeSource()->Now(), inHeader->start_time);
	
	pixel *inData = (pixel *) inBuffer->Data();
	uint32 pixelsUsed = (inHeader->size_used)/sizeof(pixel);
	uint32 pixelCount = 0;
	
	uint8 *outData = (uint8 *) outBuffer->Data();
	uint32 outSizeUsed = 0;
	
	bigtime_t start = BTimeSource::RealTime();
	while (pixelCount < pixelsUsed) {
		// set the outgoing uint8 to the red value of the incoming pixel
		outData[pixelCount] = inData[pixelCount].r << 3;
		// increment the current pixel and the outgoing position
		pixelCount++;
	}
	bigtime_t end = BTimeSource::RealTime();

	INFO("latency: %Ld\n", end-start);

	// fill out all of the fun header information and send the buffer
	media_header *outHeader = outBuffer->Header();
	outHeader->type = B_MEDIA_RAW_VIDEO;
	outHeader->time_source = inHeader->time_source;
	outHeader->size_used = outSizeUsed;
	outHeader->start_time = inHeader->start_time;
	outHeader->u.raw_video = inHeader->u.raw_video;
	
	//Send the stinking buffer
	if (TimeSource()->Now() > outHeader->start_time) {
		ERROR("Buffer late!\n");
		outBuffer->Recycle();
	}
	
	if (SendBuffer(outBuffer, fOut.out.destination) != B_OK) {
		ERROR("buffer not sent!\n");
		outBuffer->Recycle();
	}
	inBuffer->Recycle();
}

BBufferGroup *
ChromeVideo::CreateBufferGroup(media_output &output)
{
	CALL("ChromeVideo::CreateBufferGroup\n");
	// look at the format in output and build an appropriate buffer group
	fOutBufferSize = fOut.out.format.u.raw_video.display.line_count * fOut.out.format.u.raw_video.display.bytes_per_row;
	return new BBufferGroup(fOutBufferSize, 10);
}

void 
ChromeVideo::Preroll()
{
}

status_t 
ChromeVideo::AcceptFormat(const media_destination &dest, media_format *format)
{
	char format_string[256];
	string_for_format(*format, format_string, 256);
	CALL("ChromeVideo::AcceptFormat(%s)\n", format_string);
/*
	check to see if the destination matches one of your input destinations 
		if not return B_MEDIA_BAD_DESTINATION 

	then check the format->type field and if set to B_MEDIA_NO_TYPE
		set it to your preferred type
	if format->type is not your preferred format type
		return B_MEDIA_BAD_FORMAT
	
	Step through the ENTIRE format structure.  For every field check to
	see if you support a filled in value, if not return B_MEDIA_BAD_FORMAT
	If it is a wildcard, fill it in with your preferred information
	If it is supported, move to the next field.
	If the format is supported, return B_OK
*/
	if (dest == fIn.in.destination) {
		if (format->type == B_MEDIA_NO_TYPE)
			format->type = B_MEDIA_RAW_VIDEO;
		
		if (format->type != B_MEDIA_RAW_VIDEO)
			return B_MEDIA_BAD_FORMAT;
			
		/* start with the display info */
		/* color space - only 32 bit right now */
		if (format->u.raw_video.display.format == media_video_display_info::wildcard.format)
			format->u.raw_video.display.format = B_RGB32;
		if (format->u.raw_video.display.format != B_RGB32)
			return B_MEDIA_BAD_FORMAT;
		
		/* line width - if unspecified, use 320 */
		if (format->u.raw_video.display.line_width == media_video_display_info::wildcard.line_width)
			format->u.raw_video.display.line_width = 320;
		/* line count - if unspecified use 240 */
		if (format->u.raw_video.display.line_count == media_video_display_info::wildcard.line_count)
			format->u.raw_video.display.line_count = 240;

		/* bytes per row - needs to be 8bytes * line_width*/
		if (format->u.raw_video.display.bytes_per_row == media_video_display_info::wildcard.bytes_per_row)
			format->u.raw_video.display.bytes_per_row =
				(format->u.raw_video.display.line_width * 4);
		if (format->u.raw_video.display.bytes_per_row != (format->u.raw_video.display.line_width * 4)) {
			return B_MEDIA_BAD_FORMAT;
		}
		
		/* pixel offset */
		if (format->u.raw_video.display.pixel_offset == media_video_display_info::wildcard.pixel_offset)
			format->u.raw_video.display.pixel_offset = 0;
		if (format->u.raw_video.display.pixel_offset != 0)
			return B_MEDIA_BAD_FORMAT;
			
		/* line_offset */
		if (format->u.raw_video.display.line_offset == media_video_display_info::wildcard.line_offset)
			format->u.raw_video.display.line_offset = 0;
		if (format->u.raw_video.display.line_offset != 0)
			return B_MEDIA_BAD_FORMAT;

		/* field rate */
		if (format->u.raw_video.field_rate == media_raw_video_format::wildcard.field_rate)
			format->u.raw_video.field_rate = 30;	// 30 fps
		
		/* interlace */
		if (format->u.raw_video.interlace == media_raw_video_format::wildcard.interlace)
			format->u.raw_video.interlace = 1;
		if (format->u.raw_video.interlace != 1)
			return B_MEDIA_BAD_FORMAT;
		
		/* first active */
		if (format->u.raw_video.first_active == media_raw_video_format::wildcard.first_active)
			format->u.raw_video.first_active = 0;
		if (format->u.raw_video.first_active != 0)
			return B_MEDIA_BAD_FORMAT;
		
		/* last_active - needs to be display.line_count - 1*/
		if (format->u.raw_video.last_active == media_raw_video_format::wildcard.last_active)
			format->u.raw_video.last_active = format->u.raw_video.display.line_count - 1;
		if (format->u.raw_video.last_active != format->u.raw_video.display.line_count -1) {
			ERROR("last_active problem: %d %d\n", format->u.raw_video.last_active, format->u.raw_video.display.line_count - 1);
			return B_MEDIA_BAD_FORMAT;
		}	
		/* orientation */
		if (format->u.raw_video.orientation == media_raw_video_format::wildcard.orientation)
			format->u.raw_video.orientation = B_VIDEO_TOP_LEFT_RIGHT;
		if (format->u.raw_video.orientation != B_VIDEO_TOP_LEFT_RIGHT)
			return B_MEDIA_BAD_FORMAT;
			
		/* pixel width aspect */
		if (format->u.raw_video.pixel_width_aspect == media_raw_video_format::wildcard.pixel_width_aspect)
			format->u.raw_video.pixel_width_aspect = 1;
		if (format->u.raw_video.pixel_width_aspect != 1)
			return B_MEDIA_BAD_FORMAT;
			
		/* pixel height aspect */
		if (format->u.raw_video.pixel_height_aspect == media_raw_video_format::wildcard.pixel_height_aspect)
			format->u.raw_video.pixel_height_aspect = 1;
		if (format->u.raw_video.pixel_height_aspect != 1)
			return B_MEDIA_BAD_FORMAT;
			
		string_for_format(*format, format_string, 256);
		CALL("AcceptFormat OK!: %s\n", format_string);
		return B_OK;
	}
	else return B_MEDIA_BAD_DESTINATION;
}

status_t 
ChromeVideo::Connected(const media_source &producer, const media_destination &where, const media_format &with_format, media_input *out_input)
{
	CALL("ChromeVideo::Connected\n");

	if (where != fIn.in.destination)
		return B_MEDIA_BAD_DESTINATION;
	if (producer == fIn.in.source)
		return B_MEDIA_ALREADY_CONNECTED;
	if (fIn.in.source != media_source::null)
		return B_MEDIA_NOT_CONNECTED;
		
	fIn.in.source = producer;
	fIn.in.format = with_format;
	*out_input = fIn.in;

	fInBufferSize = fIn.in.format.u.raw_video.display.line_count * fIn.in.format.u.raw_video.display.bytes_per_row;

	CALL("ChromeVideo::Connected OK!\n");

	return B_OK;
}

status_t 
ChromeVideo::FormatChanged(const media_source &producer, const media_destination &consumer, int32 from_change_format, const media_format &format)
{
	char format_string[256];
	string_for_format(format, format_string, 256);
	CALL("ChromeVideo::FormatChanged(%s)\n", format_string);

	if (consumer != fIn.in.destination)
		return B_MEDIA_BAD_DESTINATION;
	if (producer != fIn.in.source)
		return B_MEDIA_BAD_SOURCE;
		
	fIn.in.format = format;
	
	/* the format has been accepted - so make sure the buffers are correct */
	
	return B_OK;
}

status_t 
ChromeVideo::FormatSuggestionRequested(media_type type, int32 quality, media_format *format)
{
	char format_string[256];
	string_for_format(*format, format_string, 256);
	CALL("ChromeVideo::FormatSuggestionRequested(%s)\n", format_string);

	if (type != B_MEDIA_RAW_VIDEO)
		return B_MEDIA_BAD_FORMAT;
	
	/* fill in all of the preferred bits */
	
}

status_t 
ChromeVideo::FormatProposal(const media_source &output, media_format *format)
{
	CALL("ChromeVideo::FormatProposal\n");

	if (output == fOut.out.source) {
		/* fill in your flexible format. */
		/* set any fields in format that you require for that output */
		/* return B_OK */
		format->type = B_MEDIA_RAW_VIDEO;
		format->u.raw_video = media_raw_video_format::wildcard;
		format->u.raw_video.field_rate = fIn.in.format.u.raw_video.field_rate;
		format->u.raw_video.interlace = 1;
		format->u.raw_video.first_active = 0;
		format->u.raw_video.last_active = fIn.in.format.u.raw_video.last_active;
		format->u.raw_video.orientation = B_VIDEO_TOP_LEFT_RIGHT;
		format->u.raw_video.pixel_width_aspect = 1;
		format->u.raw_video.pixel_height_aspect = 1;
		format->u.raw_video.display.format = B_GRAY8;
		format->u.raw_video.display.line_width = fIn.in.format.u.raw_video.display.line_width;
		format->u.raw_video.display.line_count = fIn.in.format.u.raw_video.display.line_count;
		format->u.raw_video.display.bytes_per_row = fIn.in.format.u.raw_video.display.line_width; // 8 bit only
		format->u.raw_video.display.pixel_offset = 0;
		format->u.raw_video.display.line_offset = 0;
		
		char format_string[256];
		string_for_format(*format, format_string, 256);
		CALL("FormatProposal OK!: %s\n", format_string);
		return B_OK;
	}
	else return B_MEDIA_BAD_SOURCE;
}

status_t 
ChromeVideo::FormatChangeRequested(const media_source &source, const media_destination &dest, media_format *io_format, int32 *out_change_count)
{
	CALL("ChromeVideo::FormatChangeRequested\n");
	if (source == fOut.out.source) {
		if (dest != fOut.out.destination)
			return B_MEDIA_BAD_DESTINATION;
		/* change your output format to ioformat, filling in any wildcards */
		/* return the new fOut.out.format and issue a new change tag with */
		/* MintChangeTag */
		/* return B_ERROR if you do not support the requested format */
		return B_ERROR;
	}
	else return B_MEDIA_BAD_SOURCE;
}

status_t 
ChromeVideo::VideoClippingChanged(const media_source &for_source, int16 num_shorts, int16 *clip_data, const media_video_display_info &display, int32 *out_from_change_count)
{
	if (for_source != fOut.out.source)
		return B_MEDIA_BAD_SOURCE;
	/* if you handle video make the appropriate change and update the change */
	/* tag and return B_OK */
	return B_ERROR;
}

status_t 
ChromeVideo::PrepareToConnect(const media_source &what, const media_destination &where, media_format *format, media_source *out_source, char *out_name)
{
	CALL("ChromeVideo::PrepareToConnect\n");
	if (what == fOut.out.source) {
		if (where == fOut.out.destination)
			return B_MEDIA_ALREADY_CONNECTED;
		if (fOut.out.destination != media_destination::null)
			return B_MEDIA_NOT_CONNECTED;
		/* reserve the output for this connection */
		fOut.out.destination = where;
		/* check to see if the format is compatible */
		/* set the format appropriately */
		fOut.out.format = *format;
		*out_source = fOut.out.source;
		strcpy(out_name, fOut.out.name);
		return B_OK;
	}
	else return B_MEDIA_BAD_SOURCE;
}

void 
ChromeVideo::Connect(status_t error, const media_source &source, const media_destination &dest, const media_format &format, char *io_name)
{
	CALL("ChromeVideo::Connect\n");
	BBufferFilter::Connect(error, source, dest, format, io_name);
}

void 
ChromeVideo::LateNoticeReceived(const media_source &what, bigtime_t how_much, bigtime_t performance_time)
{
	if (what == fOut.out.source) {
		switch(RunMode()) {
			case B_OFFLINE:
			case B_DECREASE_PRECISION:
			case B_INCREASE_LATENCY:
			case B_DROP_DATA:
			case B_RECORDING:
				break;
		}
	}
	else return;
}

status_t 
ChromeVideo::SetBufferGroup(const media_source &for_source, BBufferGroup *group)
{
	CALL("ChromeVideo::SetBufferGroup\n");

	if (for_source == fOut.out.source) {
		/* if you want to support buffer groups created by other people */
		/* check the size of the buffers and make sure they are large enough */
		/* to hold your information. also check to make sure there are enough */
		/* buffers to handle your latency issues */
		/* if you accept this buffer group you must reclaim all outstanding */
	 	if (fOut.buffers && fOut.own_buffers == true) {
			fOut.buffers->ReclaimAllBuffers();
			delete fOut.buffers;
			fOut.buffers == NULL;
		}
		
		fOut.buffers = group;
		fOut.own_buffers = false;
		
		return B_OK;
	}
	else return B_MEDIA_BAD_SOURCE;
}


