
#include "ChromeTV.h"

#include <MediaDefs.h>
#include <MediaRoster.h>
#include <MediaNode.h>
#include <scheduler.h>
#include <TimeSource.h>

#include "rt_allocator.h"
#include <stdio.h>
#include <string.h>

extern rtm_pool * _rtm_pool;

#include <Alert.h>

static void
ErrorAlert(const char * message, status_t err)
{
	char msg[256];
	sprintf(msg, "%s\n%s [%x]", message, strerror(err), err);
	(new BAlert("", msg, "Quit"))->Go();
	be_app->PostMessage(B_QUIT_REQUESTED);
}


ChromeTVApp::ChromeTVApp() :
	BApplication("application/x-vnd.BeDTS-ChromeTVApp"),
	fMediaRoster(NULL),
	fFilter(NULL)
{
	// do not use or copy!!! - this is an intermediate solution
	rtm_create_pool(&_rtm_pool, (B_PAGE_SIZE * 64));
}


ChromeTVApp::~ChromeTVApp()
{
	fMediaRoster->ReleaseNode(fProducerNode);
	fMediaRoster->ReleaseNode(fConsumerNode);
	
	if (fFilter) {
		fFilter->Release();
		fFilter = NULL;
	}
	// do not use or copy!!! - this is an intermediate solution
	rtm_delete_pool(_rtm_pool);
}

void 
ChromeTVApp::ReadyToRun()
{
	status_t status = SetUpNodes();
	if (status != B_OK) {
		ErrorAlert("SetUpNodes() error", status);
		return;
	}

	StopWindow *stopWindow = new StopWindow;
	stopWindow->Show();
}

bool 
ChromeTVApp::QuitRequested()
{
	TearDownNodes();
	return true;
}

status_t 
ChromeTVApp::SetUpNodes()
{
	status_t status = B_OK;

	/* find the media roster */
	fMediaRoster = BMediaRoster::Roster(&status);
	if (status != B_OK) {
		ErrorAlert("cannot find the media roster", status);
		return status;
	}	
	/* find the time source */
	status = fMediaRoster->GetTimeSource(&fTimeSourceNode);
	if (status != B_OK) {
		ErrorAlert("failed to get a time source", status);
		return status;
	}
	/* find a video producer node */
	status = fMediaRoster->GetVideoInput(&fProducerNode);
	if (status != B_OK) {
		ErrorAlert("could not find a video input!", status);
		return status;
	}

	/* find the video consumer node */
	status = fMediaRoster->GetVideoOutput(&fConsumerNode);
	if (status != B_OK) {
		ErrorAlert("could not find a video output node!", status);
		return status;
	}
	/* create the filter node */
	fFilter = new ChromeVideo(NULL);
	if (!fFilter) {
		ErrorAlert("could not instantiate ChromeVideo node", B_ERROR);
		return B_ERROR;
	}
	
	/* find free producer output */
	int32 cnt = 0;
	status = fMediaRoster->GetFreeOutputsFor(fProducerNode, &fProducerOut, 1,  &cnt, B_MEDIA_RAW_VIDEO);
	if (status != B_OK || cnt < 1) {
		ErrorAlert("could not find output for video producer", status);
		return status;
	}

	/* find free filter input */
	cnt = 0;
	status = fMediaRoster->GetFreeInputsFor(fFilter->Node(), &fFilterIn, 1, &cnt, B_MEDIA_RAW_VIDEO);
	if (status != B_OK || cnt < 1) {
		ErrorAlert(" could not find input for filter!", status);
		return status;
	}
	
	/* find free filter output */
	cnt = 0;
	status = fMediaRoster->GetFreeOutputsFor(fFilter->Node(), &fFilterOut, 1, &cnt, B_MEDIA_RAW_VIDEO);
	if (status != B_OK || cnt < 1) {
		ErrorAlert("could not find output for filter!", status);
		return status;
	}

	/* find free consumer input */
	cnt = 0;
	status = fMediaRoster->GetFreeInputsFor(fConsumerNode, &fConsumerIn, 1, &cnt, B_MEDIA_RAW_VIDEO);
	if (status != B_OK || cnt < 1) {
		ErrorAlert("could not find input for video consumer!", status);
		return status;
	}

	/* Connect The Nodes!!! */
	media_format format;
	format.type = B_MEDIA_RAW_VIDEO;
	format.u.raw_video = media_raw_video_format::wildcard;
	
	/* connect producer to filter */
	status = fMediaRoster->Connect(fProducerOut.source, fFilterIn.destination,
				&format, &fProducerOut, &fFilterIn);
	if (status != B_OK) {
		ErrorAlert("cannot connect producer to filter!", status);
		return status;
	}
	
	/* connect filter to consumer */
	format.type = B_MEDIA_RAW_VIDEO;
	format.u.raw_video = media_raw_video_format::wildcard;
	status = fMediaRoster->Connect(fFilterOut.source, fConsumerIn.destination,
				&format, &fFilterOut, &fConsumerIn);
	if (status != B_OK) {
		ErrorAlert("cannot connect filter to consumer!", status);
		return status;
	}
	
	
	/* set time sources */
	status = fMediaRoster->SetTimeSourceFor(fProducerNode.node, fTimeSourceNode.node);
	if (status != B_OK) {
		ErrorAlert("cannot set producer timesource!", status);
		return status;
	}
	
	status = fMediaRoster->SetTimeSourceFor(fFilter->ID(), fTimeSourceNode.node);
	if (status != B_OK) {
		ErrorAlert("cannot set filter timesource!", status);
		return status;
	}

	status = fMediaRoster->SetTimeSourceFor(fConsumerNode.node, fTimeSourceNode.node);
	if (status != B_OK) {
		ErrorAlert("cannot set consumer timesource!", status);
		return status;
	}
	
	/* start the nodes */
	bigtime_t start_delay = 0;
	fMediaRoster->GetStartLatencyFor(fTimeSourceNode, &start_delay);
	start_delay += estimate_max_scheduling_latency();
	
	BTimeSource *timeSource = fMediaRoster->MakeTimeSourceFor(fTimeSourceNode);
	bool running = timeSource->IsRunning();
	
	/* workaround for people without sound cards */
	bigtime_t real = BTimeSource::RealTime();
	if (!running)
	{
		status = fMediaRoster->StartNode(fTimeSourceNode, real);
		if (status != B_OK) {
			timeSource->Release();
			ErrorAlert("cannost start time source!", status);
			return status;
		}
		status = fMediaRoster->SeekNode(fTimeSourceNode, 0, real);
		if (status != B_OK) {
			timeSource->Release();
			ErrorAlert("cannot seek time source!", status);
			return status;
		}
	}

	bigtime_t perf = timeSource->Now() + start_delay;
	timeSource->Release();
	
	status = fMediaRoster->StartNode(fProducerNode, (fProducerNode.kind & B_TIME_SOURCE) ? real : perf);
	if (status != B_OK) {
		ErrorAlert("error starting producer!", status);
		return status;
	}
	status = fMediaRoster->StartNode(fFilter->Node(), perf);
	if(status != B_OK) {
		ErrorAlert("error starting filter", status);
		return status;
	}
	status = fMediaRoster->StartNode(fConsumerNode, (fConsumerNode.kind & B_TIME_SOURCE) ? real : perf);
	if (status != B_OK) {
		ErrorAlert("error starting consumer", status);
		return status;
	}
	/* hunh??? isn't it already running! */
	status = fMediaRoster->StartNode(fTimeSourceNode, real);
	if (status != B_OK) {
		ErrorAlert("error starting time source", status);
		return status;
	}
	
	printf("SetUpNodes OK!\n");
	return status;
}

void 
ChromeTVApp::TearDownNodes()
{
	if (!fMediaRoster)
		return;
	
	/* stop */	
	printf("stopping nodes!\n");
	fMediaRoster->StopNode(fConsumerNode, 0, true);
	fMediaRoster->StopNode(fFilter->Node(), 0, true);
	fMediaRoster->StopNode(fProducerNode, 0, true);
	
	/* disconnect */
	fMediaRoster->Disconnect(fFilterOut.node.node, fFilterOut.source,
							fConsumerIn.node.node, fConsumerIn.destination);
	fMediaRoster->Disconnect(fProducerOut.node.node, fProducerOut.source,
							fFilterIn.node.node, fFilterIn.destination);
}

#include <Button.h>

StopWindow::StopWindow() :
	BWindow(BRect(50,50,200,100), "ChromeTV", B_TITLED_WINDOW, B_NOT_RESIZABLE | B_NOT_ZOOMABLE)
{
	BButton *button = new BButton(Bounds(), "stop button", "Stop", new BMessage(B_QUIT_REQUESTED));
	button->SetTarget(be_app, be_app);
	AddChild(button);
}

bool 
StopWindow::QuitRequested()
{
	be_app->PostMessage(B_QUIT_REQUESTED);
	return false;
}



main() {
	ChromeTVApp app;
	app.Run();
	return 0;	
}