
#include "BufferFilter.h"
#include "TimedEventQueue.h"

#include <MediaRoster.h>
#include <MediaDefs.h>
#include <TimeSource.h>
#include <Buffer.h>
#include <BufferGroup.h>

#include <String.h>
#include <scheduler.h>

#include <stdio.h>

#define	CALL		//printf
#define	TIMING		//printf
#define	INFO		//printf
#define ERROR		//printf
#define FORMAT		printf

BBufferFilter::BBufferFilter(const char *name, media_type consumed, media_type produced, BMediaAddOn *addOn):
	BMediaNode(name),
	BBufferConsumer(consumed),
	BBufferProducer(produced),
	fSchedulingLatency(0),
	fControlPort(-1),
	fMainThread(-1),
	fTimeToQuit(false),
	fRunning(false),
	fMediaTime(0),
	fCode(0),
	fRoster(NULL),
	fEvents(NULL),
	fAddOn(addOn)
{
	BString media_name;
	
	// create port
	media_name = "DecoderPort: ";
	media_name += name;
	media_name.Truncate(64);
	fControlPort = create_port(15, media_name.String());
	
	// spawn thread
	media_name = "DecoderThread: ";
	media_name += name;
	media_name.Truncate(64);
	fMainThread = spawn_thread(EnterMainLoop, media_name.String(), B_NORMAL_PRIORITY, this);

	// initialize i/o
	fIn.in.node = Node();
	fIn.in.source = media_source::null;
	fIn.in.destination.port = fControlPort; 
	fIn.in.destination.id = 0;
	fIn.in.format.type = consumed;
	media_name = "DecodeIn: ";
	media_name += name;
	media_name.Truncate(64);
	strcpy(fIn.in.name, media_name.String());
	
	fIn.data_status = B_DATA_NOT_AVAILABLE;
	fIn.latency = 0;
	fIn.own_buffers = false;
	fIn.buffers = NULL;

	
	fOut.out.node = Node();
	fOut.out.source.port = fControlPort;
	fOut.out.source.id = 1;
	fOut.out.destination = media_destination::null;
	fOut.out.format.type = produced;
	media_name = "DecodeOut: ";
	media_name += name;
	media_name.Truncate(64);
	strcpy(fOut.out.name, media_name.String());

	fOut.enabled = false;
	fOut.downstream_latency = 0;
	fOut.processing_latency = 0;
	fOut.own_buffers = true;
	fOut.buffers = NULL;

	fRoster = BMediaRoster::Roster();

	fEvents = CreateEventQueue();
	
	// call the subclass Initialization routine
	Init();

	//	register with the MediaRoster
	fRoster->RegisterNode(this);
	
	// resume our thread
	resume_thread(fMainThread);

	// get latency information
	fSchedulingLatency = estimate_max_scheduling_latency(fMainThread);
	PublishLatency(&(fOut.processing_latency));

}


BBufferFilter::~BBufferFilter()
{
	// stop the main thread
	fTimeToQuit = true;
	status_t status;
	wait_for_thread(fMainThread, &status);
	fMainThread = -1;
	
	// tell subclasses to clean up
	CleanUp();
	
	// clear the event queues
	
	
	// delete the buffer groups
	

	// delete the port
	delete_port(fControlPort);
	fControlPort = -1;
	
	fRoster->UnregisterNode(this);
}

port_id 
BBufferFilter::ControlPort() const
{
	return fControlPort;
}

BMediaAddOn *
BBufferFilter::AddOn(int32 *internal_id) const
{
	/* be sure to do the right thing if in an add-on */
	*internal_id = 0;
	return fAddOn;
}

BMediaRoster *
BBufferFilter::Roster() const
{
	return fRoster;
}

void 
BBufferFilter::Init()
{
	return;
}

void 
BBufferFilter::CleanUp()
{
	return;
}

void 
BBufferFilter::PublishLatency(bigtime_t *latency)
{
	*latency = 0;
}

void 
BBufferFilter::PublishMediaFormats()
{
	return;
}

void 
BBufferFilter::SetUpConnections()
{
	return;
}

status_t 
BBufferFilter::StartNow()
{
	CALL("StartNow()\n");
	return B_OK;
}

status_t 
BBufferFilter::StopNow()
{
	CALL("StopNow()\n");
	return B_OK;
}

status_t 
BBufferFilter::SeekNow(bigtime_t seekTo)
{
	CALL("SeekNow()\n");
	return B_OK;
}

status_t 
BBufferFilter::WarpNow(bigtime_t warpTo)
{
	CALL("WarpNow()\n");
	return B_OK;
}

status_t 
BBufferFilter::EnterMainLoop(void *castToBufferDecoder)
{
	BBufferFilter *decoder = (BBufferFilter *) castToBufferDecoder;
	decoder->MainLoop();
	return 0;
}

void 
BBufferFilter::MainLoop()
{
	while (!fTimeToQuit) {
		status_t err = B_OK;
		bigtime_t latency = 0;
		
		/* while there are no events or it is not time for the earliest event */
		while (!fEvents->HasEvents() ||
				fEvents->NextEvent(NULL) > TimeSource()->Now() -
					(latency = fOut.downstream_latency + fOut.processing_latency + fSchedulingLatency)) {
			/* wait for a new event and handle it */
			err = WaitForMessages(TimeSource()->RealTimeFor(fEvents->NextEvent(NULL), latency));
			if (err == B_TIMED_OUT)
				break;
		}
	
		/* we have timed out - so handle the next event */
		bigtime_t perfTime = 0;
		int32 what = BTimedEventQueue::B_NO_EVENT;
		void *pointer = NULL;
		uint32 cleanup = BTimedEventQueue::B_NO_CLEANUP;
		bigtime_t data = 0;
		if (fEvents->PopEvent(&perfTime, &what, &pointer, &cleanup, &data) == B_OK)
			HandleEvent(perfTime, what, pointer, cleanup, data);
	}
}

void 
BBufferFilter::HandleEvent(const bigtime_t time, const int32 what, const void *pointer, const uint32 flag, const int64 data)
{
	switch(what) {
		case BTimedEventQueue::B_START: {
			if (!fRunning) {
				fRunning = true;
				StartNow();
			}
			break;
		}
		
		case BTimedEventQueue::B_STOP: {
			if (fRunning) {
				fRunning = false;
				StopNow();
			}
			break;
		}
		
		case BTimedEventQueue::B_SEEK: {
			fMediaTime = (bigtime_t) data;
			break;
		}
		
		case BTimedEventQueue::B_WARP: {
			/* handle any stuff now */
			/* apply waiting events and the like! */
			
			WarpNow((bigtime_t) data);
			break;
		}
		
		case BTimedEventQueue::B_HANDLE_BUFFER: {
			CALL("BTimedEventQueue::B_HANDLE_BUFFER\n");
			BBuffer *buffer = (BBuffer *) pointer;
			if (fRunning) {
				FilterBuffer(buffer);
			} else buffer->Recycle();
			break;
		}
		
		case BTimedEventQueue::B_DATA_STATUS: {
			fIn.data_status = (status_t) data; 
			break;
		}
		default:
			break;
	}
}

void 
BBufferFilter::FilterBuffer(BBuffer *inBuffer)
{
	/* do the actual work to decode the buffer */
	
	/* potentially queue up buffers to send in the out queues */
	
	/* send the stinkin buffers */
}

BBufferGroup *
BBufferFilter::CreateBufferGroup(media_output &output)
{
	BBufferGroup *group = NULL;
	int32 bufferSize = 2048;
	int32 bufferCount = 3;

	if (output.source == fOut.out.source) {
		/* determine a buffer count */
		/* latency / frame_duration */
		/* frame_duration = 1000000 / (frame_rate * channels) */
	}	
	return new BBufferGroup(bufferSize, bufferCount);
}

BTimedEventQueue *
BBufferFilter::CreateEventQueue()
{
	return new BTimedEventQueue;
}

status_t 
BBufferFilter::WaitForMessages(bigtime_t waitUntil)
{
	/* wait until the absolute time needed */
	status_t err = B_OK;

#if (B_BEOS_VERSION == B_BEOS_VERSION_4)
	bigtime_t real = BTimeSource::RealTime();
	err = read_port_etc(fControlPort, &fCode, &fMsg, B_MEDIA_MESSAGE_SIZE, B_TIMEOUT, waitUntil - real);
#else
	err = read_port_etc(fControlPort, &fCode, &fMsg, B_MEDIA_MESSAGE_SIZE, B_ABSOLUTE_TIMEOUT, waitUntil);
#endif
	if(err == B_TIMED_OUT)
		return err;
	/* handle a message if it comes in */
	HandleMessage(fCode, fMsg, B_MEDIA_MESSAGE_SIZE);
	return B_OK;
}

status_t 
BBufferFilter::HandleMessage(int32 code, const void *msg, size_t size)
{
	CALL("BBufferFilter::HandleMessage(0x%x)\n", code);
	if (BBufferConsumer::HandleMessage(code, msg, size) != B_OK) {
		if (BBufferProducer::HandleMessage(code, msg, size) != B_OK) {
			if (BMediaNode::HandleMessage(code, msg, size) != B_OK) {
				BMediaNode::HandleBadMessage(code, msg, size);
				ERROR("bad message\n");
				return B_ERROR;
			} else INFO("BMediaNode handled message\n");
		}else INFO("BBufferProducer handled message\n");
	} else INFO("BBufferConsumer handled message\n");
	return B_OK;
}

void 
BBufferFilter::Start(bigtime_t performance_time)
{
	CALL("BBufferFilter::Start\n");
	/* add a start to each event queue */
	fEvents->PushEvent(performance_time, BTimedEventQueue::B_START, NULL, BTimedEventQueue::B_NO_CLEANUP, 0);
}

void 
BBufferFilter::Stop(bigtime_t performance_time, bool immediate)
{
	CALL("BBufferFilter::Stop\n");
	/* add a start to each event queue */
	bigtime_t timeToStop = performance_time;
	if (immediate)
		timeToStop = TimeSource()->PerformanceTimeFor(BTimeSource::RealTime());

	fEvents->PushEvent(timeToStop, BTimedEventQueue::B_STOP, NULL, BTimedEventQueue::B_NO_CLEANUP, 0);
}

void 
BBufferFilter::Seek(bigtime_t to_media_time, bigtime_t at_real_time)
{
	CALL("BBufferFilter::Seek\n");
	/* perhaps we only need to do this once? say in fIn.events ?*/
	bigtime_t timeToSeek = TimeSource()->PerformanceTimeFor(at_real_time);
	fEvents->PushEvent(timeToSeek, BTimedEventQueue::B_SEEK, NULL, BTimedEventQueue::B_NO_CLEANUP, to_media_time);
}

void 
BBufferFilter::SetRunMode(run_mode mode)
{
	BMediaNode::SetRunMode(mode);
}

void 
BBufferFilter::TimeWarp(bigtime_t at_real_time, bigtime_t to_performance_time)
{
	/* perhaps we only need to do this once? say in fIn.events ?*/
	bigtime_t timeToWarp = TimeSource()->PerformanceTimeFor(at_real_time);
	fEvents->PushEvent(timeToWarp, BTimedEventQueue::B_WARP, NULL, BTimedEventQueue::B_NO_CLEANUP, to_performance_time);
}

void 
BBufferFilter::Preroll()
{
	return;
}

void 
BBufferFilter::SetTimeSource(BTimeSource *time_source)
{
	BMediaNode::SetTimeSource(time_source);
}

/********* protected BBufferConsumer functions *********/

status_t 
BBufferFilter::AcceptFormat(const media_destination &dest, media_format *format)
{
/*
	check to see if the destination matches one of your input destinations 
		if not return B_MEDIA_BAD_DESTINATION 

	then check the format->type field and if set to B_MEDIA_NO_TYPE
		set it to your preferred type
	if format->type is not your preferred format type
		return B_MEDIA_BAD_FORMAT
	
	Step through the ENTIRE format structure.  For every field check to
	see if you support a filled in value, if not return B_MEDIA_BAD_FORMAT
	If it is a wildcard, fill it in with your preferred information
	If it is supported, move to the next field.
	If the format is supported, return B_OK
*/

	if (dest == fIn.in.destination) {
		/* check the format */
		return B_MEDIA_BAD_FORMAT;
	}
	else return B_MEDIA_BAD_DESTINATION;
}

status_t 
BBufferFilter::GetNextInput(int32 *cookie, media_input *out_input)
{
	if (*cookie != 0)
		return B_ERROR;
	else {
		*cookie = 1;
		*out_input = fIn.in;
		return B_OK;
	}
}

void 
BBufferFilter::DisposeInputCookie(int32 cookie)
{
	cookie = 0;
}

void 
BBufferFilter::BufferReceived(BBuffer *buffer)
{
	CALL("BBufferFilter::BufferReceived\n");
	if (!fRunning) {
		buffer->Recycle();
		return;
	}
	
	// add to both outgoing queues
	fEvents->PushEvent(buffer->Header()->start_time, BTimedEventQueue::B_HANDLE_BUFFER,
						buffer, BTimedEventQueue::B_RECYCLE, 0);

	INFO("now: %Ld start_time: %Ld\n", TimeSource()->Now(), buffer->Header()->start_time);
}

void 
BBufferFilter::ProducerDataStatus(const media_destination &for_whom, int32 status, bigtime_t at_media_time)
{
	if (for_whom != fIn.in.destination)
		return;
	/* media time??????? */
	fEvents->PushEvent(at_media_time, BTimedEventQueue::B_DATA_STATUS,
					&fIn.in.destination, BTimedEventQueue::B_NO_CLEANUP, status);
}

status_t 
BBufferFilter::GetLatencyFor(const media_destination &for_whom, bigtime_t *out_latency, media_node_id *out_timesource)
{
	if (for_whom != fIn.in.destination)
		return B_MEDIA_BAD_DESTINATION;

	GetLatency(out_latency);
	*out_timesource = TimeSource()->ID();
	return B_OK;
}

status_t 
BBufferFilter::Connected(const media_source &producer, const media_destination &where, const media_format &with_format, media_input *out_input)
{
	if (where != fIn.in.destination)
		return B_MEDIA_BAD_DESTINATION;
	if (producer == fIn.in.source)
		return B_MEDIA_ALREADY_CONNECTED;
	if (fIn.in.source != media_source::null)
		return B_MEDIA_NOT_CONNECTED;
		
	fIn.in.source = producer;
	fIn.in.format = with_format;
	*out_input = fIn.in;
	return B_OK;
}

void 
BBufferFilter::Disconnected(const media_source &producer, const media_destination &where)
{
	if (where == fIn.in.destination && producer == fIn.in.source) {
		// disconnect the connection
		fIn.in.source = media_source::null;
		fIn.in.format.type = B_MEDIA_UNKNOWN_TYPE;
		fIn.data_status = B_DATA_NOT_AVAILABLE;
	} else return;
}

status_t 
BBufferFilter::FormatChanged(const media_source &producer, const media_destination &consumer, int32 from_change_format, const media_format &format)
{
	if (consumer != fIn.in.destination)
		return B_MEDIA_BAD_DESTINATION;
	if (producer != fIn.in.source)
		return B_MEDIA_BAD_SOURCE;
		
	fIn.in.format = format;
	/* do what ever work a format change requires */
	return B_OK;
}

/********* protected BBufferProducer functions *********/
status_t 
BBufferFilter::FormatSuggestionRequested(media_type type, int32 quality, media_format *format)
{
	if (type == B_MEDIA_RAW_AUDIO) {
		/* set *format to your preferred raw audio format */
		/* set the quality and return B_OK */
	} else if (type == B_MEDIA_RAW_VIDEO) {
		/* set *format to your preferred raw video format */
		/* set the quality and return B_OK */
	}
	/* set any other appropriate types that you support */
	/* return B_ERROR for any type that you do not support */
	
	return B_ERROR;
}

status_t 
BBufferFilter::FormatProposal(const media_source &output, media_format *format)
{
	if (output == fOut.out.source) {
		/* fill in your flexible format. */
		/* set any fields in format that you require for that output */
		/* return B_OK */
	}
	else return B_MEDIA_BAD_SOURCE;
}

status_t 
BBufferFilter::FormatChangeRequested(const media_source &source, const media_destination &dest, media_format *io_format, int32 *out_change_count)
{
	if (source == fOut.out.source) {
		if (dest != fOut.out.destination)
			return B_MEDIA_BAD_DESTINATION;
		/* change your output format to ioformat, filling in any wildcards */
		/* return the new fOut.out.format and issue a new change tag with */
		/* MintChangeTag */
		/* return B_ERROR if you do not support the requested format */
		return B_ERROR;
	}
	else return B_MEDIA_BAD_SOURCE;
}

status_t 
BBufferFilter::GetNextOutput(int32 *cookie, media_output *out_output)
{
	char format_string[256];
	string_for_format(out_output->format, format_string, 256);
	FORMAT("BBufferFilter::GetNextOutput(%s)\n", format_string);

	if (*cookie == 0) {
		*cookie = 1;
		*out_output = fOut.out;
		return B_OK;
	}
	else return B_ERROR;
}

status_t 
BBufferFilter::DisposeOutputCookie(int32 cookie)
{
	cookie = 0;
}

status_t 
BBufferFilter::SetBufferGroup(const media_source &for_source, BBufferGroup *group)
{
	CALL("BBufferFilter::SetBufferGroup\n");
	if (for_source == fOut.out.source) {
		/* if you want to support buffer groups created by other people */
		/* check the size of the buffers and make sure they are large enough */
		/* to hold your information. also check to make sure there are enough */
		/* buffers to handle your latency issues */
		/* if you accept this buffer group you must reclaim all outstanding */
	/* 	if (fOut.buffers && fOut.own_buffers == true) {
			fOut.buffers->ReclaimAllBuffers();
			fOut.buffers == NULL;
		}
		
		fOut.buffers = group;
		fOut.own_buffers = true;
		
	*/
		return B_ERROR;
	}
	else return B_MEDIA_BAD_SOURCE;
}

status_t 
BBufferFilter::VideoClippingChanged(const media_source &for_source, int16 num_shorts, int16 *clip_data, const media_video_display_info &display, int32 *out_from_change_count)
{
	if (for_source != fOut.out.source)
		return B_MEDIA_BAD_SOURCE;
	/* if you handle video make the appropriate change and update the change */
	/* tag and return B_OK */
	return B_ERROR;
}

status_t 
BBufferFilter::GetLatency(bigtime_t *out_latency)
{
	media_node_id id;
	FindLatencyFor(fOut.out.destination, &fOut.downstream_latency, &id);
	PublishLatency(&fOut.processing_latency);
	
	fIn.latency = fSchedulingLatency + fOut.downstream_latency + fOut.processing_latency;
	*out_latency = fIn.latency;
	return B_OK;
}

status_t 
BBufferFilter::PrepareToConnect(const media_source &what, const media_destination &where, media_format *format, media_source *out_source, char *out_name)
{
	if (what == fOut.out.source) {
		if (where == fOut.out.destination)
			return B_MEDIA_ALREADY_CONNECTED;
		if (fOut.out.destination != media_destination::null)
			return B_MEDIA_NOT_CONNECTED;
		/* reserve the output for this connection */
		fOut.out.destination = where;
		/* check to see if the format is compatible */
		/* set the format appropriately */
		fOut.out.format = *format;
		*out_source = fOut.out.source;
		strcpy(out_name, fOut.out.name);
		return B_OK;
	}
	else return B_MEDIA_BAD_SOURCE;
}

void 
BBufferFilter::Connect(status_t error, const media_source &source, const media_destination &dest, const media_format &format, char *io_name)
{
	if (source == fOut.out.source && error == B_OK && fOut.out.destination == dest) {
		fOut.out.format = format;
		strcpy(io_name, fOut.out.name);
		if (fOut.own_buffers) {
			if (fOut.buffers) {
				// reclaim all of our old buffers
				fOut.buffers->ReclaimAllBuffers();
				// delete the buffer group
				delete fOut.buffers; fOut.buffers = NULL;
			}
			fOut.buffers = CreateBufferGroup(fOut.out);
		}
	}
	else return;
}

void 
BBufferFilter::Disconnect(const media_source &what, const media_destination &where)
{
	if (what == fOut.out.source) {
		fOut.enabled = false;
		fOut.out.destination = media_destination::null;
		// if we don't own the buffers, forget about them
		if (!fOut.own_buffers)
			fOut.own_buffers = NULL;
	}
}

void 
BBufferFilter::LateNoticeReceived(const media_source &what, bigtime_t how_much, bigtime_t performance_time)
{
	if (what == fOut.out.source) {
		switch(RunMode()) {
			case B_OFFLINE:
			case B_DECREASE_PRECISION:
			case B_INCREASE_LATENCY:
			case B_DROP_DATA:
			case B_RECORDING:
				break;
		}
	}
	else return;
}

void 
BBufferFilter::EnableOutput(const media_source &what, bool enabled, int32 *change_tag)
{
	if (what == fOut.out.source) {
		fOut.enabled = enabled;
		/* mint a new change tag and return it */
	
	}
	else return;
}

status_t 
BBufferFilter::SetPlayRate(int32 numer, int32 denom)
{
	return B_ERROR;
}

