#include "PCIDeviceView.h"
#include <TextControl.h>
#include "pci_code.h"

/*
	Class: BTSPCIDeviceView
	
	This view display a nice representation of PCI
	device information.
*/

BTSPCIDeviceView::BTSPCIDeviceView(BRect frame, const char *name)
	: BView(frame, name, B_FOLLOW_ALL,0)
{
	BRect aRect(10,10,66,30);
	fBusText = new BTextControl(aRect,"fBusText","Bus", "", NULL);
	fBusText->SetAlignment(B_ALIGN_RIGHT, B_ALIGN_LEFT);
	aRect.Set(70,10,170,30);
	fDeviceType = new BTextControl(aRect,"fDeviceType","Type", "", NULL);
	fDeviceType->SetAlignment(B_ALIGN_RIGHT, B_ALIGN_LEFT);
	aRect.OffsetBy(104,0);
	fFunctionText = new BTextControl(aRect,"fFunctionText","Function", "", NULL);
	fFunctionText->SetAlignment(B_ALIGN_RIGHT, B_ALIGN_LEFT);
	
	aRect.Set(5,30,406,50);
	fVendorText = new BStringView(aRect,"fVendorText", "");

	aRect.OffsetBy(10,20);
	fClassText  = new BStringView(aRect,"fClassText","");

	aRect.OffsetBy(0,20);
	fLineText  = new BStringView(aRect,"fLineText","");
	aRect.OffsetBy(0,20);
	fRomText  = new BStringView(aRect,"fRomText","");
	aRect.OffsetBy(0,20);
	fInterruptText  = new BStringView(aRect,"fInterruptText","");
	aRect.OffsetBy(0,20);
	fSubsystemText  = new BStringView(aRect,"fSubsystemText","");

	for (int i=0; i<6; i++)
	{
		aRect.OffsetBy(0,20);
		fRegisters[i] = new BStringView(aRect,"fRegisterView","");
		fRegisters[i]->SetFont(be_fixed_font);
		AddChild(fRegisters[i]);
	}
	
	AddChild(fBusText);
	AddChild(fDeviceType);
	AddChild(fFunctionText);
	AddChild(fVendorText);
	AddChild(fClassText);
	AddChild(fLineText);
	AddChild(fRomText);
	AddChild(fInterruptText);
	AddChild(fSubsystemText);
}

void
BTSPCIDeviceView::Clear()
{
	fBusText->SetText("");
	fDeviceType->SetText("");
	fFunctionText->SetText("");
	fVendorText->SetText("");
	fClassText->SetText("");
	fLineText->SetText("");
	fInterruptText->SetText("");
	fRomText->SetText("");
	fSubsystemText->SetText("");

	for (int i = 0; i<6; i++)
	{
		fRegisters[i]->SetText("");
	}
}

/*
	Method: SetDevice()
	
	This view can be used at any time to display
	a pci_info structure.  Using SetDevice will display
	the information for that device.  In some cases it will
	turn the well known numeric values into the textual equivalents.
	This makes the information a lot more useful and readable by
	humans.
*/

void	
BTSPCIDeviceView::SetDevice(const pci_info &info)
{
	char buff[256];
	char vendbuff[256];
	char chipbuff[256];
	fDeviceInfo = info;

	Clear();
	
	sprintf(buff, "%.2x", info.bus);
	fBusText->SetText(buff);
	sprintf(buff, "%.2x", info.device);
	fDeviceType->SetText(buff);
	sprintf(buff, "%.2x", info.function);
	fFunctionText->SetText(buff);
	
	char *chipPtr = FindPCIDeviceString(info.vendor_id, info.device_id);
	char *vendPtr = FindPCIVendorString(info.vendor_id);
	
	if (vendPtr)
		strcpy(vendbuff, vendPtr);
	else
		sprintf(vendbuff,"Unknown: %x\n", (int)info.vendor_id);
		
	if (chipPtr)
		strcpy(chipbuff, chipPtr);
	else
		sprintf(chipbuff,"Device: %x\n", (int)info.device_id);

	sprintf(buff, "%s, %s Rev: %.2x", vendbuff, chipbuff, info.revision);
	fVendorText->SetText(buff);
	
	sprintf (buff,"API = %s  FUNCTION = %.2x ", 
		FindPCIAPIString(info.class_base,info.class_sub, info.class_api),
		info.class_sub);
	fClassText->SetText(buff);

	sprintf (buff,"line_size=%.2x latency_timer=%.2x header_type = %.2x BIST=%.2x", 
		info.line_size, info.latency, info.header_type, info.bist);
	fLineText->SetText(buff);

	sprintf (buff,"interrupt_line=%.2x interrupt_pin=%.2x min_grant=%.2x max_latency=%.2x",
		info.u.h0.interrupt_line, info.u.h0.interrupt_pin, info.u.h0.min_grant,
		info.u.h0.max_latency);
	fInterruptText->SetText(buff);

	if (info.u.h0.rom_size > 0)
	{
		sprintf (buff,"rom_base=%.8x  pci %.8x size=%xK",
		info.u.h0.rom_base, info.u.h0.rom_base_pci, info.u.h0.rom_size/1024);
	} else
		strcpy(buff, "NO ROM");
	fRomText->SetText(buff);
	
	sprintf (buff,"cardbus_cis=%.8x subsystem_id=%.4x subsystem_vendor_id=%.4x",
		info.u.h0.cardbus_cis, info.u.h0.subsystem_id, info.u.h0.subsystem_vendor_id);
	fSubsystemText->SetText(buff);

	for (int i = 0; i<6; i++)
	{
		if (info.u.h0.base_registers[i] &&
			info.u.h0.base_registers_pci [i])
		{
		sprintf (buff,"reg %d: host %.8x pci %.8x size %xK, flags %.2x",
			i, info.u.h0.base_registers[i], info.u.h0.base_registers_pci [i], 
			info.u.h0.base_register_sizes[i]/1024, info.u.h0.base_register_flags[i]);
		fRegisters[i]->SetText(buff);
		}
	}
}

