//
// Menu World
//
// A sample program demonstrating the construction
// of a menu bar in a window.
//
// Copyright 1998, Be Incorporated
//
// Written by: Eric Shepherd
//

#include <Application.h>
#include <Window.h>
#include <View.h>
#include <MenuBar.h>
#include <Menu.h>
#include <MenuItem.h>
#include <string.h>

// Application's signature

const char *APP_SIGNATURE				= "application/x-vnd.Be-MenuWorld";

// Messages for menu commands

const uint32 MENU_FILE_NEW				= 'MFnw';
const uint32 MENU_FILE_OPEN				= 'MFop';
const uint32 MENU_FILE_SAVE				= 'MFsv';
const uint32 MENU_FILE_SAVEAS			= 'MFsa';
const uint32 MENU_FILE_PAGESETUP		= 'MFps';
const uint32 MENU_FILE_PRINT			= 'MFpr';

const uint32 MENU_OPT_HELLO				= 'MOhl';

const char *STRING_HELLO				= "Hello World!";
const char *STRING_GOODBYE				= "Goodbye World!";

//
// HelloView class
//
// This class defines the view in which the "Hello World"
// message will be drawn.
//
class HelloView : public BView {
	public:
						HelloView(BRect frame);
		virtual void	Draw(BRect updateRect);
		void			SetString(const char *s);
	
	private:
		char			message[128];
};


//
// HelloView::HelloView
//
// Constructs the view we'll be drawing in.
// As you see, it doesn't do much.
//
HelloView::HelloView(BRect frame)
			: BView(frame, "HelloView", B_FOLLOW_ALL_SIDES, B_WILL_DRAW) {
	SetString(STRING_HELLO);
}


//
// HelloView::SetString
//
// Sets the message to draw in the view.
//
void HelloView::SetString(const char *s) {
	if (strlen(s) < 127) {
		strcpy(message, s);
	}
}

//
// HelloView::Draw
//
// This function is called whenever our view
// needs to be redrawn.  This happens only because
// we specified B_WILL_DRAW for the flags when
// we created the view (see the constructor).
//
// The updateRect is the rectangle that needs to be
// redrawn.  We're ignoring it, but you can use it to
// speed up your refreshes for more complex programs.
//
void HelloView::Draw(BRect updateRect) {
	MovePenTo(BPoint(20,75));			// Move pen
	DrawString(message);
}


//
// HelloWindow class
//
// This class defines the hello world window.
//
class HelloWindow : public BWindow {
	public:
						HelloWindow(BRect frame);
		virtual bool	QuitRequested();
		virtual void	MessageReceived(BMessage *message);
	
	private:		
		BMenuBar		*menubar;
		HelloView		*helloview;
};


//
// HelloWindow::HelloWindow
//
// Constructs the window we'll be drawing into.
//
HelloWindow::HelloWindow(BRect frame)
			: BWindow(frame, "MenuWorld", B_TITLED_WINDOW,
				B_NOT_RESIZABLE|B_NOT_ZOOMABLE) {
	BRect r;
	BMenu *menu;
	BMenuItem *item;
	
	// Add the drawing view
	
	r = Bounds();
	r.top = 20;
	AddChild(helloview = new HelloView(r));
	
	// Add the menu bar
	
	r.top = 0;
	r.bottom = 19;
	menubar = new BMenuBar(r, "menu_bar");
	AddChild(menubar);
	
	// Add File menu to menu bar
	
	menu = new BMenu("File");
	menu->AddItem(new BMenuItem("New", new BMessage(MENU_FILE_NEW), 'N'));
	menu->AddItem(new BMenuItem("Open" B_UTF8_ELLIPSIS,
					new BMessage(MENU_FILE_OPEN), 'O'));
	menu->AddItem(new BMenuItem("Close", new BMessage(B_QUIT_REQUESTED), 'W'));
	menu->AddSeparatorItem();
	menu->AddItem(new BMenuItem("Save", new BMessage(MENU_FILE_SAVE), 'S'));
	menu->AddItem(new BMenuItem("Save as" B_UTF8_ELLIPSIS,
					new BMessage(MENU_FILE_SAVEAS)));
	menu->AddSeparatorItem();
	menu->AddItem(new BMenuItem("Page Setup" B_UTF8_ELLIPSIS,
					new BMessage(MENU_FILE_PAGESETUP)));
	menu->AddItem(new BMenuItem("Print" B_UTF8_ELLIPSIS,
					new BMessage(MENU_FILE_PRINT), 'P'));
	menu->AddSeparatorItem();
	menu->AddItem(new BMenuItem("Quit", new BMessage(B_QUIT_REQUESTED), 'Q'));
	menubar->AddItem(menu);

	// Add Options menu to menu bar
	
	menu = new BMenu("Options");
	item=new BMenuItem("Say Hello", new BMessage(MENU_OPT_HELLO));
	item->SetMarked(true);
	menu->AddItem(item);
	menubar->AddItem(menu);
	
	Show();
}


//
// HelloWindow::MessageReceived
//
// Called when a message is received by our
// application.
//
void HelloWindow::MessageReceived(BMessage *message) {
	switch(message->what) {
		case MENU_OPT_HELLO:
			{
				BMenuItem *item;
				const char *s;
				bool mark;
				
				message->FindPointer("source", (void **) &item);
				if (item->IsMarked()) {
					s = STRING_GOODBYE;
					mark = false;
				}
				else {
					s = STRING_HELLO;
					mark = true;
				}
				helloview->SetString(s);
				item->SetMarked(mark);
				helloview->Invalidate();
			}
			break;
		
		default:
			BWindow::MessageReceived(message);
			break;
	}
}


//
// HelloWindow::QuitRequested
//
// Here we just give permission to close the window.
//
bool HelloWindow::QuitRequested() {
	be_app->PostMessage(B_QUIT_REQUESTED);
	return true;
}


//
// HelloApp class
//
// This class, derived from BApplication, defines the
// Hello World application itself.
//
class HelloApp : public BApplication {
	public:
						HelloApp();
};


//
// HelloApp::HelloApp
//
// The constructor for the HelloApp class.  This
// will create our window.
//
HelloApp::HelloApp()
			: BApplication(APP_SIGNATURE) {
	BRect windowRect;
	windowRect.Set(50,50,349,399);
	
	new HelloWindow(windowRect);
}


//
// main
//
// The main() function's only real job in a basic BeOS
// application is to create the BApplication object
// and run it.
//
void main(void) {
	HelloApp theApp;		// The application object
	theApp.Run();
}



