//
// Text World
//
// A sample program that's gradually evolving into
// a real text editor application.
//
// Copyright 1998, Be Incorporated
//
// Written by: Eric Shepherd
//

#include <Application.h>
#include <Messenger.h>
#include <Message.h>
#include <Roster.h>
#include <Window.h>
#include <View.h>
#include <MenuBar.h>
#include <Menu.h>
#include <MenuItem.h>
#include <TextView.h>
#include <ScrollView.h>
#include <string.h>
#include <stdio.h>

// Application's signature

const char *APP_SIGNATURE				= "application/x-vnd.Be-TextWorld";

// Messages for window registry with application

const uint32 WINDOW_REGISTRY_ADD		= 'WRad';
const uint32 WINDOW_REGISTRY_SUB		= 'WRsb';
const uint32 WINDOW_REGISTRY_ADDED		= 'WRdd';

// Messages for menu commands

const uint32 MENU_FILE_NEW				= 'MFnw';
const uint32 MENU_FILE_OPEN				= 'MFop';
const uint32 MENU_FILE_CLOSE			= 'MFcl';
const uint32 MENU_FILE_SAVE				= 'MFsv';
const uint32 MENU_FILE_SAVEAS			= 'MFsa';
const uint32 MENU_FILE_PAGESETUP		= 'MFps';
const uint32 MENU_FILE_PRINT			= 'MFpr';
const uint32 MENU_FILE_QUIT				= 'MFqu';


//
// TextWindow class
//
// This class defines the hello world window.
//
class TextWindow : public BWindow {
	public:
						TextWindow(BRect frame);
						~TextWindow();
		virtual bool	QuitRequested();
		virtual void	MessageReceived(BMessage *message);
		virtual void	FrameResized(float width, float height);
	
	private:
		void			Register(bool need_id);
		void			Unregister(void);
		
		BMenuBar		*menubar;
		BTextView		*textview;
		BScrollView		*scrollview;
};


//
// TextWindow::TextWindow
//
// Constructs the window we'll be drawing into.
//
TextWindow::TextWindow(BRect frame)
			: BWindow(frame, "Untitled ", B_DOCUMENT_WINDOW, 0) {
	BRect r;
	BMenu *menu;
	
	// Add the menu bar
	
	r = Bounds();
	menubar = new BMenuBar(r, "menu_bar");

	// Add File menu to menu bar
	
	menu = new BMenu("File");
	menu->AddItem(new BMenuItem("New", new BMessage(MENU_FILE_NEW), 'N'));
	menu->AddItem(new BMenuItem("Open" B_UTF8_ELLIPSIS,
					new BMessage(MENU_FILE_OPEN), 'O'));
	menu->AddItem(new BMenuItem("Close", new BMessage(MENU_FILE_CLOSE), 'W'));
	menu->AddSeparatorItem();
	menu->AddItem(new BMenuItem("Save", new BMessage(MENU_FILE_SAVE), 'S'));
	menu->AddItem(new BMenuItem("Save as" B_UTF8_ELLIPSIS,
					new BMessage(MENU_FILE_SAVEAS)));
	menu->AddSeparatorItem();
	menu->AddItem(new BMenuItem("Page Setup" B_UTF8_ELLIPSIS,
					new BMessage(MENU_FILE_PAGESETUP)));
	menu->AddItem(new BMenuItem("Print" B_UTF8_ELLIPSIS,
					new BMessage(MENU_FILE_PRINT), 'P'));
	menu->AddSeparatorItem();
	menu->AddItem(new BMenuItem("Quit", new BMessage(MENU_FILE_QUIT), 'Q'));
	menubar->AddItem(menu);
	AddChild(menubar);

	// Add the text view
	
	BRect textframe = Bounds();
	textframe.top = menubar->Bounds().bottom + 1.0;
	textframe.right -= B_V_SCROLL_BAR_WIDTH;
	BRect textrect = textframe;
	textrect.OffsetTo(B_ORIGIN);
	r.InsetBy(3.0,3.0);
	textview = new BTextView(textframe, "text_view", textrect,
				B_FOLLOW_ALL_SIDES, B_WILL_DRAW|B_PULSE_NEEDED);
	AddChild(scrollview = new BScrollView("scroll_view", textview,
				B_FOLLOW_ALL_SIDES, 0, false, true, B_NO_BORDER));
	textview->MakeFocus(true);

	// Tell the application that there's one more window
	// and get the number for this untitled window.
	
	Register(true);
	Show();
}


//
// TextWindow::FrameResized
//
// Adjust the size of the BTextView's text rectangle
// when the window is resized.
//
void TextWindow::FrameResized(float width, float height) {
	BRect textrect = textview->TextRect();
	
	textrect.right = textrect.left + (width - B_V_SCROLL_BAR_WIDTH - 3.0);
	textview->SetTextRect(textrect);
}


//
// TextWindow::~TextWindow
//
// Destruct the window.  This calls Unregister().
//
TextWindow::~TextWindow() {
	Unregister();
}


//
// TextWindow::MessageReceived
//
// Called when a message is received by our
// application.
//
void TextWindow::MessageReceived(BMessage *message) {
	switch(message->what) {
		case WINDOW_REGISTRY_ADDED:
			{
				char s[22];
				int32 id = 0;
				if (message->FindInt32("new_window_number", &id) == B_OK) {
					sprintf(s, "Untitled %ld", id);
					SetTitle(s);
				}
			}
			break;
			
		case MENU_FILE_NEW:
			{
				BRect r;
				r = Frame();
				r.OffsetBy(20,20);
				new TextWindow(r);
			}
			break;
		case MENU_FILE_CLOSE:
			Quit();
			break;
		case MENU_FILE_QUIT:
			be_app->PostMessage(B_QUIT_REQUESTED);
			break;
		
		default:
			BWindow::MessageReceived(message);
			break;
	}
}


//
// TextWindow::Register
//
// Since MessageWorld can have multiple windows and
// we need to know when there aren't any left so the
// application can be shut down, this function is used
// to tell the application that a new window has been
// opened.
//
// If the need_id argument is true, we'll specify true
// for the "need_id" field in the message we send; this
// will cause the application to send back a
// WINDOW_REGISTRY_ADDED message containing the window's
// unique ID number.  If this argument is false, we won't
// request an ID.
//
void TextWindow::Register(bool need_id) {
	BMessenger messenger(APP_SIGNATURE);
	BMessage message(WINDOW_REGISTRY_ADD);
	
	message.AddBool("need_id", need_id);
	messenger.SendMessage(&message, this);
}


//
// TextWindow::Unregister
//
// Unregisters a window.  This tells the application that
// one fewer windows are open.  The application will
// automatically quit if the count goes to zero because
// of this call.
//
void TextWindow::Unregister(void) {
	BMessenger messenger(APP_SIGNATURE);
	
	messenger.SendMessage(new BMessage(WINDOW_REGISTRY_SUB));
}


//
// TextWindow::QuitRequested
//
// Here we just give permission to close the window.
//
bool TextWindow::QuitRequested() {
	return true;
}


//
// TextApp class
//
// This class, derived from BApplication, defines the
// Hello World application itself.
//
class TextApp : public BApplication {
	public:
						TextApp();
		virtual void	MessageReceived(BMessage *message);
	
	private:
		int32			window_count;
		int32			next_untitled_number;
};


//
// TextApp::TextApp
//
// The constructor for the TextApp class.  This
// will create our window.
//
TextApp::TextApp()
			: BApplication(APP_SIGNATURE) {
	BRect windowRect;
	windowRect.Set(50,50,349,399);
	
	window_count = 0;			// No windows yet
	next_untitled_number = 1;	// Next window is "Untitled 1"
	new TextWindow(windowRect);
}


//
// TextApp::MessageReceived
//
// Handle incoming messages.  In particular, handle the
// WINDOW_REGISTRY_ADD and WINDOW_REGISTRY_SUB messages.
//
void TextApp::MessageReceived(BMessage *message) {
	switch(message->what) {
		case WINDOW_REGISTRY_ADD:
			{
				bool need_id = false;
				
				if (message->FindBool("need_id", &need_id) == B_OK) {
					if (need_id) {
						BMessage reply(WINDOW_REGISTRY_ADDED);

						reply.AddInt32("new_window_number", next_untitled_number);
						message->SendReply(&reply);
						next_untitled_number++;
					}
					window_count++;
				}
				break;
			}
		case WINDOW_REGISTRY_SUB:
			window_count--;
			if (!window_count) {
				Quit();
			}
			break;
		default:
			BApplication::MessageReceived(message);
			break;
	}
}


//
// main
//
// The main() function's only real job in a basic BeOS
// application is to create the BApplication object
// and run it.
//
int main(void) {
	TextApp theApp;		// The application object
	theApp.Run();
	return 0;
}



