#ifndef YAGS3D_H
#define YAGS3D_H

#include "yags2d.h"

//
// 3D Point definitions
//
typedef struct Point3Ds 
{
	U16 x;
	U16 y;
	U16 z;
} Point3Ds;

typedef struct Point3Dl 
{
	U32 x;
	U32 y;
	U32 z;
} Point3Dl;

typedef struct Point3Df 
{
	float x;
	float y;
	float z;
} Point3Df, Vertex3Df, Vector3Df;


// Homogenous 3D point
typedef struct Point3Dhf 
{
	float x;	// Z component
	float y;	// Y component
	float z;	// Z component
	float w;	// homogenous part
} Point3Dhf, Vertex3Dhf, Vector3Dhf;

typedef struct dir_3d_type
{
	U16	x;		// angle relative to x-axis
	U16 y;		// angle relative to y-axis
	U16 z;		// angle relative to z-axis
} Orientation3Ds;

#define MAX_VERTICES_PER_OBJECT (100)
#define MAX_FACES_PER_OBJECT (200)
#define MAX_POINTS_PER_POLY (4)

typedef struct Polygon3D
{
	U16 fNumVertices;
	U16	fVertexList[MAX_POINTS_PER_POLY];	// Max 4 vertices per polygon
	U08	fColor;			// color of polygon
	U08	fShade;			// computed shade of polygon
	U08	fShading;		// kind of shading, flat or continuous
	U08	fTwoSided;		// is it one sided or two sided
	U08	fVisible;		// used to remove back faces
	U08	fActive;		// used to turn faces on and off
	U08	fClipped;		// flags whether it is clipped or removed
	float	fNormalLength;	// pre-computed normal length
} Polygon3D;


typedef struct Facet3D
{
	U16 fNumVertices;
	U08	fColor;			// color of polygon
	U08	fShade;			// computed shade of polygon
	U08	fShading;		// kind of shading, flat or continuous
	U08	fTwoSided;		// is it one sided or two sided
	U08	fVisible;		// used to remove back faces
	U08	fActive;		// used to turn faces on and off
	U08	fClipped;		// flags whether it is clipped or removed
	float	fNormalLength;	// pre-computed normal length
	Point3Dhf	fVertexList[MAX_POINTS_PER_POLY];
} Facet3D;


typedef struct Object3D
{
	U32	fID;
	U16 fNumVertices;
	
	Vertex3Dhf fVerticesLocal[MAX_VERTICES_PER_OBJECT];
	Vertex3Dhf fVerticesWorld[MAX_VERTICES_PER_OBJECT];
	Vertex3Dhf fVerticesCamera[MAX_VERTICES_PER_OBJECT];
	
	U16	fNumFaces;
	Polygon3D fFaces[MAX_FACES_PER_OBJECT];
	
	float	fRadius;	// average radius of object
	int		fState;		// State of object
	Point3Dhf fOrigin;	// World position
} Object3D;

typedef struct FacetedObject3D
{
	U32	fID;
	U16 fNumVertices;
	
	Vertex3Dhf fVerticesLocal[MAX_VERTICES_PER_OBJECT];
	Vertex3Dhf fVerticesWorld[MAX_VERTICES_PER_OBJECT];
	Vertex3Dhf fVerticesCamera[MAX_VERTICES_PER_OBJECT];
	
	U16	fNumFaces;
	Facet3D fFaces[MAX_FACES_PER_OBJECT];
	
	float	fRadius;	// average radius of object
	int		fState;		// State of object
	Point3Dhf fOrigin;	// World position
} FacetedObject3D;

/*
** Camera Definition
*/

typedef struct Camera
{
	float  clip_near_z;		// the near or hither clipping plane
	float  clip_far_z;		// the far or yon clipping plane
	int viewing_distance;	// distance of projection plane from camera
	Point3Dhf view_point;	// position of camera
	Vector3Df light_source;	// position of point light source
	float ambient_light;	// ambient light level
	Orientation3Ds view_angle;		// angle of camera
	matrix_4x4f global_view;	// the global inverse world to camera
							// matrix
} Camera;

//
// Camera related functions
//
Camera *CreateCamera();
void DestroyCamera(Camera*);

/*
** Matrix operations
*/
inline void 
Vec3D_Make(const Point3Dhf init, const Point3Dhf term, 
	Vector3Dhf *result)
{
	result->x = term.x - init.x;
	result->y = term.y - init.y;
	result->z = term.z - init.z;
}

// compute magnitude of vector
inline float 
Vec3D_Mag(const Vector3Dhf v)
{
	return ((float)sqrt((v.x*v.x) + (v.y*v.y) + (v.z*v.z)));	
}

inline float 
Vec3D_Dot(const Vector3Dhf u, const Vector3Dhf v)
{
	return ((u.x*v.x) + (u.y*v.y) + (u.x*v.z));
}

inline void 
Vec3D_Cross(const Vector3Dhf u, const Vector3Dhf v, Vector3Dhf *normal)
{
	normal->x = (u.y*v.z - u.z*v.y);
	normal->y = (u.x*v.z - u.z*v.x);
	normal->z = (u.x*v.y - u.y*v.x);
}

inline void
Vec3D_Add(Vector3Dhf *u, const Vector3Dhf v)
{
	u->x += v.x;
	u->y += v.y;
	u->z += v.z;
}

inline void
Vec3D_Sub(Vector3Dhf *u, const Vector3Dhf v)
{
	u->x -= v.x;
	u->y -= v.y;
	u->z -= v.z;
}

inline void
Mat4x4_Identity(matrix_4x4f *mat)
{
	// zero out all the values
	memset(mat,0,16*sizeof(float));
	// Then set the diagonals to 1
	*mat[0][0] = *mat[1][1] = *mat[2][2] = *mat[3][3] = 1;
}

// Zero out all the values in the matrix
inline void
Mat4x4_Zero(matrix_4x4f *mat)
{
	// zero out all the values
	memset(mat,0,16*sizeof(float));
}

inline void
Mat4x4_Copy(matrix_4x4f *dst, const matrix_4x4f *src)
{
	// zero out all the values
	memcpy(dst, src,16*sizeof(float));
}

void
Mat4x4_Mul4x4(const matrix_4x4f a, const matrix_4x4f b,
	matrix_4x4f res);

void
Mat1x4_Mul4x4(const matrix_1x4f a, const matrix_4x4f b, matrix_4x4f v);

void
Mat1x4_Mul4x4(const matrix_1x4f a, const matrix_4x4f b, matrix_1x4f v);



// Graphics primitives
void
Y3DLine(pixel_buffer screenInfo, 
	const Point3Dhf pt1, const Point3Dhf pt2, 
	const U08 aColor, const Camera aCamera);

void
Y3DRenderWireObject(pixel_buffer screenInfo,
	const FacetedObject3D *aObject,
	const U08 aColor, Camera *aCamera);
	
#endif
