#ifndef YAGS2D_H
#define YAGS2D_H

#include <string.h>
#include <stdio.h>

#include "ygmacros.h"
#include "ygtypes.h"

typedef enum  {
	// Single Primitives
	Y2D_POINT = 1,
	Y2D_LINE,
	Y2D_TRIANGLE,
	Y2D_RECTANGLE,
	Y2D_QUAD,
	Y2D_POLYGON,
	Y2D_CIRCLE,
	Y2D_ELLIPSE,
	Y2D_ARC,
	Y2D_BEZIER,
	
	// Disjoint, multiple primitives
	Y2D_POINTS,
	Y2D_LINES,
	Y2D_TRIANGLES,
	Y2D_RECTANGLES,
	Y2D_QUADS,
	Y2D_POLYGONS,
	Y2D_ELLIPSES,
	Y2D_ARCS,
		
	Y2D_LINESTRIP,
	Y2D_LINELOOP,
	Y2D_TRIANGLESTRIP,
	Y2D_TRIANGLEFAN
} YAGS2DPOINTPRIM;

typedef enum {
	Y2D_BITMAP,
	Y2D_STRING,
	Y2D_BITMAPS,
	Y2D_STRINGS
} YAGS2DDRAWPRIM;

// Low level primitive drawing
// Pixel Drawing
//==================================================================
//	Y2DSetPixel
//
//	Assumptions
//	1) The x and y must be within the confines of the pixel_buffer
//==================================================================
inline void 
Y2DSetPixel(pixel_buffer screenInfo, 
			const U16 x, const U16 y, 
			const U08 aColor)
{	
	((U08 *)screenInfo.pixels)[y * screenInfo.bytes_per_row + x] = aColor;
}

inline void 
Y2DSetPixel(pixel_buffer screenInfo, const Point2Ds aPoint, const U08 aColor)
{	
	((U08 *)screenInfo.pixels)[aPoint.y * screenInfo.bytes_per_row + aPoint.x] = aColor;
}

void
Y2DPoints(pixel_buffer screenInfo,
	const Point2Ds *pts, const unsigned long npts, 
	const unsigned char aColor);

// Pixel Getting
//==================================================================
//	Y2DGetPixel
//
//	Assumptions
//	1) The x and y must be within the confines of the pixel_buffer
//==================================================================
inline void 
Y2DGetPixel(pixel_buffer screenInfo, 
			const U16 x, const U16 y, 
			U08 *aColor)
{	
	*aColor = ((U08 *)screenInfo.pixels)[y * screenInfo.bytes_per_row + x];
}

/*
inline void 
Y2DGetPixel(pixel_buffer screenInfo, const Point2Ds aPoint, U08 *aColor)
{	
	*aColor = ((U08 *)screenInfo.pixels)[aPoint.y * screenInfo.bytes_per_row + aPoint.x];
}
*/
//
// Line Drawing
//
//=================================================================
//
// Draws a horizontal run of pixels, then advances the bitmap pointer to
//   the first pixel of the next run. 
//=================================================================
inline void Y2DHSpan(U08 *rowPtr, const U08 aColor, const int spanLength)
{
	memset(rowPtr, aColor, spanLength);
	
/*
	U08 *base = rowPtr;
	U32 count = spanLength;
	
	// For really short lines just do straight byte
	// assignment.
	if (count < 8) 
	{
		while(count-- >= 0)
			*base++ = aColor;
		return;
	}

	// For longer lines, try to move 4 bytes at a time
	// first line up with a 4 byte boundary.
	while ((long)base & 0x03) 
	{
		*base++ = aColor;
		count--;
	}
	
	
	long	c;
	U08	*cPtr = (U08*)&c;
	cPtr[0]=aColor;cPtr[1]=aColor;cPtr[2]=aColor;cPtr[3]=aColor;
	
	// Do the bulk of the copying 4 bytes at a time
	while(count > 3) 
	{
		*(long *)base = c;
		base += 4;
		count -= 4;
	}

	// Clean up any remaining bytes
	while(count-- >= 0) 
		*base++ = aColor;
*/
}

void
Y2DPixelSpan(pixel_buffer screenInfo, const U16 x1, const U16 y1, 
		const U08 *pixels, const U16 spanLength);

//==================================================================
// Y2DHLine
//
// Draws a horizontal line
// Assumptions
//	1) The line must already be clipped to fit within the pixel buffer
//	2) x1 <= x2
//	3) y2 is ignored
//==================================================================
void
Y2DHLine(pixel_buffer screenInfo, const U16 x1, const U16 y1, 
		const U16 x2, const U16 y2, const U08 aColor);


//==================================================================
// Y2DVLine
//
// Draws a vertical line
// Assumptions
//	1) The line must already be clipped to fit within the pixel buffer
//	2) y1 <= y2
//	3) x2 is ignored
//==================================================================
inline void
Y2DVLine(pixel_buffer screenInfo, const U16 x1, const U16 y1, 
		const U16 x2, const U16 y2, const unsigned char aColor)
{
	U32 offset = y1*screenInfo.bytes_per_row+x1;
	
	for (U32 y = y1; y <= y2; y++)
	{
		((U08*)screenInfo.pixels)[offset] = aColor;
		offset += screenInfo.bytes_per_row;
	}
}

//==================================================================
// Y2DLine
//	
//	Draw an arbitrary line.  If the line is horizontal or vertical, 
//	or 45 degree diagonal, it will be drawn in an optimized way
//==================================================================
void 
Y2DLineBres(const U16 x1, const U16 y1,const U16 x2, const U16 y2, 
	const unsigned char Color,pixel_buffer screenInfo);

void 
Y2DLine(pixel_buffer screenInfo, const U16 x1, const U16 y1, 
		const U16 x2, const U16 y2, const U08 aColor);

int Y2DClipLine(int left, int top, int right, int bottom, 
	int *x1,int *y1,int *x2, int *y2);
//=================================================================
// Function: Y2DLines
//
// Draw a set of disjoint lines using the given color value
//=================================================================

inline void
Y2DLines(pixel_buffer screenInfo,
	const Point2Ds *pts, const unsigned long npts, 
	const U08 aColor)
{
	for (int counter =0; counter < npts; counter+=2)
	{
		Y2DLine(screenInfo, pts[counter].x, pts[counter].y, 
							pts[counter+1].x,pts[counter+1].y,
							aColor);
	}	
}


//=================================================================
// Function: Y2DBezier
//
// Draws a bezier curve
//=================================================================
void	
Y2DBezier(const U16 nPts, const Point2Df *pts, 
	const U08 aColor, const U16 iters, pixel_buffer scrInfo);

//=================================================================
// Function: Y2DLineStrip
//
// Draw a set of connected lines using the given color value
//=================================================================

inline void
Y2DLineStrip(pixel_buffer screenInfo,
	const Point2Ds *pts, const unsigned long npts, 
	const U08 aColor)
{
	for (long counter =0; counter < npts-1; counter++)
	{
		Y2DLine(screenInfo, pts[counter].x, pts[counter].y, 
							pts[counter+1].x,pts[counter+1].y,
							aColor);
	}	
}


//=================================================================
//
// Triangles
//
//=================================================================
void
Y2DTriangle(pixel_buffer screenInfo,
	const U16 x1, const U16 y1,
	const U16 x2, const U16 y2,
	const U16 x3, const U16 y3,
	const U08 aColor);

void
Y2DTopTriangleFill(pixel_buffer scrInfo,
	const U16 x1, const U16 y1,
	const U16 x2, const U16 y2,
	const U16 x3, const U16 y3,
	const U08 aColor);

void
Y2DBottomTriangleFill(pixel_buffer scrInfo,
	const U16 x1, const U16 y1,
	const U16 x2, const U16 y2,
	const U16 x3, const U16 y3,
	const U08 aColor);

void	
Y2DTriangleFill(pixel_buffer screenInfo,
	const U16 x1, const U16 y1,
	const U16 x2, const U16 y2,
	const U16 x3, const U16 y3,
	const U08 aColor);
	

//=================================================================
//
// Quadrilaterals
//
//=================================================================
void	
Y2DQuad(pixel_buffer scrInfo,
	const U16 x1, const U16 y1,
	const U16 x2, const U16 y2,
	const U16 x3, const U16 y3,
	const U16 x4, const U16 y4,
	const U08 aColor);

void	
Y2DQuadFill(pixel_buffer scrInfo,
	const U16 x1, const U16 y1,
	const U16 x2, const U16 y2,
	const U16 x3, const U16 y3,
	const U16 x4, const U16 y4,
	const U08 aColor);

//=================================================================
//
// Rectangles
//
//=================================================================
//=================================================================
// Function: Y2DRect
//
// Strokes the outline of a rectangle
//=================================================================
void
Y2DRect(pixel_buffer screenInfo,
	const U16 left, const U16 top,
	const U16 right, const U16 bottom,
	const U08 aColor);

//=================================================================
// Function: Y2DRectFill
//
// Fills the given rectangle with the given color
//=================================================================
void
Y2DRectFill(pixel_buffer screenInfo,
	const U16 left, const U16 top,
	const U16 right, const U16 bottom,
	const U08 aColor);

//=================================================================
// Function: Y2DClear
//
// Clears the entire buffer to the specified color
//=================================================================
void
Y2DClear(pixel_buffer screenInfo,
	const U08 aColor);

//=================================================================
//
// Polygons
//
//=================================================================
void
Y2DPolygon(pixel_buffer screenInfo,
	const Point2Ds* pts, const unsigned long npts,
	const U08 aColor);


//=================================================================
//
// Ellipse Drawing routines
//
//=================================================================
void
Y2DEllipse(pixel_buffer screenInfo,
	const U16 x, const U16 y,
	const U16 radiusX, const U16 radiusY, 
	const U08 aColor);

void
Y2DEllipseFill(pixel_buffer screenInfo,
	const U16 x, const U16 y,
	const U16 radiusX, const U16 radiusY, 
	const U08 aColor);

//=================================================================
//
// Circles
//
//=================================================================
inline void
Y2DCircle(pixel_buffer screenInfo,
	const U16 x, const U16 y,
	const U16 radius, const U08 aColor)
{
	Y2DEllipse(screenInfo, x, y, radius, radius, aColor);
}

inline void
Y2DCircleFill(pixel_buffer screenInfo,
	const U16 x, const U16 y,
	const U16 radius, const U08 aColor)
{
	Y2DEllipseFill(screenInfo, x, y, radius, radius, aColor);
}

//=================================================================
//
// Bitmaps
//
//=================================================================
void
Y2DCopyPixels(pixel_buffer dst, 
	const pixel_buffer src);

void
Y2DCopyPixels(pixel_buffer dst, 
	const pixel_buffer src, 
	const U16 x1, const U16 y1);

/*
void
Y2DCopyPixels(pixel_buffer dst, 
	const pixel_buffer src, 
	const U16 top, const U16 left,
	const U16 right, const U16 bottom);
*/
// Creation and destruction of pixel buffers
pixel_buffer *
Y2DCreatePixelBuffer(const U16 width, const U16 height);

void
Y2DDestroyPixelBuffer(pixel_buffer *aBuffer);


//=================================================================
//
// High Level Routines
//
// High level single point of entry to drawing device
//=================================================================
void	Y2DPrimitives(pixel_buffer, YAGS2DPOINTPRIM, const Point2Ds* pts, const unsigned long npts,
						const U08 aColor);


#endif