#include "yags2d.h"

/*
//=================================================================

	Draws an ellipse of the specified X and Y axis radii and color,
	using a fast integer-only & square-root-free approach, and
	generating the arc for one octant into a buffer, drawing four
	symmetries from that buffer, then doing the same for the other
	axis.
//=================================================================
*/



/* 
	Draws the arc for an octant in which Y is the major axis. (X,Y) is the
   starting point of the arc. HorizontalMoveDirection selects whether the
   arc advances to the left or right horizontally (0=left, 1=right).
   RowOffset contains the offset in bytes from one scan line to the next,
   controlling whether the arc is drawn up or down. Length is the
   vertical length in pixels of the arc, and DrawList is a list
   containing 0 for each point if the next point is vertically aligned,
   and 1 if the next point is 1 pixel diagonally to the left or right 
*/

static void 
DrawVOctant(pixel_buffer screenInfo, const U16 X, U16 Y, 
	U08 *DrawList, int listLength,
	int RowOffset, int HDirection, 
	U08 aColor)
{
	U08 * ScreenPtr = &((U08*)screenInfo.pixels)[Y*screenInfo.bytes_per_row+X];

	U16 Length = listLength;
	
	/* Draw all points in DrawList */
	while ( Length-- ) 
	{
		// Set the current pixel to the desired color
		*ScreenPtr = aColor;
		
		// Now advance to the next pixel based on DrawList
		if ( *DrawList++ ) 
		{
			/* Advance horizontally to produce a diagonal move. Rotate
            the bit mask, advancing one byte horizontally if the bit
            mask wraps */
			ScreenPtr += HDirection;
		}

		ScreenPtr += RowOffset; // advance to the next scan line
   }
}

/* 
	Draws the arc for an octant in which X is the major axis. (X,Y) is the
   starting point of the arc. HorizontalMoveDirection selects whether the
   arc advances to the left or right horizontally (0=left, 1=right).
   RowOffset contains the offset in bytes from one scan line to the next,
   controlling whether the arc is drawn up or down. Length is the
   horizontal length in pixels of the arc, and DrawList is a list
   containing 0 for each point if the next point is horizontally aligned,
   and 1 if the next point is 1 pixel above or below diagonally 
*/

static void 
DrawHOctant(pixel_buffer screenInfo, const U16 X, const U16 Y, 
		const U08 *DrawList, const int listLength, const int RowOffset,
		const int HDirection, const U08 aColor)
{
	U08 * ScreenPtr = &((U08*)screenInfo.pixels)[Y*screenInfo.bytes_per_row+X];

	U16 Length = listLength;

	// Draw all points in DrawList
	while ( Length-- ) 
	{
		// Draw the pixel in the desired color
		*ScreenPtr = aColor;
      
		// Now advance to the next pixel based on DrawList */
		if ( *DrawList++ ) 
		{
			/* Advance vertically to produce a diagonal move */
			ScreenPtr += RowOffset; /* advance to the next scan line */
		}
		
		// Advance horizontally.
		ScreenPtr += HDirection;
	}
}

/* Draws an ellipse of X axis radius A and Y axis radius B in
 * color Color centered at screen coordinate (X,Y). Radii must
 * both be non-zero */
void 
Y2DEllipse(pixel_buffer screenInfo, const U16 X, const U16 Y, 
	const U16 A, const U16 B, 
	const U08 aColor) 
{
	short WorkingX, WorkingY;
	long Threshold;
	long ASquared = (long) A * A;
	long BSquared = (long) B * B;
	long XAdjust, YAdjust;
	U08 *PixListPtr;
	static U08 PixList[1600/2];
                                    /* maximum major axis length is
                                       1/2 screen width, because we're
                                       assuming no clipping is needed */


   /* Draw the four symmetric arcs for which X advances faster (that is,
      for which X is the major axis) */

   /* Draw the four arcs; set draw parameters for initial point (0,B) */
   /* Calculate all points along an arc of 1/8th of the ellipse and
      store that info in PixList for later drawing */
	PixListPtr = PixList;
	WorkingX = 0;
	XAdjust = 0;
	YAdjust = ASquared * 2 * B;
	Threshold = ASquared / 4 - ASquared * B;

	for (;;) 
	{
      /* Advance the threshold to the value for the next X point
         to be drawn */
		Threshold += XAdjust + BSquared;

      /* If the threshold has passed 0, then the Y coordinate has
         advanced more than halfway to the next pixel and it's time
         to advance the Y coordinate by 1 and set the next threshold
         accordingly */
		if ( Threshold >= 0 ) 
		{
			YAdjust -= ASquared * 2;
			Threshold -= YAdjust;
			*PixListPtr++ = 1;   /* advance along both axes */
		} else 
		{
			*PixListPtr++ = 0;   /* advance only along the X axis */
		}

		// Advance the X coordinate by 1
		XAdjust += BSquared * 2;
		WorkingX++;

		/* Stop if X is no longer the major axis (the arc has passed the
         45-degree point) */
		if ( XAdjust >= YAdjust )
			break;
	}

	/* Now draw each of 4 symmetries of the octant in turn, the
      octants for which X is the major axis. Adjust every other arc
      so that there's no overlap */
	DrawHOctant(screenInfo, X,Y-B,PixList,WorkingX,screenInfo.bytes_per_row,-1,aColor);
	DrawHOctant(screenInfo, X+1,Y-B+(*PixList),PixList+1,WorkingX-1,screenInfo.bytes_per_row,1,
aColor);
	DrawHOctant(screenInfo, X,Y+B,PixList,WorkingX,-screenInfo.bytes_per_row,-1,aColor);
	DrawHOctant(screenInfo, X+1,Y+B-(*PixList),PixList+1,WorkingX-1,-screenInfo.bytes_per_row,1,aColor);

   /* Draw the four symmetric arcs for which X advances faster (that is,
      for which Y is the major axis) */

   /* Draw the four arcs; set draw parameters for initial point (A,0) */
   /* Calculate all points along an arc of 1/8th of the ellipse and
      store that info in PixList for later drawing */
	PixListPtr = PixList;
	WorkingY = 0;
	XAdjust = BSquared * 2 * A;
	YAdjust = 0;
	Threshold = BSquared / 4 - BSquared * A;

	for (;;) 
	{
      /* Advance the threshold to the value for the next Y point
         to be drawn */
		Threshold += YAdjust + ASquared;

      /* If the threshold has passed 0, then the X coordinate has
         advanced more than halfway to the next pixel and it's time
         to advance the X coordinate by 1 and set the next threhold
         accordingly */
		if ( Threshold >= 0 ) 
		{
			XAdjust -= BSquared * 2;
			Threshold = Threshold - XAdjust;
			*PixListPtr++ = 1;   /* advance along both axes */
		} else 
		{
			*PixListPtr++ = 0;   /* advance only along the X axis */
		}

		/* Advance the Y coordinate by 1 */
		YAdjust += ASquared * 2;
		WorkingY++;

		/* Stop if Y is no longer the major axis (the arc has passed the
         45-degree point) */
		if ( YAdjust > XAdjust )
			break;
	}

	/* Now draw each of 4 symmetries of the octant in turn, the
      octants for which Y is the major axis. Adjust every other arc
      so that there's no overlap */
	DrawVOctant(screenInfo, X-A,Y,PixList, WorkingY,-screenInfo.bytes_per_row,1, aColor);
	DrawVOctant(screenInfo, X-A+(*PixList),Y+1,PixList+1,WorkingY-1,screenInfo.bytes_per_row,1,aColor);
	DrawVOctant(screenInfo, X+A,Y,PixList, WorkingY,-screenInfo.bytes_per_row,-1,aColor);
	DrawVOctant(screenInfo, X+A-(*PixList),Y+1,PixList+1, WorkingY-1,screenInfo.bytes_per_row,-1,aColor);

}



/* 
	Fills an ellipse of X axis radius A and Y axis radius B in
	color Color centered at screen coordinate (X,Y). Radii must
	both be non-zero 
*/
void 
Y2DEllipseFill(pixel_buffer screenInfo, const U16 X, const U16 Y, 
	const U16 A, const U16 B, const U08 aColor) 
{
	short WorkingX, WorkingY;
	long Threshold;
	long ASquared = (long) A * A;
	long BSquared = (long) B * B;
	long XAdjust, YAdjust;


   /* Draw the four symmetric arcs for which X advances faster (that is,
      for which X is the major axis) */

   /* Draw the four arcs; set draw parameters for initial point (0,B) */
   /* Calculate all points along an arc of 1/8th of the ellipse and
      draw the line segment */
	WorkingX = 0;
	XAdjust = 0;
	YAdjust = ASquared * 2 * B;
	Threshold = ASquared / 4 - ASquared * B;

	short YOffset = -B;
	
	for (;;) 
	{
      /* Advance the threshold to the value for the next X point
         to be drawn */
		Threshold += XAdjust + BSquared;

		/* 
			If the threshold has passed 0, then the Y coordinate has
			advanced more than halfway to the next pixel and it's time
			to advance the Y coordinate by 1 and set the next threshold
			accordingly and draw the current line.
		*/
		if ( Threshold >= 0 ) 
		{
			// Draw the current lines before moving on
			Y2DHLine(screenInfo, X-WorkingX,Y+YOffset, X+WorkingX,Y+YOffset,aColor);
			Y2DHLine(screenInfo, X-WorkingX,Y-YOffset, X+WorkingX,Y-YOffset,aColor);
			YOffset ++;
			YAdjust -= ASquared * 2;
			Threshold -= YAdjust;
		}
		// Advance the X coordinate by 1
		XAdjust += BSquared * 2;
		WorkingX++;

		/* Stop if X is no longer the major axis (the arc has passed the
         45-degree point) */
		if ( XAdjust >= YAdjust )
			break;
	}

	
   /* Draw the four symmetric arcs for which X advances faster (that is,
      for which Y is the major axis) */

   /* Draw the four arcs; set draw parameters for initial point (A,0) */
   /* 
		Calculate all points along an arc of 1/8th of the ellipse
		Draw as we calculate.
	*/
	XAdjust = BSquared * 2 * A;
	YAdjust = 0;
	Threshold = BSquared / 4 - BSquared * A;
	short XOffset = 0;

	YOffset = 0;

	for (;;) 
	{
      /* Advance the threshold to the value for the next Y point
         to be drawn */
		Threshold += YAdjust + ASquared;

      /* If the threshold has passed 0, then the X coordinate has
         advanced more than halfway to the next pixel and it's time
         to advance the X coordinate by 1 and set the next threhold
         accordingly */
		if ( Threshold >= 0 ) 
		{
			XAdjust -= BSquared * 2;
			Threshold = Threshold - XAdjust;
			// advance along both axes
			XOffset++;			
		}

		// Draw the current lines
		// upper half
		Y2DHLine(screenInfo, X-A+XOffset, Y-YOffset, X+A-XOffset, Y-YOffset,aColor);
		// Lower half
		Y2DHLine(screenInfo, X-A+XOffset, Y+YOffset, X+A-XOffset, Y+YOffset,aColor);
		
		/* Advance the Y coordinate by 1 */
		YAdjust += ASquared * 2;
		YOffset++;
		
		/* Stop if Y is no longer the major axis (the arc has passed the
         45-degree point) */
		if ( YAdjust > XAdjust )
			break;
	}
}



//==================================================================
// Function: Y2DCircle
//
// Draws a circle using an incremental Bresenham algorithm.
// This method is here just to look at.  It is actually slower
// than the general ellipse drawing code.  The yags2d.h header
// contains an inline function that maps directly to the ellipse
// drawing code.
//==================================================================
/*
void
Y2DCircle(pixel_buffer screenInfo,
	const unsigned short x, const unsigned short y,
	const unsigned short radius, const unsigned char aColor)
{
	int MajorAxis, MinorAxis;
	unsigned long RadiusSqMinusMajorAxisSq;
	unsigned long MinorAxisSquaredThreshold;
	
	MajorAxis = 0;
	MinorAxis = radius;
	
	RadiusSqMinusMajorAxisSq = (unsigned long)radius*radius;
	MinorAxisSquaredThreshold = (unsigned long)MinorAxis*MinorAxis - MinorAxis;
	
	// Draw the point along a 1/8 arc of the circle
	// Draw all eight symmetrical parts at once.
	do {
		Y2DSetPixel(screenInfo, x+MajorAxis, y-MinorAxis, aColor);
		Y2DSetPixel(screenInfo, x-MajorAxis, y-MinorAxis, aColor);
		Y2DSetPixel(screenInfo, x+MajorAxis, y+MinorAxis, aColor);
		Y2DSetPixel(screenInfo, x-MajorAxis, y+MinorAxis, aColor);
		Y2DSetPixel(screenInfo, x+MinorAxis, y-MajorAxis, aColor);
		Y2DSetPixel(screenInfo, x-MinorAxis, y-MajorAxis, aColor);
		Y2DSetPixel(screenInfo, x+MinorAxis, y+MajorAxis, aColor);
		Y2DSetPixel(screenInfo, x-MinorAxis, y+MajorAxis, aColor);
		
		MajorAxis++;
		if ((RadiusSqMinusMajorAxisSq -= MajorAxis +MajorAxis-1) <=
			MinorAxisSquaredThreshold)
		{
			MinorAxis--;
			MinorAxisSquaredThreshold -= MinorAxis +MinorAxis;
		}
	} while (MajorAxis <= MinorAxis);
}
*/