//******************************************************************************
// 
//	File:		pulse.cpp
//
//	Description:	RAMPulse(tm) shows CPU and RAM usage on Be Box
//
//	Copyright 1994-1995, Be Incorporated. All Rights Reserved.
//
//******************************************************************************
 
#include <Debug.h>
#include <stdio.h>
#include <string.h>

#include <OS.h>
 
#ifndef PBAR_H
#include "progressbar.h"
#endif

#ifndef PULSE_H
#include "pulse.h"
#endif

#include "Pictures"


//
// Globals
//
const short		kWindowWidth = 263;
const short		kWindowHeight = 82;
const short		kBorderWidth = 5;		// Width of window border
const double	kInterval = 100000.0;	// interval between updates (usec)
short			gCPUCount;

char *gProcessorList[] = {
		"unknown",
		"601",
		"603",
		"603e",
		"604",
		"604e",
		"unknown",
		"unknown",
		"unknown",
		"unknown",
		"unknown",
		"unknown",
		"unknown",
		"686"
};

BPicture *BitmapToPicture(BBitmap *bitmap);


//====================================================
//
//====================================================


TPulseWindow::TPulseWindow(BRect r)
	 :BWindow(r, "CPU Monitor", B_TITLED_WINDOW, B_NOT_RESIZABLE | B_NOT_ZOOMABLE)
{
	r.OffsetTo(BPoint(0, 0));
	fPulseView = new TPulseView(r);
	Lock();
	AddChild(fPulseView);
	Unlock();
	Show();

	SetPulseRate(200000.0);
}

// --------------------------------------------------------------
bool TPulseWindow::QuitRequested()
{
	be_app->PostMessage(B_QUIT_REQUESTED);
	return TRUE;
}


//---------------------------------------------------------------

BPicture *BitmapToPicture(BBitmap *bitmap)
{
	BBitmap *wind = new BBitmap(bitmap->Bounds(), bitmap->ColorSpace(), TRUE);
	BView	*view = new BView(wind->Bounds(), "", B_FOLLOW_NONE, B_WILL_DRAW);

	wind->Lock();
	wind->AddChild(view);

	BPicture *picture = new BPicture();

	view->BeginPicture(picture);
	view->DrawBitmap(bitmap, B_ORIGIN);
	view->EndPicture();

	wind->Unlock();
	
	delete wind;

	return picture;
}

CPUButton *
CreateCPUButton(BTSCPUMonitor *aMonitor, BPoint thePoint)
{
	CPUButton	*theButton;
	BBitmap*	normalBits;
	BBitmap*	pushedBits;
	BPicture	*normalPict;
	BPicture	*pushedPict;

	//
	// Add the button for the first processor
	//
	BRect theRect(BPoint(0.,0.), BPoint(17.,17.));
	
	normalBits = new BBitmap(theRect, B_COLOR_8_BIT);
	normalBits->SetBits(But1Up, normalBits->BitsLength(), 0, B_COLOR_8_BIT);
	pushedBits = new BBitmap(theRect, B_COLOR_8_BIT);
	pushedBits->SetBits(But1Down, pushedBits->BitsLength(), 0, B_COLOR_8_BIT);

	normalPict = BitmapToPicture(normalBits);
	pushedPict = BitmapToPicture(pushedBits);

	theRect.Set(thePoint.x,thePoint.y,
			thePoint.x+17., thePoint.y+17.);
	theButton = new CPUButton(aMonitor, theRect,"CPU","", normalPict, pushedPict);

	delete(normalBits);
	delete(pushedBits);
	delete(normalPict);
	delete(pushedPict);

	return theButton;
}

//====================================================
//
//====================================================

BTSCPUMonitor::BTSCPUMonitor(long cpuid, double aTime, BView *aView)
{
	BRect theRect;
	
	fCPUID = cpuid;
	fGraphPort = aView;
	fThen = aTime;
	fNext = 0;
	
	float yOffset = 27.*cpuid;
	theRect.Set(101,18+yOffset,101 + 146, 18 + 20+yOffset);
	fProgressBar = new TProgressBar(theRect, "CPU", 0);
	fProgressBar->SetPrev( aTime );
	fGraphPort->AddChild(fProgressBar);

	float xOffset = 79;
	yOffset = 20. + (27.*cpuid);
	fCPUButton = CreateCPUButton(this, BPoint(xOffset,yOffset));

	fGraphPort->AddChild(fCPUButton);
	if (_kget_cpu_state_(cpuid))		{ // Processor is active
		fCPUButton->SetValue(1);	// Push the button
		gCPUCount++;
	} else
		fCPUButton->SetValue(0);
	
}


void
BTSCPUMonitor::Render(bool aTruth)
{
	fProgressBar->Render(aTruth);
}

void
BTSCPUMonitor::Update(system_info *sysInfo, double aTime, bool aTruth)
{
	short CPUTime;
	
	//printf("BTSCPUMonitor::Update - %d\n", fCPUID);
	
	//
	// Check to see if a CPU has been turned off or on
	//
	if (!_kget_cpu_state_(fCPUID) && (fCPUButton->Value()))
			fCPUButton->SetValue(0);
	if (_kget_cpu_state_(fCPUID) && (!fCPUButton->Value()))
			fCPUButton->SetValue(1);


	//
	// Compute time spent in idle threads
	//
	CPUTime = 100 * (sysInfo->cpu_infos[fCPUID].active_time - fProgressBar->PrevTick()) / (aTime - fThen);
	fProgressBar->SetPrev( sysInfo->cpu_infos[fCPUID].active_time );
	fProgressBar->Set(max(0,CPUTime));
	
	fThen = aTime;
}

void
BTSCPUMonitor::ToggleProcessor()
{
	if (fCPUButton->Value())	{	
		_kset_cpu_state_(fCPUID, 1);
		gCPUCount++;
	} else	
	{ 
		// Disable progress bar
		fProgressBar->Set(0);
		//if (!_kget_cpu_state_(1-fCPUID))		// Other CPU disabled!
		//	Window()->SetTitle("That wasn't very smart");
		_kset_cpu_state_(fCPUID, 0);
		gCPUCount--;
	}
	
}

//---------------------------------------------------------------
// T P U L S E V I E W
//---------------------------------------------------------------

TPulseView::TPulseView(BRect r)
	 :BView(r, "PulseView", B_FOLLOW_NONE, B_WILL_DRAW | B_PULSE_NEEDED)
{
	BRect		theRect;

	gCPUCount = 0;
	
	theRect.Set(0,0,63,62);
	fAnim1 = new BBitmap(theRect, B_COLOR_8_BIT);
	fAnim1->SetBits(Anim1,11718,0,B_COLOR_8_BIT);

	//
	// Add CPU activity bars
	//
	get_system_info(&fSysInfo);

	
	// Create a linked list of monitors for each of
	// the CPUs
	fCPUMonitors = 0;
	BTSCPUMonitor *lastMonitor = 0;
	
	for (int ctr = 0; ctr < fSysInfo.cpu_count; ctr++)
	{
		BTSCPUMonitor *tmpMonitor = new BTSCPUMonitor(ctr, 
				fSysInfo.cpu_infos[ctr].active_time,this);
		
		if (!fCPUMonitors)
			fCPUMonitors = tmpMonitor;
		else
			lastMonitor->fNext = tmpMonitor;
			
		lastMonitor = tmpMonitor;
	}
}

// --------------------------------------------------------------
TPulseView::~TPulseView()
{
	delete fAnim1;
	// Delete cpu monitors
	//delete fRAM;
}
	


// --------------------------------------------------------------
// Draw and label the graph. 
// --------------------------------------------------------------
void TPulseView::Draw(BRect updateRect)
{
	BRect		frame;
	char		buf[500];
	char*		processor;
	long		width;
	int			cpu_speed;
	
	frame = Bounds();
	//
	// Black frame
	///
	SetHighColor(0,0,0);
	frame.right--;
	frame.bottom--;
	StrokeRect(frame);

	//
	// Bevelled edges
	//
	SetHighColor(255,255,255);
	StrokeLine(BPoint(1,1),BPoint(frame.right-1,1));
	StrokeLine(BPoint(1,1),BPoint(1,frame.bottom-1));
	SetHighColor(80,80,80);
	StrokeLine(BPoint(frame.right,1),BPoint(frame.right,frame.bottom));
	StrokeLine(BPoint(2,frame.bottom),BPoint(frame.right-1,frame.bottom));
	//
	// Dividing line
	//
	SetHighColor(96,96,96);
	StrokeLine(BPoint(1,82),BPoint(frame.right,82));
	SetHighColor(255,255,255);
	StrokeLine(BPoint(1,83),BPoint(frame.right,83));
	DrawBitmap(fAnim1,BPoint(10,10));		// Processor picture
	//DrawBitmap(fRAM,BPoint(14,86));			// Memory picture

	
	//
	// Draw the processor with its speed
	//
	SetDrawingMode(B_OP_OVER);
	SetHighColor(240,240,240);
	if (fSysInfo.cpu_type > 0 && fSysInfo.cpu_type < 14)
		processor = gProcessorList[fSysInfo.cpu_type];
	else
		processor = "unknown";
		
	width = StringWidth(processor);
	MovePenTo(10 + (32 - width / 2), 48);
	DrawString(processor);

	//
	// Draw processor speed
	//		
	cpu_speed = fSysInfo.cpu_clock_speed / 1000000;
	if (cpu_speed > 70)
		cpu_speed++;
	sprintf(buf, "%d MHz", cpu_speed);


	width = StringWidth(buf);
	MovePenTo(10 + (32 - width / 2), 60);
	DrawString(buf);

	BTSCPUMonitor *tmpMonitor = fCPUMonitors;
	while (tmpMonitor)
	{
		tmpMonitor->Render(TRUE);
		tmpMonitor = tmpMonitor->fNext;
	}

	
	inherited::Draw(updateRect);

}


// --------------------------------------------------------------
// This routine is called by the Interface Kit when a view is
// added to a window. We use it here to spawn our memory checker
// thread. This is a good place in general to put any kind of
// initialization code a view might need.
// --------------------------------------------------------------
void TPulseView::AttachedToWindow()
{
	BRect	 r;
	rgb_color	c = { 168, 168, 168, 0 };

	SetFontName("Emily");
	SetFontSize(9);
	fThen = system_time();
	SetViewColor(c);
	SetLowColor(c);
	Invalidate();
}

void TPulseView::Pulse()
{
	Update();
}

// --------------------------------------------------------------
// This routine updates the CPU thermometer
// If the "Doit" parameter is TRUE, we update the graph even
// if the activity level hasn't changed.
// --------------------------------------------------------------
void TPulseView::Update( bool DoIt )
{
	short		CPUTime1, CPUTime2;
	short		level;
	BRect		tempRect;
	short		free_blocks;
	double		now;
	
	if (Window()->Lock()) 
	{
		//
		// Get system info
		//
		get_system_info(&fSysInfo);
		now = system_time ();

		BTSCPUMonitor *tmpMonitor = fCPUMonitors;
		while (tmpMonitor)
		{
			tmpMonitor->Update(&fSysInfo, now, DoIt);
			tmpMonitor = tmpMonitor->fNext;
		}
				
		Sync();

		Window()->Unlock();
	}
}


//---------------------------------------------------------------
// C P U B U T T O N
//---------------------------------------------------------------

CPUButton::CPUButton(BTSCPUMonitor *aMonitor, BRect r, const char* name, const char* label,
				   BPicture* normal, BPicture* pushed)
		: BPictureButton(r, name, normal, pushed,
						new BMessage('BUTN'), B_TWO_STATE_BUTTON,
						B_FOLLOW_LEFT + B_FOLLOW_TOP, B_WILL_DRAW)
{
	fMonitor = aMonitor;
}

// --------------------------------------------------------------
// Note: We check this instead of using BControl messaging 'cause
// the BControls send the same message for button up or down...
// --------------------------------------------------------------
void	CPUButton::MouseDown(BPoint thePoint)
{	
	inherited::MouseDown(thePoint);
		
	fMonitor->ToggleProcessor();
	
}

// --------------------------------------------------------------

main()
{
	BApplication*	app;
	BRect			r;
	screen_info 	sInfo;
	TPulseWindow	*theWindow;
	
	app = new BApplication('PULS');

	// Get the screen info so that we can put the
	// app down in the lower right corner.
	get_screen_info(&sInfo);
	r.Set((sInfo.frame.right - kWindowWidth) - kBorderWidth, 
		(sInfo.frame.bottom - kWindowHeight) - kBorderWidth,
		sInfo.frame.right - kBorderWidth, 
		sInfo.frame.bottom - kBorderWidth);
	theWindow = new TPulseWindow(r);

	app->Run();

	delete(app);
	
	return(0);
}
