//******************************************************************************
//
//	File:		main.cpp
//
//	Description:	Font Demo application.
//
//	Copyright 1993-97, Be Incorporated
//
//******************************************************************************

#ifndef _DEBUG_H
#include <Debug.h>
#endif

#ifndef _ALERT_H
#include <Alert.h>
#endif
#ifndef _BOX_H
#include <Box.h>
#endif

#ifndef MAIN_H
#include "main.h"
#endif

/*------------------------------------------------------------*/

TMyTEView::TMyTEView(const char *name, BRect view_bound, long view_flags,
		BRect dest_rect) :
	BTextView(view_bound, name, dest_rect, 0, view_flags | B_NAVIGABLE)
{
}

/*------------------------------------------------------------*/

void	TMyTEView::KeyDown(ulong aChar)
{
	switch (aChar) {
		case B_RETURN:
			ASSERT(font_window);
			BMessage *msg = new BMessage(NEW_TEXT);
			msg->AddString("text", Text());
			font_window->PostMessage(msg);
			break;
		case B_TAB:
			BView::KeyDown(aChar);
			break;
		default:
			BTextView::KeyDown(aChar);
			break;
	}
}

/*------------------------------------------------------------*/

bool	TMyTEView::AcceptsChar(ulong aChar) const
{
	return TRUE;
}

/*------------------------------------------------------------*/

TControlScroller::TControlScroller(BRect view_bound, BWindow *target,
		ulong command, BStringView *text, long min, long max)
	: BScrollBar(view_bound, "", NIL, min, max, B_HORIZONTAL)
{
	fTarget = target;
	fCommand = command;
	fStringView = text;
}

/*------------------------------------------------------------*/

void	TControlScroller::ValueChanged(long v)
{
	char 				buf[100];
	TControlWindow		*my_owner;

	my_owner = (TControlWindow *)Window();
	
	sprintf(buf, "%d", v);
	fStringView->SetText(buf);

	BMessage *msg = new BMessage(fCommand);
	msg->AddLong("value", v);
	msg->AddLong("sem", my_owner->control_sem);
	fTarget->PostMessage(msg);
	acquire_sem(my_owner->control_sem);	
}

/*------------------------------------------------------------*/

TControlWindow::TControlWindow(BWindow *main_window, BRect bound,
		window_type type, ulong flags, const char *name) :
	BWindow(bound, name, type, flags | B_NOT_RESIZABLE | B_NOT_ZOOMABLE)
{
	BRect		a_rect;
	BRect		dst_rect;
	BBox		*parent;
	BScrollBar	*sb;
	BStringView *str_view;
	
	Lock();
	
	fAutoCycle = FALSE;

#define set_rect(R, t, l, b, r) R.Set(l, t, r, b)

	/*
	 The 'Size' area of the control window
	*/
	control_sem = create_sem(1, "update_lock");
	acquire_sem(control_sem);	
	set_rect(a_rect, 5, 20, 69, 187);
	parent = new BBox(a_rect);
	parent->SetLabel("Size");
	AddChild(parent);
	parent->SetFontName("Emily");

	set_rect(a_rect, 30, 20, 42, 52);
	str_view = new BStringView(a_rect, "", "");
	parent->AddChild(str_view);

	set_rect(a_rect, 30, 53, 30 + B_H_SCROLL_BAR_HEIGHT, 140);
	sb = new TControlScroller(a_rect, main_window, NEW_SIZE, str_view, 4, 360);
	parent->AddChild(sb);
	sb->SetValue(50);
	
	/*
	 The 'Shear' area of the control window
	*/
	set_rect(a_rect, 79, 20, 141, 187);
	parent = new BBox(a_rect);
	parent->SetLabel("Shear");
	AddChild(parent);
	parent->SetFontName("Emily");

	set_rect(a_rect, 30, 20, 42, 52);
	str_view = new BStringView(a_rect, "", "");
	parent->AddChild(str_view);

	set_rect(a_rect, 30, 53, 30 + B_H_SCROLL_BAR_HEIGHT, 140);
	sb = new TControlScroller(a_rect, main_window, NEW_SHEAR, str_view, 45,135);
	parent->AddChild(sb);
	sb->SetValue(90);
	
	/*
	 The 'Rotation' area of the control window
	*/
	set_rect(a_rect, 79+(79-5), 20, 141+(141-69), 187);
	parent = new BBox(a_rect);
	parent->SetLabel("Rotation");
	AddChild(parent);
	parent->SetFontName("Emily");

	set_rect(a_rect, 30, 20, 42, 52);
	str_view = new BStringView(a_rect, "", "");
	parent->AddChild(str_view);

	set_rect(a_rect, 30, 53, 30 + B_H_SCROLL_BAR_HEIGHT, 140);
	sb = new TControlScroller(a_rect, main_window, NEW_ROTATION,
		str_view, 0, 360);
	parent->AddChild(sb);
	sb->SetValue(0);
	
	set_rect(a_rect, 79+((79-5)*2), 20, 79+((79-5)*2) + 15, 187);
	set_rect(dst_rect, 2, 2, a_rect.bottom - a_rect.top - 4,
		a_rect.right - a_rect.left - 4);
	TMyTEView *string_view = new TMyTEView("string", a_rect, B_WILL_DRAW,
		dst_rect);
	AddChild(new BScrollView("", string_view));	
	string_view->MakeFocus();	
	string_view->SetMaxChars(maxLength);
	string_view->SetWordWrap(FALSE);
	string_view->SetText(sampleText, strlen(sampleText));
	string_view->SelectAll();

	set_rect(a_rect, 255, 50, 278, 157);
	AddChild(new BButton(a_rect, BUTTON_NAME, "Cycle Fonts",
			     new BMessage(RUN_THROUGH)));
	
	set_rect(a_rect, 79+((79-5)*2)+(16*4), 20, 
			 79+((79-5)*2)+(16*4)+(FONT_NAME_HEIGHT*5) + 1, 187);
	AddChild(parent = new BBox(a_rect, "Rect", B_FOLLOW_NONE, B_WILL_DRAW));
	a_rect.InsetBy(1, 1);
	a_rect.right -= B_V_SCROLL_BAR_WIDTH;
	a_rect.OffsetTo(1, 1);
	parent->AddChild(font_list = new BListView(a_rect, "FontListView",
		B_FOLLOW_NONE, B_WILL_DRAW | B_NAVIGABLE));
	font_list->SetTarget(main_window);
	font_list->SetSelectionMessage(new BMessage(NEW_FONT));
	a_rect.left = a_rect.right + 1;
	a_rect.right += B_V_SCROLL_BAR_WIDTH;
	a_rect.top--;
	a_rect.bottom++;
	parent->AddChild(new BScrollBar(a_rect, "fontScroll", font_list,
				     0, 0, B_VERTICAL));
	if (fontNames) {
		for (int i = 0; i < numFonts; i++)
			font_list->AddItem(fontNames[i]);
	}
	else
		printf("fontNames not ready.\n");

	set_rect(a_rect, 79+((79-5)*2)+(16*5)+(FONT_NAME_HEIGHT*5) + 1, 20, 
			 79+((79-5)*2)+(16*5)+(FONT_NAME_HEIGHT*10) + 2, 187);
	AddChild(parent = new BBox(a_rect, "Rect", B_FOLLOW_NONE, B_WILL_DRAW));
	a_rect.InsetBy(1, 1);
	a_rect.right -= B_V_SCROLL_BAR_WIDTH;
	a_rect.OffsetTo(1, 1);
	parent->AddChild(set_list = new BListView(a_rect, "SetListView",
		B_FOLLOW_NONE, B_WILL_DRAW | B_NAVIGABLE));
	set_list->SetTarget(main_window);
	set_list->SetSelectionMessage(new BMessage(NEW_SET));
	a_rect.left = a_rect.right + 1;
	a_rect.right += B_V_SCROLL_BAR_WIDTH;
	a_rect.top--;
	a_rect.bottom++;
	parent->AddChild(new BScrollBar(a_rect, "setScroll", set_list,
				     0, 0, B_VERTICAL));
	if (fontSets) {
		for (int i = 0; i < numSets; i++)
			set_list->AddItem(fontSets[i]);
	}
	else
		printf("fontSets not ready.\n");
	
	Unlock();
}

/*------------------------------------------------------------*/

void	TControlWindow::DispatchMessage(BMessage* msg, BHandler *rec)
{
	if ((msg->what == B_PULSE) && fAutoCycle) {
		if ((fCyclePhase++ % 2) == 0) {
			long i = font_list->CurrentSelection();
			if (++i == font_list->CountItems())
				i = 0;
			font_list->Select(i);
		}
	}
	BWindow::DispatchMessage(msg, rec);
}

/*------------------------------------------------------------*/

void	TControlWindow::MessageReceived(BMessage* an_event)
{
	if (an_event->what == RUN_THROUGH) {
		BButton* b = (BButton*)FindView(BUTTON_NAME);
		ASSERT(b);

		if (fAutoCycle == FALSE) {
			fAutoCycle = TRUE;
			b->SetLabel("Stop Cycle");
		} else {
			fAutoCycle = FALSE;
			fCyclePhase = 0;
			b->SetLabel("Cycle Fonts");
		}
	} else {
		BWindow::MessageReceived(an_event);
	}
}

/*------------------------------------------------------------*/

TMainWindow::TMainWindow(BRect bound, window_type type, long flags,
		const char *name) :
	BWindow(bound, name, type, flags)
{
	BRect		a_rect;
	long		i;

	// Create offscreen size of full screen so we can resize window/view
	screen_info screen;
	get_screen_info(&screen);
	a_rect = screen.frame;
	a_rect.OffsetTo(B_ORIGIN);
	the_offscreen = new BBitmap(a_rect, B_MONOCHROME_1_BIT, TRUE);

	// Create font view and add it to offscreen
	fFontView = new TFontView(a_rect);

	the_offscreen->Lock();

	the_offscreen->AddChild(fFontView);

	// Avoid garbage flash at start-up by erasing offscreen to white
	fFontView->FillRect(fFontView->Bounds(), B_SOLID_LOW);
	fFontView->Sync();
	
	// Make font list
	numFonts = count_fonts();
	if (numFonts>3) {
		fontNames = (font_name*)malloc(sizeof(font_name) * numFonts);
		int k;
		for (i = 0, k = 0; i < numFonts; i++) {
			get_font_name(i, (font_name*)fontNames[k]);
			if (strcmp(fontNames[k], "Kate")!=0 && strcmp(fontNames[k], "Emily")!=0
				&& strcmp(fontNames[k], "Erich")!=0)
				k++;
		}
		numFonts=k;
	} else {
		fontNames = (font_name*)malloc(sizeof(font_name) * numFonts);
		for (i = 0; i < numFonts; i++)
			get_font_name(i, (font_name*)fontNames[i]);
	}

	// Make set list
	numSets = count_symbol_sets();
	fontSets = (symbol_set_name*)malloc(sizeof(symbol_set_name) * numSets);
	for (i = 0; i < numSets; i++)
		get_symbol_set_name(i, (symbol_set_name*)fontSets[i]);

	// Set initial font, set, and size
	fFontView->SetFontSize(50);
	the_offscreen->Unlock();

	// Add main window view
	set_rect(a_rect, 0, 0, mw_sizeV, mw_sizeH);
	fMainView = new TMainView(a_rect, "Main");
	Lock();
	AddChild(fMainView);
	Unlock();
}

/*------------------------------------------------------------*/

TMainWindow::~TMainWindow()
{
	delete the_offscreen;

	free(fontNames);
	free(fontSets);
}

/*------------------------------------------------------------*/

void	TMainWindow::MessageReceived(BMessage* msg)
{
	BRect	r;
	ulong 	what = msg->what;
	
	if (!the_offscreen->Lock()) {
		TRACE();
		return;
	}

	switch(what) {
		case 	NEW_SIZE:
			{
			long value = msg->FindLong("value");
			fFontView->SetFontSize(value);
			Redraw();
			release_sem(msg->FindLong("sem"));
			break;
			}
		case	NEW_SHEAR:
			{
			long value = msg->FindLong("value");
			fFontView->SetFontShear(value);
			Redraw();
			release_sem(msg->FindLong("sem"));
			break;
			}
		case	NEW_ROTATION:
			{
			long value = msg->FindLong("value");
			fFontView->SetFontRotation(value);
			Redraw();
			release_sem(msg->FindLong("sem"));
			break;
			}
		case	NEW_TEXT:
			{
			const char *text = msg->FindString("text");
			fFontView->SetText(text);
			Redraw();
			break;
			}
		case	NEW_FONT:
			if (font_list->CurrentSelection() >= 0) {
				fFontView->SetFontName(
				(const char *)font_list->ItemAt(font_list->CurrentSelection()));
				Redraw();
			}
			break;
		case	NEW_SET:
			if (set_list->CurrentSelection() >= 0) {
				fFontView->SetSymbolSet((const char *)set_list->ItemAt(
											set_list->CurrentSelection()));
				Redraw();
			}
			break;
		default		:
			BWindow::MessageReceived(msg);
			break;
	}

	the_offscreen->Unlock();
}


/*-------------------------------------------------------------*/

void	TMainWindow::FrameResized(float new_width, float new_height)
{
	if (the_offscreen->Lock()) {
		Redraw();
		the_offscreen->Unlock();
	}
}


/*------------------------------------------------------------*/

bool	TMainWindow::QuitRequested()
{
	be_app->PostMessage(new BMessage(B_QUIT_REQUESTED));
	return FALSE;
}

/*------------------------------------------------------------*/

void	TMainWindow::Redraw(bool forceCache)
{
	BRect r = fMainView->Bounds();
	fFontView->Draw(r);
	fFontView->Sync();
	fMainView->Draw(r);
	fMainView->Flush();
}

/*------------------------------------------------------------*/

TMainView::TMainView(BRect bound, const char* name):
	BView(bound, "", B_FOLLOW_ALL, B_WILL_DRAW)
{
}

/*------------------------------------------------------------*/

void	TMainView::Draw(BRect update_rect)
{
	DrawBitmap(the_offscreen, update_rect, update_rect);
}

/*-------------------------------------------------------------*/

TFontView::TFontView(BRect view_bound):
	BView(view_bound, "", B_FOLLOW_ALL, B_WILL_DRAW)
{
	fText = (char *) malloc(strlen(sampleText)+1);
	strcpy(fText, sampleText);
}

/*------------------------------------------------------------*/

void	TFontView::SetText(const char *text)
{
	if (fText) {
		free(fText);
		fText = NULL;
	}

	if (text) {
		fText = (char *) malloc(strlen(text)+1);
		strcpy(fText, text);
	}
}

/*------------------------------------------------------------*/

void	TFontView::Draw(BRect update_rect)
{
	FillRect(Bounds(), B_SOLID_LOW);
	BRect r = font_window->Bounds();
	r.InsetBy(5, 5);
	font_info	font;
	GetFontInfo(&font);
	BPoint where;
	float	h=r.Height();
	float	w=r.Width();
	float	s=font.size*0.7;
	float	t=font.rotation*3.1459/180.0;
	where.x=w*(1-cos(t))/2+sin(t)*s/2;
	where.y=h*(1+sin(t))/2+cos(t)*s/2;
	MovePenTo(where);
	if (fText)
		DrawString(fText);
}

/*-------------------------------------------------------------*/

TFontApp::TFontApp()
	: BApplication('FDEM')
{
}

/*-------------------------------------------------------------*/

bool TFontApp::QuitRequested()
{
	// need to close windows in specific order
	ctrl_window->Quit();
	font_window->Quit();

	return TRUE;
}

/*-------------------------------------------------------------*/

void TFontApp::AboutRequested()
{
	BAlert				*myAlert;

	myAlert = new BAlert("", "Font Demo Application", "OK");
	myAlert->Go();
}

/*-------------------------------------------------------------*/

main(int argc, char* argv[])
{
	BApplication*		fontApp;

	fontApp = new TFontApp();

	font_window = new TMainWindow(mw_rect, B_TITLED_WINDOW, 0, "Display");

	ctrl_window = new TControlWindow(font_window, cw_rect, B_TITLED_WINDOW, 
					 B_NOT_CLOSABLE | B_NOT_RESIZABLE, 
					 "Controls");

	srand(system_time());

	font_list->Select(rand() % numFonts);
	set_list->Select(9);
	if (ctrl_window->Lock()) {
		font_list->Invalidate();
		ctrl_window->Unlock();
	}
	font_window->Show();
	ctrl_window->Show();

	fontApp->Run();

	delete fontApp;
	return(0);
}
