/* DynaDraw --- this is main.cpp */

#include <Application.h>
#include <Polygon.h>

/*-------------------------------------------------------------------------*/

class DDWindow : public BWindow
{
  public:
  	DDWindow(BRect R, const char* title, window_type type, uint32 flags);
  	bool QuitRequested();
};

/*-------------------------------------------------------------------------*/

DDWindow::DDWindow(BRect R, const char* title, window_type type, uint32 flags)
	: BWindow(R, title, type, flags)
{
	// do nothing
}

/*-------------------------------------------------------------------------*/

bool
DDWindow::QuitRequested()
{
	/* we're the last window open, so shut down the application */
 	be_app->PostMessage(B_QUIT_REQUESTED);
	return true;
}

/*-------------------------------------------------------------------------*/


class FilterView : public BView
{
 public:
 	/* overridden BView functions */
 	FilterView(BRect R);
 	void MouseDown(BPoint point);
	void Draw(BRect updateRect);
	 	
 private:
 	/* state variables, formerly from the filter structure */
 	float curmass, curdrag, width;
 	float velx, vely, vel;
 	float accx, accy, acc;
 	float angx, angy;
 	BPoint odel, m, cur, last;
 	
 	/* a list of polygons which make up our brushstokes */
	BList polyList;
	 	
 	/* this is where the calculations get done, and the drawing */
 	void DrawSegment();
 	bool Apply(BPoint m, float curmass, float curdrag);
 	
 	/* little helper functions */
 	float flerp(float f0, float f1, float p);
 	void setpos(BPoint point);
};

/*-------------------------------------------------------------------------*/

FilterView::FilterView(BRect R)
	: BView(R, "filter", B_FOLLOW_ALL_SIDES, B_WILL_DRAW)
{
	curmass = 0.50;
	curdrag = 0.46;
	width = 0.50;

}
/*-------------------------------------------------------------------------*/

void
FilterView::MouseDown(BPoint point)
{
	uint32 buttons=0;
  	float mx, my;
	BRect B(Bounds());
	   
  	GetMouse(&point, &buttons, true);

	if(buttons == B_PRIMARY_MOUSE_BUTTON)
	{
		mx = (float)point.x / B.right;
        my = (float)point.y / B.bottom;
        m.Set(mx,my);	
        setpos(m);
        odel.Set(0,0);
    
		/* loop through until the button is no longer held down */
		while(buttons == B_PRIMARY_MOUSE_BUTTON)
        {  
    		GetMouse(&point, &buttons, true);
	
	 	    mx = (float)point.x / B.right;
	    	my = (float)point.y / B.bottom;
	      	m.Set(mx,my);
	
	      	if(Apply(m, curmass, curdrag)) DrawSegment();

          	snooze(15000);
		}
	}
	else if (buttons == B_SECONDARY_MOUSE_BUTTON)
	{
		/* clears the screen by deleting the list of polygons */
		int32 count = polyList.CountItems();
		for(int i=0; i < count; i++)
		{
			delete(polyList.ItemAt(0));
			polyList.RemoveItem(0L);
		}
		Invalidate();
	}
}
/*-------------------------------------------------------------------------*/

bool 
FilterView::Apply(BPoint m, float curmass, float curdrag)
{
    float mass, drag;
    float fx, fy;

/* calculate mass and drag */
    mass = flerp(1.0, 100.0,curmass);
    drag = flerp(0.00,0.5,curdrag*curdrag);

/* calculate force and acceleration */
    fx = m.x-cur.x;
    fy = m.y-cur.y;
    acc = sqrt(fx*fx+fy*fy);
    if(acc<0.000001)
	  return false;
    accx = fx/mass;
    accy = fy/mass;

/* calculate new velocity */
    velx += accx;
    vely += accy;
    vel = sqrt(velx*velx+vely*vely);
    angx = -vely;
    angy = velx;
    if(vel<0.000001) 
      return false;

/* calculate angle of drawing tool */
    
    /*
    angx /= vel;
    angy /= vel;
    
    if(fixedangle) 
    {
	*/
	  angx = 0.6;
	  angy = 0.2;
    // }
	
	
/* apply drag */
    velx = velx*(1.0-drag);
    vely = vely*(1.0-drag);

/* update position */
	last = cur;
	cur.Set(cur.x+velx, cur.y+vely);
    
    return true;
}

/*-------------------------------------------------------------------------*/
void 
FilterView::DrawSegment()
{
 	BPoint del; 
    BPoint polypoints[4];
    float wid;
    float px, py, nx, ny;
	BRect B(Bounds());
	
    /* calculate the width of the moving pen */
    wid = 0.04-vel;
    wid = wid*width;
    if(wid<0.00001)
    	wid = 0.00001;
    del.x = angx*wid;
    del.y = angy*wid;

    px = last.x;
    py = last.y;
    nx = cur.x;
    ny = cur.y;

	SetHighColor(0,0,0);
	
	/* set up the points of the polygon */
    polypoints[0].Set((B.right*(px+odel.x)), (B.bottom*(py+odel.y)));
    polypoints[1].Set((B.right*(px-odel.x)), (B.bottom*(py-odel.y)));
    polypoints[2].Set((B.right*(nx-del.x)), (B.bottom*(ny-del.y)));
    polypoints[3].Set((B.right*(nx+del.x)), (B.bottom*(ny+del.y)));

	polyList.AddItem(new BPolygon(polypoints, 4));
	
	/* actually draw the polygon.  we fill AND stroke the polygon */
	/* because if we do not, "thin" polygons will not be drawn. */
	FillPolygon(polypoints, 4);
	StrokePolygon(polypoints, 4);
		
	odel = del;

}

/*-------------------------------------------------------------------------*/
void
FilterView::Draw(BRect updateRect)
{
	/* reconstruct the view after an invalidation by looping through */
	/* the polygon segments which make up our brushstrokes */

	BPolygon* bp;
	int32 count = polyList.CountItems();
	for(int i =0; i < count; i++)
	{
		bp = (BPolygon*)polyList.ItemAt(i);
		FillPolygon(bp);
		StrokePolygon(bp);
	}
}

/*-------------------------------------------------------------------------*/
float
FilterView::flerp(float f0, float f1, float p)
{

  return ((f0*(1.0-p))+(f1*p));
}

/*-------------------------------------------------------------------------*/
void 
FilterView::setpos(BPoint point)
{
  cur = point;
  last = point;
  velx = 0;
  vely = 0;
  accx = 0;
  accy = 0;
}

/*-------------------------------------------------------------------------*/
int main()
{
	BApplication App("application/x-vnd.Be-dynadraw");
	DDWindow* W = new DDWindow(BRect(50,50,800,600), "DynaDraw!", B_TITLED_WINDOW, 0);

	FilterView* F = new FilterView(W->Bounds());
	W->AddChild(F);
	
	W->Show();
	App.Run();
	
}