////////////////////////////////////////////////////////////////////////////////
// Signal.h
// --------
// Defines signal catching and receiving classes.
//
// Copyright 1999, Be Incorporated.   All Rights Reserved.
// This file may be used under the terms of the Be Sample Code License.

#ifndef SIGNALS_H_
#define SIGNALS_H_

#include <OS.h>

// Some pseudo-signal functions, useful to properly sync a fast data producer
// and a slow data consumer when the only interesting value is the last one.

// Usage : 
// -create a SignalSender with a pointer to the data that's going to be used.
// -create a SignalCatcher with a pointer to the SignalSender.
// -to send data, call LockData() on the SignalSender, store the data, and call SendSignal().
// -when some new data is available, UseData() is called in the SignalCatcher. the program must
//     make a local copy of the data and call DataCopied() as quickly as possible, then process
//     the data. Delaying the call to DataCopied will probably block the sender, which is
//     exactly what we're trying to avoid with this class
// -the SignalCatcher must be destroyed before the SignalSender. Since the thread is killed
//     violently (will a kill_thread), you might have to be clean up memory after the thread
//     gets killed. the CleanUp() function gets called after the thread gets killed.

class SignalSender {
public:
	SignalSender(void* dat);
	virtual ~SignalSender();
	void LockData();
	void SendSignal();
private:
	friend class SignalCatcher;

	sem_id data_lock,signal_blocker;
	volatile bool data_available,thread_running;
	void* data;
};

class SignalCatcher {
public:
	SignalCatcher(SignalSender*);
	virtual ~SignalCatcher();
	virtual void UseData()=0;
	virtual void CleanUp();
	void DataCopied();
	void* Data() { return sender->data; }
private:
	static int32 _TaskLoop(void* p) { return ((SignalCatcher*)p)->TaskLoop(); }
	int32 TaskLoop();
	SignalSender* sender;
	thread_id thread;
};

#endif // SIGNALS_H_
