////////////////////////////////////////////////////////////////////////////////
// Gamma.cpp
// ---------
// Implements the gamma correction class.
//
// Copyright 1999, Be Incorporated.   All Rights Reserved.
// This file may be used under the terms of the Be Sample Code License.

#include <math.h>
#include "Gamma.h"

// The gamma-correction class. You probably don't want to know all the details.
// m is the midgray, i.e. the gray that's half-way between black and white
// e is the offset that has to be added to the framebuffer value to get a 
//        quantity that's proportional to the CRT's input voltage
// a and b are the renormalization factors

GammaCorrect::GammaCorrect(float m) {
	SetM(m);
}

void GammaCorrect::SetM(float m) {
	e=EfromM(m);
	a=AfromE(e);
	b=BfromE(e);
}

float GammaCorrect::EfromM(float m) {
	if (m<.5) return -1e6;
	if (m>.7575) return (1-m)*255/61.75-1;

	float e=0,s=1;
	while (s>=.0001) {
		while (MfromE(e+s)>m) {
			e+=s;
		}
		s/=2;
	}
	return e;
}

float GammaCorrect::MfromE(float e) {
	if (e<0) return -1e6; // this is not true, but you can directly get E from M in that case
	return exp(.4*log((exp(2.5*log(1+e))+exp(2.5*log(e)))/2))-e;
}

float GammaCorrect::AfromE(float e) {
	if (e>=0) {
		return 1/(exp(2.5*log(1+e))-exp(2.5*log(e)));
	} else {
		return exp(-2.5*log(1+e));
	}
}

float GammaCorrect::BfromE(float e) {
	if (e>=0) {
		return -exp(2.5*log(e))/(exp(2.5*log(1+e))-exp(2.5*log(e)));
	} else {
		return 0;
	}
}

float GammaCorrect::Correct(float i) {
	return exp(.4*log((i-b)/a))-e;
}
