////////////////////////////////////////////////////////////////////////////////
// App.cpp
// -------
// Implements the Gamma application and main window classes.
//
// Copyright 1999, Be Incorporated.   All Rights Reserved.
// This file may be used under the terms of the Be Sample Code License.

#include <Application.h>
#include <Window.h>
#include <Control.h>
#include <Bitmap.h>
#include <StringView.h>
#include <View.h>
#include <Rect.h>
#include <OS.h>

#include <math.h>
#include <stdio.h>

#include "Signals.h"
#include "App.h"
#include "Gamma.h"

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// The application. Nothing fancy here. Note that Run() is not called from inside the constructor (bad things could happen,
// although they wouldn't in that case).

GApplication::GApplication():BApplication("application/x-vnd.Be-jbq-gamma-sample") {
}

void GApplication::ReadyToRun() {
	(new GWindow())->Show();
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// The window. Notice the B_ASYNCHRONOUS_CONTROLS flag. Also notice that some views are given names (so that they can be easily
// found with FindView().

GWindow::GWindow():
			BWindow(BRect(100,100,99+256,99+256+32),"Gamma demo",B_TITLED_WINDOW_LOOK,B_NORMAL_WINDOW_FEEL
						,B_NOT_RESIZABLE|B_NOT_ZOOMABLE|B_ASYNCHRONOUS_CONTROLS),
			SignalSender((void*)&render_val),
			SignalCatcher(this) {
	pre=new BBitmap(BRect(0,0,255,255),B_RGB32);
	AddChild(new GControl(BRect(120,264,120,264)));
	AddChild(new GView(BRect(0,0,0,0)));
	AddChild(new BStringView(BRect(8,256,111,271),"midpoint",""));
	AddChild(new BStringView(BRect(8,272,111,287),"quality",""));
}

// Dispatch the message (Not hard to guess!...)

void GWindow::DispatchMessage(BMessage*m,BHandler*h) {
	if ((h==this)&&(m->what=='gchg')) {
		UseNewValue(m->FindInt32("be:value"));
	} else {
		BWindow::DispatchMessage(m,h);
	}
}
	
// This is the function that's called by the Looper thread when a new message arrives.
// As explained in the article, changing #if 0 to #if 1 will make it process the message synchronously in the looper thread, whereas the
// default version processes it asynchronously in anotherr thread.

void GWindow::UseNewValue(int val) {
#if 0
	DisplayNewValue(val); // This is a 'bad' implementation.
#else
	LockData();
	render_val=val;
	SendSignal();
#endif
}

// Overridden from SignalCatcher. Keeps a local copy of the new data, unlocks the new data by calling DataCopied(), and finally
// processes it

void GWindow::UseData() {
	int val=*(volatile int*)Data();
	DataCopied();
	DisplayNewValue(val);
}

// This function does part of the actual processing. The bitmap drawing part is in another function.

void GWindow::DisplayNewValue(int val) {
	MakeBitmap(val);
	Lock();
	FindView("preview")->DrawBitmap(pre,BPoint(0,0));
	char buff[30];
	sprintf(buff,"mid-light @ %d",val);
	((BStringView*)FindView("midpoint"))->SetText(buff);
	int qual;
	if (val>193) {
		qual=int(100*(1+gc.e)+.5);
	} else {
		float mx=exp(2.5*log(1+gc.e));
		float mn=exp(2.5*log(gc.e));
		qual=int(100*(mx-mn)/mx+.5);
	}
	sprintf(buff,"quality : %d%%",qual);
	((BStringView*)FindView("quality"))->SetText(buff);
	Sync(); // so that the changes are displayed immediately
	Unlock();
}

// The bitmap is drawn here.

void GWindow::MakeBitmap(int val) {
	gc.SetM(val/255.);
	uint32* bits=(uint32*)pre->Bits();
	for (int j=0;j<256;j++) {
		for (int i=0;i<256;i++) {
			if (j<128) {
				if (i<128) {
					*(bits++)=0xffffffff*(j&1);
				} else {
					*(bits++)=val*0x01010101;
				}
			} else {
				if (i<128) {
					*(bits++)=int(255*gc.Correct(Ripple(i,j)))*0x01010101;
				} else {
					*(bits++)=0;
				}
			}
		}
		bits+=pre->BytesPerRow()/4-256;
	}
	bits-=128*pre->BytesPerRow()/4+128;
	for (int x=0;x<128;x++) {
		for (int i=1;i<6;i++) {
			float y=i*(5+.05*x);
			int iy=int(floor(y));
			float fy=y-iy;
			bits[x+iy*256+64*256]=0xffffffff;
			bits[x+iy*256]=int(255*gc.Correct(1-fy))*0x01010101;
			bits[x+iy*256+256]=int(255*gc.Correct(fy))*0x01010101;
		}
	}
}

float GWindow::Ripple(int i,int j) {
	return .5+.5*sin(2*M_PI*(((i-63)*(i-64)+(j+63)*(j+64))/1024.));
}

bool GWindow::QuitRequested() {
	be_app->PostMessage(B_QUIT_REQUESTED);
	return true;
}

GWindow::~GWindow() {
	delete pre;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Workaround for a limitation where SetViewBitmap cannot be called while the view is not attached to a window.

GView::GView(BRect r):BView(r,"preview",0,0) {
	ResizeTo(255,255);
}

void GView::AttachedToWindow() {
	SetViewBitmap(((GWindow*)Window())->pre);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// A custom control. It might have been done with a BSlider, but doing it this way shows more details about the actual
// implementation of asynchronous controls.

GControl::GControl(BRect r):BControl(r,"gamma control","mid-gray",new BMessage('gchg'),B_FOLLOW_NONE,B_WILL_DRAW) {
	ResizeTo(127,15);

	bmp=new BBitmap(BRect(0,0,127,15),B_RGB32);
	uint32* bits=(uint32*)bmp->Bits();
	for (int j=0;j<16;j++) {
		for (int i=128;i<256;i++) {
			*(bits++)=i*0x01010101;
		}
		bits+=bmp->BytesPerRow()/4-128;
	}
}

void GControl::AttachedToWindow() {
	SetTarget(Window());
	SetValue(193);
	MakeFocus();
}

// This is the heart of an asnychronous control. Set the event mask and the tracking flag in MouseDown, and everything works
// just like magic.

void GControl::MouseDown(BPoint p) {
	SetMouseEventMask(B_POINTER_EVENTS,B_LOCK_WINDOW_FOCUS|B_SUSPEND_VIEW_FOCUS);
	SetTracking(true);
	DoMouse(p.x);
}

void GControl::MouseMoved(BPoint p,uint32,const BMessage*) {
	if (IsTracking()) {
		DoMouse(p.x);
	}
}

void GControl::MouseUp(BPoint) {
	SetTracking(false);
}

// The function that actually processes the new mouse coordinates. 

void GControl::DoMouse(int x) {
	x+=128;
	if (x<128) {
		x=128;
	}
	if (x>254) { // midpoint=255 is forbidden
		x=254;
	}
	if (x==Value()) {
		return;
	}
	SetValue(x);
}

// everything else in that glue should be fairly trivial.

void GControl::KeyDown(const char*c,int32) {
	switch(*c) {
		case B_RIGHT_ARROW : {
			if (Value()<254) {
				SetValue(Value()+1);
			}
			break;
		}
		case B_LEFT_ARROW : {
			if (Value()>128) {
				SetValue(Value()-1);
			}
			break;
		}
		default : ;
	}
}

void GControl::SetValue(int32 v) {
	BControl::SetValue(v);
	Invoke();
}

void GControl::Draw(BRect) {
	DrawBitmap(bmp,BPoint(0,0));
	SetHighColor(255,0,0);
	StrokeLine(BPoint(Value()-128,0),BPoint(Value()-128,15));
}

////////////////////////////////////////////////////////////

int main() {
	(new GApplication)->Run();
	delete be_app;
}
