/*************************************************************************
/
/	Thumbnailer.cpp
/
/	Written by Robert Polic
/
/	Copyright 1999, Be Incorporated.   All Rights Reserved.
/	This file may be used under the terms of the Be Sample Code License.
/
*************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <Alert.h>
#include <Application.h>
#include <Autolock.h>
#include <Bitmap.h>
#include <Directory.h>
#include <Entry.h>
#include <fs_attr.h>
#include <MediaFile.h>
#include <MediaTrack.h>
#include <Node.h>
#include <NodeInfo.h>
#include <Path.h>
#include <Screen.h>
#include <StringView.h>
#include <TrackerAddOn.h>
#include <TranslationUtils.h>
#include <View.h>
#include <Window.h>

#include "Thumbnailer.h"


#define kDISPLAYED_ICON_SIZE	B_LARGE_ICON
#define kMARGIN					8
#define kLABEL_GAP				4
#define kTHREAD_NAME			"create_icon"
#define kWINDOW_WIDTH			(kDISPLAYED_ICON_SIZE * 4)
#define kWINDOW_HEIGHT			(kDISPLAYED_ICON_SIZE + kMARGIN + (kLABEL_GAP * 2))


//========================================================================

int main()
{
	ThumbnailerApp	app;
	app.Run();
	return B_NO_ERROR;
}


//========================================================================

ThumbnailerApp::ThumbnailerApp()
	: BApplication	("application/x-vnd.Thumbnailer"),
	  fHaveRefs		(false),
	  fTerminate	(false)
{
	(fWindow = new ThumbnailWindow(&fTerminate))->Show();
	fThumbnailer = new Thumbnailer(&fTerminate, fWindow);
}

//------------------------------------------------------------------------

ThumbnailerApp::~ThumbnailerApp()
{
	delete fThumbnailer;
}

//------------------------------------------------------------------------

void ThumbnailerApp::ArgvReceived	(int32	argc,
									 char**	argv)
{
	thumbnail_function	function = eCreateThumbnail;
	int32 				index = 1;

	while (index < argc) {
		if (strcasecmp(argv[index], "-f") == 0)
			function = eForceThumbnail;
		else if (strcasecmp(argv[index], "-r") == 0)
			function = eRemoveThumbnail;
		else
		{
			// process any valid refs
			BEntry entry(argv[index]);
			if (entry.InitCheck() == B_NO_ERROR)
			{
				entry_ref	ref;

				entry.GetRef(&ref);
				fThumbnailer->ProcessRef(&ref, function);
				fHaveRefs = true;
			}
		}
		index++;
	}
}

//------------------------------------------------------------------------

void ThumbnailerApp::MessageReceived(BMessage* msg)
{
	switch (msg->what)
	{
		case B_SIMPLE_DATA:
			// handle "drag-n-drop" message
			RefsReceived(msg);
			break;
	}
}

//------------------------------------------------------------------------

void ThumbnailerApp::ReadyToRun()
{
	if (fHaveRefs) {
		// if we received any refs at start, wait for threads to complete
		// then terminate the app
		while (find_thread(kTHREAD_NAME) != B_NAME_NOT_FOUND)
			snooze(10000);
		PostMessage(B_QUIT_REQUESTED);
	}
	else
		// otherwise go to interactive mode
		SetStatusMessage("Drop file here!");
}

//------------------------------------------------------------------------

void ThumbnailerApp::RefsReceived(BMessage* msg)
{
	if (msg->HasRef("refs"))
	{
		int32				index = 0;
		entry_ref			ref;
		thumbnail_function	function = eCreateThumbnail;

		SetStatusMessage("working"B_UTF8_ELLIPSIS);
		fHaveRefs = true;

		uint32 mods = modifiers();
		// option key forces a thumbnail to be created (even if one exists)
		if (mods & B_OPTION_KEY)
			function = eForceThumbnail;
		// control key will remove any thumbnails
		if (mods & B_CONTROL_KEY)
			function = eRemoveThumbnail;

		while (msg->FindRef("refs", index++, &ref) == B_NO_ERROR)
			fThumbnailer->ProcessRef(&ref, function);
	}
}

//------------------------------------------------------------------------

void ThumbnailerApp::SetStatusMessage(const char* message)
{
	BAutolock	window(fWindow);

	if (window.IsLocked())
		fWindow->fThumbnailLabel->SetText(message);
}


//========================================================================

ThumbnailWindow::ThumbnailWindow(bool* terminate)
	: BWindow	(BRect(0, 0, 0, 0),
				 "Thumbnailer",
				 B_TITLED_WINDOW,
				 B_NOT_ZOOMABLE | B_NOT_MINIMIZABLE | B_NOT_RESIZABLE),
	  fTerminate(terminate),
	  fCount	(0)
{
	BFont		font(be_plain_font);
	BScreen		screen(B_MAIN_SCREEN_ID);
	font_height	height;

	// adjust window size to users font
	font.GetHeight(&height);
	float h = height.ascent + height.descent;
	ResizeTo(kWINDOW_WIDTH, kWINDOW_HEIGHT + h);
	// center the window
	MoveTo(screen.Frame().left + ((screen.Frame().Width() - kWINDOW_WIDTH) / 2),
		   screen.Frame().top + ((screen.Frame().Height() - (kWINDOW_HEIGHT + h)) / 2));

	// add views
	AddChild(fThumbnail = new ThumbnailView(
				BRect((kWINDOW_WIDTH - kDISPLAYED_ICON_SIZE) / 2,
					  kMARGIN,
					  (kWINDOW_WIDTH - kDISPLAYED_ICON_SIZE) / 2 + kDISPLAYED_ICON_SIZE - 1,
					  kMARGIN + kDISPLAYED_ICON_SIZE - 1),
				"thumbnail_bitmap"));
	AddChild(fThumbnailsRemaining = new BStringView(
				BRect(kLABEL_GAP,
					  kLABEL_GAP,
					  kLABEL_GAP + font.StringWidth("00000"),
					  kLABEL_GAP + h),
				"thumbnail_count",
				""));
	AddChild(fThumbnailLabel = new BStringView(
				BRect(kMARGIN,
					  kMARGIN + kDISPLAYED_ICON_SIZE + kLABEL_GAP,
					  kWINDOW_WIDTH - kMARGIN,
					  kMARGIN + kDISPLAYED_ICON_SIZE + kLABEL_GAP + h),
				"thumbnail_label",
				""));
	fThumbnailLabel->SetAlignment(B_ALIGN_CENTER);
}

//------------------------------------------------------------------------

void ThumbnailWindow::MessageReceived(BMessage* msg)
{
	switch (msg->what)
	{
		case B_SIMPLE_DATA:
			// "drag-n-drop" message, let the app object handle it
			be_app->PostMessage(msg);
			break;

		default:
			BWindow::MessageReceived(msg);
	}
}

//------------------------------------------------------------------------

bool ThumbnailWindow::QuitRequested()
{
	*fTerminate = true;
	be_app->PostMessage(B_QUIT_REQUESTED);
	return true;
}

//------------------------------------------------------------------------

void ThumbnailWindow::AdjustCount(int32 count)
{
	fCount += count;
	if (fCount)
	{
		char	label[16];

		sprintf(label, "%d", (int)fCount);
		fThumbnailsRemaining->SetText(label);
	}
	else
		fThumbnailsRemaining->SetText("");
}


//========================================================================

ThumbnailView::ThumbnailView	(BRect r,
								 const char* name)
	: BView			(r,
					 name,
					 B_FOLLOW_NONE,
					 B_WILL_DRAW),
	  fLargeIcon	(NULL),
	  fMiniIcon		(NULL)
{
	SetDrawingMode(B_OP_OVER);
}

//------------------------------------------------------------------------

ThumbnailView::~ThumbnailView()
{
	delete fLargeIcon;
	delete fMiniIcon;
}

//------------------------------------------------------------------------

void ThumbnailView::Draw(BRect /* where */)
{
	BBitmap*	icon;

	(kDISPLAYED_ICON_SIZE == B_LARGE_ICON) ?
		icon = fLargeIcon : icon = fMiniIcon;
	if (icon)
		DrawBitmapAsync(icon);
}

//------------------------------------------------------------------------

void ThumbnailView::MouseDown(BPoint where)
{
	if (fLargeIcon)
	{
		int32	clicks;

		if ((Window()->CurrentMessage()->FindInt32("clicks", &clicks) == B_NO_ERROR) &&
			(clicks == 2))
		{
			// bonus 1: dbl-clicking the icon launches it
			status_t result = be_roster->Launch(&fRef);
			if ((result != B_NO_ERROR) && (result != B_ALREADY_RUNNING))
				(new BAlert("", strerror(result), "Darn"))->Go();
		}
		else
		{
			// bonus 2: do "drag-n-drop" when icon is clicked on
			BMessage	large_icon;
			BMessage	mini_icon;
			BMessage	msg;

			// package up a message that at least the FileTypes preference panel understands
			fLargeIcon->Archive(&large_icon);
			msg.AddMessage("icon/large", &large_icon);
			if (fMiniIcon)
			{
				fMiniIcon->Archive(&mini_icon);
				msg.AddMessage("icon/mini", &mini_icon);
			}

			// construct a nice transparent image to drag
			BBitmap* drag_bits = new BBitmap(fLargeIcon->Bounds(), B_RGBA32, true);
			BView* offscreen_view = new BView(drag_bits->Bounds(), "", B_FOLLOW_NONE, 0);
			drag_bits->AddChild(offscreen_view);
			drag_bits->Lock();
			offscreen_view->SetHighColor(0, 0, 0, 0);
			offscreen_view->FillRect(offscreen_view->Bounds());
			offscreen_view->SetDrawingMode(B_OP_ALPHA);
			offscreen_view->SetHighColor(0, 0, 0, 128);
			offscreen_view->SetBlendingMode(B_CONSTANT_ALPHA, B_ALPHA_COMPOSITE);
			offscreen_view->DrawBitmap(fLargeIcon);
			drag_bits->Unlock();

			DragMessage(&msg, drag_bits, B_OP_ALPHA, where);
		}
	}
}

//------------------------------------------------------------------------

void ThumbnailView::SetIcons	(BBitmap*	large_icon,
								 BBitmap*	mini_icon,
								 entry_ref*	ref)
{
	delete fLargeIcon;
	delete fMiniIcon;
	fLargeIcon = large_icon;
	fMiniIcon = mini_icon;
	// erase the current icon and draw the new one
	SetHighColor(255, 255, 255, 255);
	FillRect(Bounds());
	Draw(Bounds());
	fRef = *ref;
}


//========================================================================

Thumbnailer::Thumbnailer	(bool* 				terminate,
							 ThumbnailWindow*	window)
	: fTerminate(terminate),
	  fWindow	(window)
{
	system_info	info;

	// only run as many threads as there are processors
	get_system_info(&info);
	fThumbnailerSem = create_sem(info.cpu_count, "thumbnail_sem");
}

//------------------------------------------------------------------------

Thumbnailer::~Thumbnailer()
{
	// wait for all thread to terminate
	while (find_thread(kTHREAD_NAME) != B_NAME_NOT_FOUND)
		snooze(1000);
	delete_sem(fThumbnailerSem);
}

//------------------------------------------------------------------------

void Thumbnailer::ProcessRef	(entry_ref*			ref,
								 thumbnail_function	function)
{
	BEntry	entry(ref, true);
	if (entry.InitCheck() == B_NO_ERROR)
	{
		if (entry.IsDirectory())
		{
			// recursively go through directories
			BDirectory	dir(&entry);
			BMessage	ref_message;
			entry_ref	r;

			while ((!*fTerminate) && (dir.GetNextRef(&r) == B_NO_ERROR))
				ProcessRef(&r, function);
		}
		else
		{
			// package the data needed by the thread
			thread_id		id;
			thumbnail_data* data = new thumbnail_data();

			data->ref = new entry_ref(*ref);
			data->thumbnailer = this;
			data->function = function;
			// try spawning the thread
			while ((id = spawn_thread((status_t (*)(void *))CreateIcon,
						 kTHREAD_NAME, B_DISPLAY_PRIORITY, data)) < 0)
				// reached the thread limit, wait for some to terminate and try again
				snooze(1000);

			if ((fWindow) && (fWindow->Lock()))
			{
				fWindow->AdjustCount(1);
				fWindow->Unlock();
			}

			// start the thread
			resume_thread(id);
		}
	}
}

//------------------------------------------------------------------------

BBitmap* Thumbnailer::ConvertBitmap	(BBitmap*		src,
									 BRect			dst,
									 color_space	color,
									 bool			preserve_aspect_ratio)
{
	// convert a bitmap from one size/color_space into a different
	// size color_space (possibly maintaining the aspect ratio) by
	// using the app_servers DrawBitmap

	// create a temporary bitmap and attach a view to it so we can
	// draw into it
	BBitmap*	tmp = new BBitmap(dst, color, true);
	BBitmap*	final = new BBitmap(dst, color);
	BView*		view = new BView(dst, "", B_FOLLOW_ALL, B_WILL_DRAW);

	tmp->AddChild(view);
	tmp->Lock();
	view->SetHighColor(B_TRANSPARENT_32_BIT);
	view->FillRect(dst);

	if (preserve_aspect_ratio)
	{
		float	f;
		BRect	bounds(src->Bounds());

		if (bounds.Width() > bounds.Height())
		{
			// leave width, adjust height to maintain aspect ratio
			f = bounds.Height() / bounds.Width() * dst.Height();
			dst.top = dst.top + ((dst.Height() - f) / 2);
			dst.bottom = dst.top + f;
		}
		else
		{
			// leave height, adjust width to maintain aspect ratio
			f = bounds.Width() / bounds.Height() * dst.Width();
			dst.left = dst.left + ((dst.Width() - f) / 2);
			dst.right = dst.left + f;
		}
	}

	view->DrawBitmap(src, src->Bounds(), dst);
	tmp->Unlock();
	// copy bits from temporary bitmap into final bitmap
	memcpy(final->Bits(), tmp->Bits(), tmp->BitsLength());
	// delete the temporary bitmap (deletes the attached view as well)
	delete tmp;
	return final;
}

//------------------------------------------------------------------------

status_t Thumbnailer::CreateIcon(thumbnail_data* data)
{
	BBitmap*	large_icon = NULL;
	BBitmap*	mini_icon = NULL;

	// wait for semaphore
	acquire_sem(data->thumbnailer->fThumbnailerSem);
	BNode node(data->ref);
	if ((!*data->thumbnailer->fTerminate) && (node.InitCheck() == B_NO_ERROR))
	{
		BNodeInfo	info(&node);
		attr_info	attribute;

		if (data->function == eRemoveThumbnail)
		{
			// this removes the thumbnail
			info.SetIcon(NULL, B_LARGE_ICON);
			info.SetIcon(NULL, B_MINI_ICON);
		}
		else if ((data->function == eForceThumbnail) ||
				 (node.GetAttrInfo("BEOS:L:STD_ICON", &attribute) == B_ENTRY_NOT_FOUND) ||
				 (node.GetAttrInfo("BEOS:M:STD_ICON", &attribute) == B_ENTRY_NOT_FOUND))
		{
			char		type[B_FILE_NAME_LENGTH];
			BBitmap*	image = NULL;

			info.GetType(type);
			if (strncasecmp(type, "image", 5) == 0)
				image = data->thumbnailer->GetImage(data->ref);
			else if (strncasecmp(type, "video", 5) == 0)
				image = data->thumbnailer->GetFrame(data->ref);

			if (image)
			{
				// this sets the thumbnail
				large_icon = data->thumbnailer->ConvertBitmap(
						image,
						BRect(0, 0, B_LARGE_ICON - 1, B_LARGE_ICON - 1));
				info.SetIcon(large_icon, B_LARGE_ICON);

				mini_icon = data->thumbnailer->ConvertBitmap(
						image,
						BRect(0, 0, B_MINI_ICON - 1, B_MINI_ICON - 1));
				info.SetIcon(mini_icon, B_MINI_ICON);

				delete image;
				if (!data->thumbnailer->fWindow)
				{
					delete large_icon;
					delete mini_icon;
				}
			}
		}

		if (data->thumbnailer->fWindow)
		{
			if (!large_icon)
			{
				// if we didn't create a thumbnail, display the current icon
				large_icon = new BBitmap(BRect(0, 0, B_LARGE_ICON - 1, B_LARGE_ICON - 1), B_CMAP8);
				info.GetTrackerIcon(large_icon, B_LARGE_ICON);
				mini_icon = new BBitmap(BRect(0, 0, B_MINI_ICON - 1, B_MINI_ICON - 1), B_CMAP8);
				info.GetTrackerIcon(mini_icon, B_MINI_ICON);
			}

			if (data->thumbnailer->fWindow->Lock())
			{
				// display icon in window
				data->thumbnailer->fWindow->fThumbnail->SetIcons(large_icon, mini_icon, data->ref);
				data->thumbnailer->fWindow->fThumbnailLabel->SetText(data->ref->name);
				data->thumbnailer->fWindow->Unlock();
			}
			else
			{
				delete large_icon;
				delete mini_icon;
			}
		}
	}

	if ((data->thumbnailer->fWindow) && (data->thumbnailer->fWindow->Lock()))
	{
		data->thumbnailer->fWindow->AdjustCount(-1);
		data->thumbnailer->fWindow->Unlock();
	}

	release_sem(data->thumbnailer->fThumbnailerSem);
	delete data->ref;
	delete data;
	return B_NO_ERROR;
}

//------------------------------------------------------------------------

bool Thumbnailer::FrameLooksGood(BBitmap* src)
{
	// raster-scan the image measuring the color frequency, if the frequency of any
	// color goes above 75%, reject the image
	bool	result = true;
	int32	frequency[100];
	int32	index;

	// init the frequency array
	for (index = 0; index < (int)(sizeof(frequency) / sizeof(int32)); frequency[index++] = 0);

	// convert image to a grayscale icon to make it easier to work with
	BBitmap* tmp = ConvertBitmap(src,
			BRect(0, 0, B_LARGE_ICON - 1, B_LARGE_ICON - 1),
			B_GRAY8,
			false);
	uchar* bits = (uchar*)tmp->Bits();
	for (index = 0; index < tmp->BitsLength(); index++)
	{
		// convert gray level to a percentage
		int32 n = (int32)((float)*bits++ / 256 * (float)(sizeof(frequency) / sizeof(int32)));
		frequency[n]++;
		if (frequency[n] > B_LARGE_ICON * B_LARGE_ICON * .75)
		{
			result = false;
			break;
		}
	}
	delete tmp;
	return result;
}

//------------------------------------------------------------------------

BBitmap* Thumbnailer::GetFrame(entry_ref* ref)
{
	BBitmap*	frame = NULL;
	BMediaFile	file(ref);

	// try opening the media file
	if (file.InitCheck() == B_NO_ERROR)
	{
		int32 tracks = file.CountTracks();
		for (int32 index = 0; index < tracks; index++)
		{
			// search for a media track
			BMediaTrack* track = file.TrackAt(index);
			if (track)
			{
				int64			count;
				media_format	track_format;

				track->EncodedFormat(&track_format);
				if ((track_format.type == B_MEDIA_RAW_VIDEO) ||
					(track_format.type == B_MEDIA_ENCODED_VIDEO))
				{
					bool		done = false;
					bigtime_t	offset = 0;
					bigtime_t	tmp_offset;

					// select format we're interested in
					track_format.type = B_MEDIA_RAW_VIDEO;
					track->DecodedFormat(&track_format);
					// construct a bitmap of the same size/color_space as the video source
					frame = new BBitmap(BRect(0, 0, track_format.Width() - 1, track_format.Height() - 1), track_format.ColorSpace());
					while (1)
					{
						// loop until we find a frame we like or run out of frames
						if (offset > track->Duration())
							done = true;
						tmp_offset = offset;
						track->SeekToTime(&tmp_offset, B_MEDIA_SEEK_CLOSEST_FORWARD);
						if (track->ReadFrames((char *)frame->Bits(), &count) == B_NO_ERROR)
						{
							if ((done) || (FrameLooksGood(frame)))
								break;
							else
								// frame didn't look good, skip ahead 1 second and try again
								offset += 1000000;
						}
						else if (offset)
						{
							// we reached the end and never found a frame we liked
							// reluctantly accept frame 0
							offset = 0;
							done = true;
						}
						else
						{
							// hmmm, couldn't read a single frame, clean up and bail
							delete frame;
							frame = NULL;
							break;
						}
					}
					break;
				}
			}
		}
	}
	return frame;
}

//------------------------------------------------------------------------

BBitmap* Thumbnailer::GetImage(entry_ref* ref)
{
	BBitmap*	image = NULL;

	BEntry	entry(ref);
	if (entry.InitCheck() == B_NO_ERROR)
	{
		// pretty simple, just use the translation kit to get image
		BPath	path(&entry);
		image = BTranslationUtils::GetBitmap(path.Path());
	}
	return image;
}


//========================================================================

// bonus 3: symlink this app into ~/config/add-ons/tracker and you get
// a tracker add-on
extern "C" void process_refs	(entry_ref	/* dir_ref */,
								 BMessage*	msg,
								 void*		/* reserved */)
{
	bool				terminate = false;
	int32				index = 0;
	uint32				mods = modifiers();
	entry_ref			ref;
	Thumbnailer*		thumbnailer = new Thumbnailer(&terminate);
	thumbnail_function	function = eCreateThumbnail;

	// option key forces a thumbnail to be created (even if one exists)
	if (mods & B_OPTION_KEY)
		function = eForceThumbnail;
	// control key will remove any thumbnails
	if (mods & B_CONTROL_KEY)
		function = eRemoveThumbnail;

	while (msg->FindRef("refs", index++, &ref) == B_NO_ERROR)
		thumbnailer->ProcessRef(&ref, function);

	delete thumbnailer;
}
