// TrackReader.cpp
// ---------------
// A BSoundFile-ish wrapper around BMediaTrack. Allows you
// to read data from sound files. Like BSoundFile, you can
// read an arbitrary number of frames and seek precisely to
// a specific frame.
//
// Copyright 1999, Be Incorporated.   All Rights Reserved.
// This file may be used under the terms of the Be Sample Code License.

#include <RealtimeAlloc.h>
#include <File.h>
#include <MediaFile.h>
#include <MediaTrack.h>
#include <assert.h>

#include "TrackReader.h"


static status_t
init_check(
	BMediaFile * file,
	ATrackReader ** outReader)
{
	*outReader = NULL;
	status_t err = file->InitCheck();
	if (err < B_OK) return err;
	media_format fmt;
	for (int ix=0; ix<file->CountTracks(); ix++) {
		fmt.type = B_MEDIA_RAW_AUDIO;
		fmt.u.raw_audio = media_raw_audio_format::wildcard;
		BMediaTrack * trk = file->TrackAt(ix);
		err = trk->DecodedFormat(&fmt);
		if ((err < B_OK) || (fmt.type != B_MEDIA_RAW_AUDIO)) {
			file->ReleaseTrack(trk);
			continue;
		}
		*outReader = new ATrackReader(trk, fmt, file);
		break;
	}
	if (*outReader != 0) return B_OK;
	return (err < B_OK) ? err : B_MEDIA_BAD_FORMAT;
}

static status_t
try_raw_file(
	const entry_ref * ref,
	ATrackReader ** reader)
{
	BFile * file = new BFile;
	status_t err = file->SetTo(ref, O_RDONLY);
	if (err < B_OK) {
		delete file;
		return err;
	}
	media_format fmt;
	fmt.type = B_MEDIA_RAW_AUDIO;
	fmt.u.raw_audio = media_raw_audio_format::wildcard;
	fmt.u.raw_audio.frame_rate = 44100.0;
	fmt.u.raw_audio.format = media_raw_audio_format::B_AUDIO_SHORT;
	fmt.u.raw_audio.channel_count = 2;
	fmt.u.raw_audio.byte_order = B_HOST_IS_LENDIAN ? B_MEDIA_LITTLE_ENDIAN : B_MEDIA_BIG_ENDIAN;
	fmt.u.raw_audio.buffer_size = 4096;
	*reader = new ATrackReader(file, fmt);
	return B_OK;
}






ATrackReader *
open_sound_file(
	const entry_ref * file,
	status_t * outError)
{
	//	create a BMediaFile so we can try to get data that way
	ATrackReader * trackReader = NULL;
	BMediaFile * soundFile = new BMediaFile(file);
	status_t err = init_check(soundFile, &trackReader);
	//	if that didn't work, we assume it's a raw, CD-audio file
	if (err < B_OK) {
		delete soundFile;
		soundFile = NULL;
		err = try_raw_file(file, &trackReader);
	}
	if (outError) {
		*outError = err;
	}
	//	that might still not work, because the file might not exist...
	if (err < B_OK) {
		return NULL;
	}
	return trackReader;
}



//	constructor for the BMediaFile case
ATrackReader::ATrackReader(BMediaTrack * track, const media_format & fmt, BMediaFile * mFile) :
	m_track(track), m_fmt(fmt), m_file(NULL)
{
	m_bufSize = m_fmt.u.raw_audio.buffer_size;
	if (m_bufSize < 1) {
		m_bufSize = B_PAGE_SIZE;
	}
	m_bufUnused = m_inBuffer = 0;
	m_buffer = 0;
	m_frameSize = (m_fmt.u.raw_audio.format & 0xf) * m_fmt.u.raw_audio.channel_count;
	if (m_frameSize == 0) m_frameSize = 4;
	m_mediaFile = mFile;
}

//	constructor for the raw CD-audio file case
ATrackReader::ATrackReader(BFile * file, const media_format & fmt) :
	m_track(NULL), m_fmt(fmt), m_file(file)
{
	m_bufSize = m_fmt.u.raw_audio.buffer_size;
	if (m_bufSize < 1) {
		m_bufSize = B_PAGE_SIZE;
	}
	m_bufUnused = m_inBuffer = 0;
	m_buffer = 0;
	m_frameSize = (m_fmt.u.raw_audio.format & 0xf) * m_fmt.u.raw_audio.channel_count;
	if (m_frameSize == 0) m_frameSize = 4;
	m_mediaFile = 0;
}

ATrackReader::~ATrackReader()
{
	rtm_free(m_buffer);
	delete m_file;
	delete m_mediaFile;
}

ssize_t
ATrackReader::ReadFrames(void * dest, int32 frameCount)
{
	ssize_t total = 0;
	ssize_t temp = 0;
	while ((temp = read_glob(dest, frameCount-total)) > 0) {
		total += temp;
		dest = ((char *)dest)+temp*FrameSize();
	}
	if (temp == B_LAST_BUFFER_ERROR) {
		temp = B_OK;
	}
	return (total > 0) ? total : (temp < 0) ? temp : 0;
}

ssize_t
ATrackReader::FrameSize()
{
	return m_frameSize;
}

ssize_t
ATrackReader::SampleSize()
{
	return m_fmt.u.raw_audio.format & 0xf;
}

BMediaTrack *
ATrackReader::Track()
{
	return m_track;
}

ssize_t
ATrackReader::CountFrames()
{
	int64 frameCount;
	if (m_track) {
		frameCount = m_track->CountFrames();
	}
	else {
		assert(m_file != 0);
		off_t pos = m_file->Position();
		frameCount = m_file->Seek(0, 2);
		m_file->Seek(pos, 0);
		frameCount /= m_frameSize;
	}
		//	clamp to something that will not wrap when calculating size
	if (frameCount > LONG_MAX/FrameSize()) return LONG_MAX/FrameSize();
	return frameCount;
}

ssize_t
ATrackReader::read_glob(void * dest, int32 frameCount)
{
	ssize_t ret = 0;
	if (m_file != 0) {
		ret = m_file->Read(dest, frameCount*m_frameSize);
		if (ret > 0) {
			ret /= m_frameSize;
		}
		return ret;
	}
	//	empty buffer, if any there
	if (m_bufUnused > 0) {
		ssize_t toCopy = frameCount;
		if (toCopy > m_bufUnused) toCopy = m_bufUnused;
		memcpy(dest, m_buffer+(m_inBuffer-m_bufUnused)*m_frameSize,
				toCopy*m_frameSize);
		m_bufUnused -= toCopy;
		ret += toCopy;
		frameCount -= toCopy;
		dest = ((char *)dest)+toCopy*m_frameSize;
	}
	if ((frameCount <= 0) || (ret < 0)) {
		return ret;
	}
	//	read as much as we can directly from file
	bool fileDone = false;
	while ((size_t)frameCount*FrameSize() >= m_bufSize) {
		int64 nRead = 0;
		status_t err = m_track->ReadFrames((char *)dest, &nRead);
		if (err < B_OK) {
			ret = (ret > 0) ? ret : err;
			fileDone = true;
			break;
		}
		if (nRead == 0) {
			fileDone = true;
		}
		dest = ((char *)dest) + nRead*m_frameSize;
		frameCount -= nRead;
		ret += nRead;
	}
	if ((frameCount <= 0) || (ret < 0) || (fileDone)) {
		return ret;
	}
	//	read new block into buffer
	if (!m_buffer) {
		m_buffer = (char *)rtm_alloc(NULL, m_bufSize);
		if (m_buffer == 0) {
			return (ret > 0) ? ret : B_NO_MEMORY;
		}
	}
	{
		//	... read ...
		int64 nRead = 0;
		status_t err = m_track->ReadFrames(m_buffer, &nRead);
		if (err < B_OK) {
			ret = (ret > 0) ? ret : err;
		}
		else {
			//	... copy ...
			m_inBuffer = nRead;
			m_bufUnused = m_inBuffer;
			ssize_t toCopy = frameCount;
			if (toCopy > m_bufUnused) {
				toCopy = m_bufUnused;
			}
			memcpy(dest, m_buffer, toCopy*m_frameSize);
			ret += toCopy;
			frameCount -= toCopy;
			m_bufUnused -= toCopy;
		}
	}
	return ret;
}

const media_format &
ATrackReader::Format() const
{
	return m_fmt;
}

status_t
ATrackReader::SeekToFrame(
	int64 * ioFrame)
{
	m_inBuffer = m_bufUnused = 0;
	if (m_file != 0) {
		off_t where = m_file->Seek(*ioFrame * m_frameSize, 0);
		if (where < 0) return where;
		*ioFrame = where / m_frameSize;
		return B_OK;
	}
	return m_track->SeekToFrame(ioFrame, B_MEDIA_SEEK_CLOSEST_BACKWARD);
}

BFile *
ATrackReader::File()
{
	return m_file;
}
