/*
 *	PC speaker driver
 * 	Accepts 8 bit unsigned 20k mono audio data
 */
#include <SupportDefs.h>
#include <KernelExport.h>
#include <Drivers.h>
#include <ISA.h>

#define OUTPUT_SAMPLE_PERIOD 60
#define MIN_INTERRUPT_INTERVAL 30
#define MAX_PLAY_BUFFER_LENGTH 0x4000

/*
 * If this macro is set to 0, The driver will play the sound directly,
 * shutting off interrupts.  This is provided for illustration only.
 * It is a bad thing to do because the OS will be totally frozen while
 * the sound plays.
 *
 * If this macro is set to 1, the driver will play the sound in the
 * background, using the timers to periodically move the speaker
 * cone.
 */
#define PLAY_IN_BACKGROUND 1

/*
 * If this is 1, when playing in the background, the processor will
 * spin in the interrupt handler to get better timing.  This results
 * in better accuracy, (hence quality) but degrades machine performance
 * significantly.  
 */
#define SPIN 0

#if SPIN
	/*
	 * If we spin, this defines the maximum interrupt latency in
	 * microseconds.
	 *
	 * We generally always want to interrupt early and spin
	 * until the proper time.  In general, the actual overhead
	 * for an interrupt is small, however, if other drivers have
	 * interrupts disabled or are servicing interrupts, the
	 * actual latency could increase.  The MIN_INTERRUPT_INTERVAL + 
	 * INTERRUPT_LATENCY must be less than the OUTPUT_SAMPLE_PERIOD,
	 * Otherwise your machine will hang.
	 */
	#define INTERRUPT_LATENCY 20
#else
	/*
	 * ...otherwise this defines the average interrupt latency.
	 * Since we don't spin, we rely on this being relatively 
	 * accurate most of the time.
	 */
	#define INTERRUPT_LATENCY 5
#endif

#define SPEAKER_ON() \
	{ isa->write_io_8(0x61, isa->read_io_8(0x61) & ~2); \
	is_speaker_on = true; }
	
#define SPEAKER_OFF() \
	{ isa->write_io_8(0x61, isa->read_io_8(0x61) | 2); \
	is_speaker_on = false; }

static status_t speaker_open(const char *name, uint32 flags, void **cookie);
static status_t speaker_close(void *cookie);
static status_t speaker_free(void *cookie);
static status_t speaker_control(void *cookie, uint32 op, void *buf, size_t len);
static status_t speaker_read(void *cookie, off_t pos, void *buf, size_t *len);
static status_t speaker_write(void *cookie, off_t pos, const void *buf, size_t *len);
static status_t play_chunk(char *buffer, long len);
status_t init_driver(void);
const char **publish_devices(void);
device_hooks* find_device(const char *name);
static int32 speaker_interrupt(timer *_t, uint32 flags);

typedef struct {
	struct timer	timer;
	int				error_term;
	int 			current_sample;
	bigtime_t 		input_sample_period;
	char			*buffer;
} speaker_timer_info;

const char *speaker_names[] = {
	"audio/pcspeaker",
	NULL
};

static device_hooks speaker_device = {
	speaker_open,
	speaker_close,
	speaker_free,
	speaker_control,
	speaker_read,
	speaker_write,
	NULL,			/* speaker_select */
	NULL,			/* speaker_deselect */
	NULL,			/* speaker_readv */
	NULL			/* speaker_writev */
};

static speaker_timer_info speaker_timer;
static int32 open_count = 0;
static isa_module_info *isa = NULL;
static sem_id done_sem = -1;
static bool is_speaker_on = false;
static int32 frame_rate;
static sem_id write_lock = -1;
static char play_buffer[MAX_PLAY_BUFFER_LENGTH];
static long play_buffer_length = 0;
static bigtime_t next_cone_movement;

static status_t
speaker_open(const char *name, uint32 flags, void **cookie)
{
	if (atomic_add(&open_count, 1) < 1) {
		*cookie = NULL;
		return B_OK;
	} else {
		atomic_add(&open_count, -1);
		return B_ERROR;
	}
}

static status_t speaker_close(void *cookie)
{
	atomic_add(&open_count, -1);
	return B_OK;
}

status_t init_driver(void)
{
	status_t ret = B_OK;

	ret = get_module(B_ISA_MODULE_NAME, (module_info**) &isa);
	if (ret != B_OK)
		goto error1;

	done_sem = create_sem(0, "finished playing");
	if (done_sem < B_OK) {
		ret = done_sem;
		goto error2;
	}

	write_lock = create_sem(1, "write lock");
	if (write_lock < B_OK) {
		ret = write_lock;
		goto error3;
	}

	is_speaker_on = false;
	frame_rate = 20000;
	return B_OK;

error3:
	delete_sem(done_sem);

error2:
	put_module(B_ISA_MODULE_NAME);

error1:
	return ret;
}

static status_t speaker_free(void *cookie)
{
	delete_sem(write_lock);
	delete_sem(done_sem);
	put_module(B_ISA_MODULE_NAME);
	return B_OK;
}

static status_t speaker_control(void *cookie, uint32 op, void *buf, size_t len)
{
	return B_ERROR;
}

static status_t speaker_read(void *cookie, off_t pos, void *buf, size_t *len)
{
	return B_ERROR;
}

static int32 speaker_interrupt(timer *_t, uint32 flags)
{
	speaker_timer_info *t = (speaker_timer_info*) _t;
#if SPIN
	bigtime_t now;
#endif
	uchar sample;
	
	while (1) {

#if SPIN
		/* Spin waiting for the next event time */
		while ((now = system_time()) < next_cone_movement) ;
#endif
	
		sample = t->buffer[t->current_sample];
		if (is_speaker_on) {
			SPEAKER_OFF();
			next_cone_movement += (255 - sample) * OUTPUT_SAMPLE_PERIOD / 255;
		} else {
			SPEAKER_ON();
			next_cone_movement += sample * OUTPUT_SAMPLE_PERIOD / 255;

			/* Determine if its time to get the next sample */	
			t->error_term += OUTPUT_SAMPLE_PERIOD;
			while (t->error_term > t->input_sample_period) {
				t->current_sample++;
				t->error_term -= t->input_sample_period;
			}

			/* Check to see if we're done playing this buffer */	
			if (t->current_sample >= play_buffer_length) {
				release_sem_etc(done_sem, 1, B_DO_NOT_RESCHEDULE);
				return B_INVOKE_SCHEDULER;
			}
		}
	
#if SPIN
		/*
		 * If the time until the next cone movement is longer than,
		 * our minimum interrupt interval, break out of the loop and
		 * set another timer.  If not, loop and spin until the next
		 * event time
		 */
		if (next_cone_movement - now > MIN_INTERRUPT_INTERVAL)
			break ;
#else
		break;
#endif

	}

	/*
	 * Program an interrupt for the next cone movement.  Note that we
	 * program to occur a little before the deadline.
	 */
	add_timer((timer *)&speaker_timer, speaker_interrupt, next_cone_movement -
		INTERRUPT_LATENCY, B_ONE_SHOT_ABSOLUTE_TIMER);

	return B_HANDLED_INTERRUPT;
}

#if PLAY_IN_BACKGROUND

static status_t play_chunk(char *buffer, long len)
{
	/*
	 * Play the sound in the background.  Set up a timer for
	 * the first interrupt, then wait on a completion semaphore.
	 */
	status_t ret;
	speaker_timer.error_term = 0;
	speaker_timer.current_sample = 0;
	speaker_timer.input_sample_period = 1000000 / frame_rate;
	speaker_timer.buffer = buffer;
	ret = add_timer((timer*) &speaker_timer, speaker_interrupt,
		next_cone_movement - INTERRUPT_LATENCY, B_ONE_SHOT_ABSOLUTE_TIMER);
	if (ret < 0)
		return ret;

	acquire_sem(done_sem);
	return B_OK;
}

#else

static status_t play_chunk(char *buffer, long len)
{
	cpu_status st;
	int error = 0;
	int current_sample = 0;
	bigtime_t frame_length = 1000000 / frame_rate;
	bigtime_t next_frame_time = system_time() + frame_length;
	uchar sample;

	st = disable_interrupts();
	while (true) {
		error = 0;
		sample = ((char*) buffer)[current_sample];
		if (sample < 128) {
			while (system_time() < next_frame_time) {
				error += sample;
				if (error > 255) {
					error -= 255;
					SPEAKER_ON();
				} else {
					SPEAKER_OFF();
				}
			}
		} else {
			while (system_time() < next_frame_time) {
				error += (255 - sample);
				if (error > 255) {
					error -= 255;
					SPEAKER_OFF();
				} else {
					SPEAKER_ON();
				}
			}
		}

		current_sample++;
		if (current_sample >= len)
			break;

		next_frame_time += frame_length;
	}	

	restore_interrupts(st);
	return B_OK;
}

#endif


static status_t speaker_write(void *cookie, off_t pos, const void *buf, size_t *len)
{
	int i;
	acquire_sem(write_lock);
	next_cone_movement = system_time() + OUTPUT_SAMPLE_PERIOD;
	for (i = 0; i < *len; i += MAX_PLAY_BUFFER_LENGTH) {
		play_buffer_length = i + MAX_PLAY_BUFFER_LENGTH < *len ? MAX_PLAY_BUFFER_LENGTH :
			*len - i;
			
		/*
		 * We must copy the chunk from the user buffer to our own buffer
		 * in kernel space.  If we are playing in the background,
		 * the interrupt may occur in the context of another team.
		 * If we are playing directly with interrupts off, reading from
		 * the user buffer could page fault, which is bad to do with
		 * interrupts off.
		 */
		memcpy(play_buffer, (char*) buf + i, play_buffer_length);
		if (play_chunk(play_buffer, play_buffer_length) < B_OK)
			goto done;
	}

done:

	release_sem(write_lock);
	return B_OK;
}



const char** publish_devices(void)
{
	return speaker_names;
}

device_hooks* find_device(const char *name)
{
	return &speaker_device;
}