/* wakeup.c, by Ficus Kirkpatrick (ficus@be.com)
 *
 * A driver for waking up a thread at sub-millisecond granularity.
 *
 * Copyright (C) 1998 by Be Incorporated.  All rights reserved.
 */

#include <KernelExport.h>
#include <Drivers.h>
#include <SupportDefs.h>

#define RTC_IRQ			0x08			/* IRQ for RTC interrupt */
#define RTC_ADDR		0x70			/* CMOS memory address */
#define RTC_DATA		0x71			/* CMOS memory data */
#define RTC_REGA		0x0a			/* RTC status register A */
#define RTC_REGB		0x0b			/* RTC status register B */
#define RTC_REGC		0x0c			/* RTC status register C */

static int32						wakeup_enter = 0;
static sem_id						wakeup_sem = -1;
static uchar						save_a;
static uchar						save_b;
static uchar						*hack_area;

/* read a value from an RTC register */
static uchar
rtc_read(uchar addr)
{
	uchar save;
	
	/* make sure we preserve the NMI enable bit */
	save = read_io_8(RTC_ADDR) & 0x80;
	addr &= 0x7f;
	write_io_8(RTC_ADDR, addr | save);

	return read_io_8(RTC_DATA);
}

/* write a value to an RTC register */
static void
rtc_write(uchar addr, uchar data)
{
	write_io_8(RTC_ADDR, addr);
	write_io_8(RTC_DATA, data);
}

/* ---------------------------------------------------------- */

/* The release_sem "springboard" code, with mnemonic listings.
 * An ellipsis means that the value will be patched in later.
 */
static uchar springboard[] = "\xb8\x00\x00\x00\x00"		/* mov   eax, ...  */
							 "\x50"						/* push  eax       */
							 "\xe8\x00\x00\x00\x00"		/* call  ...       */
							 "\x58"						/* pop   eax       */
							 "\xb8\x01\x00\x00\x00"		/* mov   eax, 1    */
							 "\xc3";					/* ret             */


__declspec(naked)	/* don't create a stack frame */
static bool
wakeup_handler(void *data)
{
	asm {
		/* acknowledge the timer interrupt */
		push		RTC_REGC
		call		rtc_read
		pop			eax
		
		/* jump to the release_sem springboard */
		mov			eax, hack_area
		jmp			eax
		
		/* we should never reach this point, as the
		 * springboard should send us back to our caller
		 */
		ret
	}
}

/* ---------------------------------------------------------- */

static status_t
wakeup_open(const char *name, uint32 flags, void **cookie)
{
	area_info ai;
	area_id aid;
	
	/* only one in at a time */
	if (atomic_add(&wakeup_enter, 1) >= 1)
		return B_ERROR;

	/* create the semaphore that will block the client thread */
	if ((wakeup_sem = create_sem(0, "wakeup")) < B_OK)
		goto bail;

	/* Create an area for the springboard.  This is an enormous hack --
	 * don't try this at home, kids!  See the related newsletter article
	 * for an explanation of how this works.
	 */		
	if ((aid = find_area("wakeup hack")) < 0) {
		/* if the area doesn't exist, create it */
		if (create_area("wakeup hack", &hack_area, B_ANY_KERNEL_ADDRESS,
					B_PAGE_SIZE, B_FULL_LOCK, B_READ_AREA|B_WRITE_AREA) < 0)
				goto bail2;
	} else {
		/* otherwise just reuse it -- we are only single-open */
		if (get_area_info(aid, &ai) < 0)
			goto bail2;
		hack_area = (uchar *)ai.address;
	}

	/* copy the springboard code into the area */
	memcpy(hack_area, springboard, sizeof(springboard));

	/* patch the springboard code to have the relocated address of
	 * release_sem, as well as the correct sem_id to pass to it
	 */
	*(sem_id *)(hack_area + 1) = wakeup_sem;
	*(uint32 *)(hack_area + 7) = (uint32)release_sem - (uint32)hack_area - 11;

	/* save the value of RTC status registers A & B */
	save_a = rtc_read(RTC_REGA);	
	save_b = rtc_read(RTC_REGB);
	
	/* install a handler for the timer interrupt */
	install_io_interrupt_handler(RTC_IRQ, wakeup_handler, NULL, 0);

	/* Set the interrupt rate to once every 244us.  Try changing the
	 * lower nibble of the value for different rates.
	 */
	rtc_write(RTC_REGA, 0x24);

	/* enable the periodic interrupt */
	rtc_write(RTC_REGB, (save_b | (1 << 6)));

	return B_OK;

bail2:
	delete_sem(wakeup_sem);
bail:
	atomic_add(&wakeup_enter, -1);
	return B_ERROR;	
}

static status_t
wakeup_close(void *cookie)
{
	return B_OK;
}

static status_t
wakeup_free(void *cookie)
{
	/* remove the interrupt handler */
	remove_io_interrupt_handler(RTC_IRQ, wakeup_handler);
	
	/* restore the value of RTC status registers A & B */
	rtc_write(RTC_REGA, save_a);
	rtc_write(RTC_REGB, save_b);

	/* it's ok for others to open the driver now */
	atomic_add(&wakeup_enter, -1);
	
	return B_OK;
}

static status_t
wakeup_control(void *cookie, uint32 msg, void *buf, size_t len)
{
	return B_ERROR;
}

static status_t
wakeup_read(void *cookie, off_t pos, void *buf, size_t *len)
{
	/* wakeup_sem is released by the interrupt handler */
	return acquire_sem(wakeup_sem);
}

static status_t
wakeup_write(void *cookie, off_t pos, const void *buf, size_t *len)
{
	return B_ERROR;
}

static status_t
init_driver(void)
{
	return B_OK;
}

static const char *wakeup_name[] = { "misc/wakeup", NULL };

const char **
publish_devices(void)
{
	return wakeup_name;
}

static device_hooks wakeup_device = {
	wakeup_open,
	wakeup_close,
	wakeup_free,
	wakeup_control,
	wakeup_read,
	wakeup_write
};

device_hooks *
find_device(const char *name)
{
	return &wakeup_device;
}

