// Copyright Be, Inc. 1996-1998
// Use at your own risk :-)

// videotest.c
// by Scott Bronson
// 28 Oct 1996

// Whacked around by Trey Boudreau
// 02 Apr 1998

#include <GraphicsCard.h>
#include <PCI.h>
#include <image.h>

#include <stdio.h>
#include <stdarg.h>


// Everything marked with @@ means improvements I want
// to make to the appserver.
// @@ we should do a read-modify-write to pci_config space


typedef enum {
	kNoTest,
	kSolidPaletteTest,
	kScreenSpaces
} GraphicsCardTests;

enum {
	// masks for B_X_BIT_XxX constants from InterfaceDefs.h
	B_8_BIT_SPACE_MASK 	= 0x8000801F,
	B_16_BIT_SPACE_MASK	= 0x000103E0,
	B_32_BIT_SPACE_MASK	= 0x00027C00
};


bool MapSpaceToRes( ulong space, int *xsiz, int *ysiz, int *depth );
bool MapResToSpace( int xsiz, int ysiz, int depth, ulong *space );


struct {
	int depth;
	int width;
	int height;
	ulong space;
} const static space_lut[] = {
	8, 640, 480, B_8_BIT_640x480,
	8, 800, 600, B_8_BIT_800x600,
	8, 1024, 768, B_8_BIT_1024x768,
	8, 1280, 1024, B_8_BIT_1280x1024,
	8, 1600, 1200, B_8_BIT_1600x1200 ,
	16, 640, 480, B_16_BIT_640x480,
	16, 800, 600, B_16_BIT_800x600,
	16, 1024, 768, B_16_BIT_1024x768,
	16, 1280, 1024, B_16_BIT_1280x1024,
	16, 1600, 1200, B_16_BIT_1600x1200,
	32, 640, 480, B_32_BIT_640x480,
	32, 800, 600, B_32_BIT_800x600,
	32, 1024, 768, B_32_BIT_1024x768,
	32, 1280, 1024, B_32_BIT_1280x1024,
	32, 1600, 1200, B_32_BIT_1600x1200,
   	8, 1152, 900, B_8_BIT_1152x900,
   	16, 1152, 900, B_16_BIT_1152x900,
   	32, 1152, 900, B_32_BIT_1152x900,
	// not supporting B_8_BIT_640x400
};


bool MapSpaceToRes( ulong space, int *xsiz, int *ysiz, int *depth )
{
	int i;
	
	for( i=0; i<sizeof(space_lut)/sizeof(space_lut[0]); i++ ) {
		if( space_lut[i].space == space ) {
			*xsiz = space_lut[i].width;
			*ysiz = space_lut[i].height;
			*depth = space_lut[i].depth;
			return TRUE;
		}
	}
	
	return FALSE;
}


bool MapResToSpace( int xsiz, int ysiz, int depth, ulong *space )
{
	int i;
	
	for( i=0; i<sizeof(space_lut)/sizeof(space_lut[0]); i++ ) {
		if( space_lut[i].width == xsiz && space_lut[i].height == ysiz && space_lut[i].depth == depth ) {
			*space = space_lut[i].space;
			return TRUE;
		}
	}
	
	return FALSE;
}



enum {

	// --------------------------------------------
	// --------------------------------------------
	//
	//   CHANGE THESE ENUMS TO CONFIGURE THE TEST
	//
	// --------------------------------------------


	// PCI configuration space board identification
	// Put your vendor and device IDs in here.
	VENDOR_ID	= 0x5333,	// S3
	DEVICE_ID	= 0x5631,	// ViRGE
	

	// This is the mode the driver should set the board to
	PREFERRED_SPACE = B_8_BIT_640x480,

	// This is the refresh rate the driver should set the board to
	#define REFRESH_RATE 60.1

	// This affects how we test the driver
	// This value must be one the GraphicsCardTests above.
	TEST_TO_PERFORM = kScreenSpaces

	// --------------------------------------------
	// --------------------------------------------
};


// The entrypoint for the video driver...
typedef long (*ctrl_grph_card)(ulong message,void *buf);
static ctrl_grph_card cgc;

// Implement the video driver's diagnostic output...
int xprintf( char *format, ... );


int xprintf( char *format, ... )
{
	// send all diagnostic strings to standard out...
	
	va_list ap;
	int retval;
	
	va_start( ap, format );
	retval = vprintf( format, ap );
	va_end( ap );
	
	return retval;
}


static long CallDriver( ulong msg, void *data )
{
	const static char *msglut[] = {
		"B_OPEN_GRAPHICS_CARD",
		"B_CLOSE_GRAPHICS_CARD",
		"B_GET_GRAPHICS_CARD_INFO",
		"B_GET_GRAPHICS_CARD_HOOKS",
		"B_SET_INDEXED_COLOR",
		"B_GET_SCREEN_SPACES",
		"B_CONFIG_GRAPHICS_CARD",
		"B_GET_REFRESH_RATES",
		"B_SET_SCREEN_GAMMA",
		"B_GET_INFO_FOR_CLONE_SIZE",
		"B_GET_INFO_FOR_CLONE",
		"B_SET_CLONED_GRAPHICS_CARD",
		"B_CLOSE_CLONED_GRAPHICS_CARD",	
		"B_PROPOSE_FRAME_BUFFER",
		"B_SET_FRAME_BUFFER",
		"B_SET_DISPLAY_AREA",
		"B_MOVE_DISPLAY_AREA"
	};
	
	const char *msg_str;
	long err;
	
	if( msg >= 0 && msg < sizeof(msglut)/sizeof(msglut[0]) ) {
		msg_str = msglut[msg];
	} else {
		msg_str = "UNKNOWN_MESSAGE";
	}
	
	printf( "calling %s...", msg_str );
	fflush( stdout );
	
	err = (cgc)( msg, data );
	if( err == B_NO_ERROR ) {
		printf( "  done, no error.\n" );
	} else {
		printf( "  ### ERROR %ld!\n", err );
	}
	
	return err;
}


static bool get_pci_info_by_id( ushort vid, ushort did, pci_info *pci )
{
	int i;
	for(i = 0; get_nth_pci_info(i,pci) == B_NO_ERROR; i++ ) {
		if( pci->vendor_id == vid && pci->device_id == did ) {
			return true;
		}
	}
	
	return false;
}


static void PrintGraphicsCardInfo( graphics_card_info *info )
{
	enum { kVersion = 2 };

	if( info->version != kVersion ) {
		printf( "### we don't recognize the version of this info!  (should be %d)\n",
			(int)kVersion );
	}

	printf( "  graphics_card_info:   version=%d, id=%d\n",
			(int)info->version,
			(int)info->id
		);
	
	printf( "    frame_buffer=0x%08lX, rgba_order=%c%c%c%c, flags=0x%04X\n",
			(long)info->frame_buffer,
			info->rgba_order[0], info->rgba_order[1],
			info->rgba_order[2], info->rgba_order[3],
			(int)info->flags
		);
	
	printf( "    bpp=%d, rowbytes=%d, width=%d, height=%d\n",
			(int)info->bits_per_pixel,
			(int)info->bytes_per_row,
			(int)info->width,
			(int)info->height
		);
}


static void PrintSupportedScreenSpaces( ulong space, char *str )
{
	int i, xsiz, ysiz, depth;
	
	printf( "  screen spaces represented by %08lX:\n", space );
	for( i=0; i<31; i++ ) {
		if( space & (1<<i) ) {
			if( MapSpaceToRes( 1<<i, &xsiz, &ysiz, &depth ) ) {
				printf( "    %2d) %s %d x %d at %d bpp\n",
					i, str, xsiz, ysiz, depth );
			} else {
				printf( "### PrintSupportedScreenSpaces: Could not MapSpaceToRes\n" );
			}
		}
	}
}


static long set_screen_res( long screen, ulong res )
{
	graphics_card_config	config;
	graphics_card_hook		vga_dispatch[B_HOOK_COUNT];
	long err;
	
	config.space = res;
	config.refresh_rate = REFRESH_RATE;
	config.h_position = 50;
	config.h_size = 50;
	config.v_position = 50;
	config.v_size = 50;

	printf( "> preparing to set screen res to 0x%08lX @ %2.1f Hz\n",
			(long)config.space,
			(float)config.refresh_rate
		);
		
	err = CallDriver( B_CONFIG_GRAPHICS_CARD, &config );
	// @@@ sic:
	if( err == B_ERROR ) return B_ERROR;
	
	err = CallDriver( B_GET_GRAPHICS_CARD_HOOKS, &vga_dispatch );
	if( err == B_ERROR ) {
		// Fill in defaults
		return B_ERROR;
	}
	
	return err;
}


static bool TST_OpenGraphicsCard( void )
{
	// returns true if the card was successfully opened.
	
	long err = B_NO_ERROR;
	graphics_card_spec cardrec, *card = &cardrec;
	pci_info pcirec, *pci = &pcirec;
	
	if( !get_pci_info_by_id( VENDOR_ID, DEVICE_ID, pci ) ) {
		printf( "### Card not installed (Vendor ID = 0x%04X, Device ID = 0x%04X)\n",
			(int)VENDOR_ID, (int)DEVICE_ID );
		return false;
	}

	card->screen_base = (void*)pci->u.h0.base_registers[0];
	card->io_base = 0;
#if !defined(__INTEL__)
	{
	area_id aid;
	area_info ai;
	// io_base is the same for all cards :-(
	aid = find_area("pci_bus0_isa_io");
	//aid = find_area("isa_io");
	if (aid < B_NO_ERROR) {
		printf("couldn't get area_id for pci_bus0_isa_io: %d\n", aid);
		return false;
	}
	get_area_info(aid, &ai);
	card->io_base = ai.address;
	printf("io_base is 0x%08X\n", ai.address);
	}
#endif

	card->vendor_id = VENDOR_ID;
	card->device_id = DEVICE_ID;		
	
	err = CallDriver( B_OPEN_GRAPHICS_CARD, card );
	if( err != B_NO_ERROR ) {
		printf( "Driver refused to open, here comes the close...\n", err );
		return false;
	}
	
	return true;
}


static bool TST_SetMode( void )
{
	graphics_card_info info;
	long err = B_NO_ERROR;
	
	err = CallDriver( B_GET_GRAPHICS_CARD_INFO, &info );
	if( err == B_NO_ERROR ) {
		PrintGraphicsCardInfo( &info );
		
		err = set_screen_res( 0, PREFERRED_SPACE );
		// @@@ We do as the AppServer does, but this should be err!=B_NO_ERROR.
		if( err == B_ERROR ) {
			
			printf( "### Space 0x%08lX failed, trying B_8_BIT_640x480...\n",
				(long)PREFERRED_SPACE );
				
			err = set_screen_res( 0, B_8_BIT_640x480 );
			
			if( err ) {
				printf( "The AppServer ignores this error result and proceeds\n"
						"as if nothing bad happened.  This might be a bug.\n" );
						// @@@
				err = B_NO_ERROR;
			}
			
		}
		
		if( err == B_NO_ERROR ) {
			refresh_rate_info refresh;
			
			// We should have a good raster.  The AppServer does not do
			// this, but just to be sure everything is working, we'll
			// print out what the driver thinks it is displaying...
			
			err = CallDriver( B_GET_GRAPHICS_CARD_INFO, &info );
			if( err == B_NO_ERROR ) {
				PrintGraphicsCardInfo( &info );
			}
			
			err = CallDriver( B_GET_REFRESH_RATES, &refresh );
			if( err == B_NO_ERROR ) {
				printf( "  refresh rates: cur=%2.1f, min=%2.1f, max=%2.1f\n",
					refresh.current, refresh.min, refresh.max );
			}
			
		}
		
	} else {
		printf(	"We'll do as the AppServer does, and simply blow\n"
				"away the driver if it returns an error from the\n"
				"first GetGraphicsCardInfo call (AppServer doesn't\n"
				"close it first).\n"
				// @@@
			);
		exit( err );
	}
	
	return err == B_NO_ERROR;
}


static void Paint8BitArea( void *base, int width, int height, int rowbytes, int color )
{
	char *cp = (char*)base;
	int row, col;
	
	if( width > rowbytes ) {
		printf( "### Param error to Clear8BitScreenToColor: width was greater than rowbytes." );
		return;
	}
	
	for( row=0; row<height; row++ ) {
		cp = (char*)( (long)base + (long)row*(long)rowbytes );
		if( color >= 0 ) {
			for( col=0; col<width; col++ ) {
				*cp++ = color;
			}
		} else {
			for( col=0; col<width; col++ ) {
				*cp++ = col % 256;
			}
		}
	}
}


static void Clear8BitScreenToColor( int color )
{
	long err;
	graphics_card_info info;
	
	err = CallDriver( B_GET_GRAPHICS_CARD_INFO, &info );
	if( err == B_NO_ERROR ) {
		if( info.version != 2 ) {
			printf( "### I do not understand this graphics_card_info.version: %d\n", (int)info.version );
		} else {
			Paint8BitArea(
					info.frame_buffer,
					info.width,
					info.height,
					info.bytes_per_row,
					color
				);
		}
	}
}


static void Checkpoint( char *curstatus, char *nextaction )
{
	char c;

	printf( "\n%s.  Hit return to %s...\n", curstatus, nextaction );
	fflush( stdout );
	scanf( "%c", &c );
	printf( "\n" );
}


static void SetCLUTToColor( char red, char green, char blue )
{
	indexed_color color;
	int i, x;
	
	for( i=0; i<256; i++ ) {
		color.index = i;
		color.color.red = red;
		color.color.green = green;
		color.color.blue = blue;
		
		CallDriver( B_SET_INDEXED_COLOR, &color );
	}
	
	snooze( 100000 );
}


static void SetCLUTToUsableColors( void )
{
	indexed_color color;
	int i, x;
	
/*	color.index = 0;
	color.color.red = 0;
	color.color.green = 0;
	color.color.blue = 0;
	color.color.alpha = 0;
	CallDriver( B_SET_INDEXED_COLOR, &color );
*/

	color.color.alpha = 0;
	for( i=0; i<256; i++ ) {
		color.index = i;
		color.color.red = (i*73)%256;
		color.color.green = (i*13)%256;
		color.color.blue = (i*179)%256;
		
		CallDriver( B_SET_INDEXED_COLOR, &color );
	}
	
	snooze( 100000 );
}



static void Frame8BitRect( void *base, int width, int height, int rowbytes,
	int color, int pensize )
{
	char *cp;
	int row, col;
	int i;
	
	row = 0;
	
	for( i=0; i<pensize; i++ ) {
		cp = (char*)( (long)base + (long)row*(long)rowbytes );
		for( col=0; col<width; col++ ) {
			*cp++ = color;
		}
		row += 1;
	}
	
	for( ; row<height-pensize; row++ ) {
		cp = (char*)( (long)base + (long)row*(long)rowbytes );
		for( i=0; i<pensize; i++ ) {
			*cp++ = color;
		}
		cp += width - 2*pensize;
		for( i=0; i<pensize; i++ ) {
			*cp++ = color;
		}
	}

	for( ; row<height; i++ ) {
		cp = (char*)( (long)base + (long)row*(long)rowbytes );
		for( col=0; col<width; col++ ) {
			*cp++ = color;
		}
		row += 1;
	}
}


static void Paint8BitConcentricRects( void *base, int width, int height, int rowbytes, int step, int wid )
{
	int wd, ht;
	int x, y;
	int col = 1;
	
	x = 0;
	y = 0;
	wd = width;
	ht = height;
	
	while( wd > step && ht > step ) {
		Frame8BitRect( (void*)( (char*)base + y*rowbytes + x ),
			wd, ht, rowbytes, col, wid );
		
		wd -= 2*step;
		ht -= 2*step;
		x += step;
		y += step;
		col = (col+1)%256;
	}
}


static void Draw8BitDesignOnScreen( void )
{
	long err;
	graphics_card_info info;
	
	err = CallDriver( B_GET_GRAPHICS_CARD_INFO, &info );
	if( err == B_NO_ERROR ) {
		if( info.version != 2 ) {
			printf( "### I do not understand this graphics_card_info.version: %d\n", (int)info.version );
		} else {
			Paint8BitArea(
					info.frame_buffer,
					info.width,
					info.height,
					info.bytes_per_row,
					0
				);
			Paint8BitConcentricRects(
					info.frame_buffer,
					info.width,
					info.height,
					info.bytes_per_row,
					32, 30
				);
		}
	}
}


static void Test_ScreenSpaces( void )
{
	graphics_card_config config;
	graphics_card_info info;
	refresh_rate_info refresh;

	int xsiz, ysiz, depth;
	long err;
	float f, nextrefresh = -1.0;
	
	SetCLUTToUsableColors();
	
	for( ;; ) {
		printf( "\n" );
		err = CallDriver( B_GET_SCREEN_SPACES, &config );
		if( err != B_NO_ERROR ) {
			break;
		}
		PrintSupportedScreenSpaces( config.space, "Set card to" );
		
		err = CallDriver( B_GET_REFRESH_RATES, &refresh );
		if( err != B_NO_ERROR ) {
			break;
		}
		if( nextrefresh < 0 ) {
			nextrefresh = refresh.current;
		}
		printf( "  refresh rates: current=%2.1f, min=%2.1f, max=%2.1f\n",
			refresh.current, refresh.min, refresh.max );
		printf( "next mode will be set to a %2.1f Hz refresh rate\n", nextrefresh );
		
		printf( "\nWhat should we do (0-31, %2.1f-%2.1f, or -1 to stop)? ",
			refresh.min, refresh.max );
		fflush( stdout );
		scanf( "%f", &f );
		
		printf( "\n" );
		
		if( f < 0 ) {
			// We quit the loop if the user enters a negative number.
			break;
		} else if( f >= 0 && f < 31 ) {
			// setting to a new mode...
		
			if( MapSpaceToRes( 1<<(int)f, &xsiz, &ysiz, &depth ) ) {
				printf( "Setting screen to %d x %d at %d bpp and %2.1f Hz refresh...\n",
					xsiz, ysiz, depth, nextrefresh );
					
				config.space = 1<<(int)f;
				config.refresh_rate = nextrefresh;
				config.h_position = config.v_position = 50;
				config.h_size = config.v_size = 50;
				
				err = CallDriver( B_CONFIG_GRAPHICS_CARD, &config );
				if( err != B_NO_ERROR ) {
					break;
				}
				
				Draw8BitDesignOnScreen();
					
				err = CallDriver( B_GET_GRAPHICS_CARD_INFO, &info );
				if( err != B_NO_ERROR ) {
					break;
				}
				
				PrintGraphicsCardInfo( &info );
				
			} else {
				printf( "Unknown screen space.  Try again.\n" );
			}
		} else if( f >= refresh.min && f <= refresh.max ) {
			// we'll set the next mode to a different refresh rate
			nextrefresh = f;
		} else {
			printf( "Number (%f) is out of range.\n", f );
		}
		
		printf( "\n" );
	}
}


static void TST_PerformTests( void )
{
	switch( TEST_TO_PERFORM ) {
	
		case kSolidPaletteTest:
			Checkpoint( "You should now have a raster", "solidify the screen" );
			Clear8BitScreenToColor( -1 );	
			Checkpoint( "You should now have a one-color screen", "set clut to red" );
			SetCLUTToColor( 0xFF, 0x00, 0x00 );
			Checkpoint( "You should now have a red clut", "set clut to green" );	
			SetCLUTToColor( 0x00, 0xFF, 0x00 );
			Checkpoint( "You should now have a green clut", "set clut to blue" );	
			SetCLUTToColor( 0x00, 0x00, 0xFF );
			Checkpoint( "You should now have a blue clut", "set clut to purple" );	
			SetCLUTToColor( 0xFF, 0x00, 0xFF );
			Checkpoint( "You should now have a purple clut", "quit" );
			break;
		
		case kScreenSpaces:
			Test_ScreenSpaces();
			break;
			
		default:
			printf( "... not performing any tests ...\n" );
	}
}


void main( void )
{
	image_id iid;
	status_t status;

	// load the add-on
	iid = load_add_on("ViRGE");
	if (iid < B_NO_ERROR) {
		printf("Couldn't load ViRGE driver add-on! %d\n", iid);
		return;
	}
	status = get_image_symbol(iid, "control_graphics_card__FUlPv", B_SYMBOL_TYPE_TEXT, &cgc);
	if (status < B_NO_ERROR) {
		status = get_image_symbol(iid, "control_graphics_card", B_SYMBOL_TYPE_TEXT, &cgc);
		if (status < B_NO_ERROR) {
			status = get_image_symbol(iid, "__imp__control_graphics_card", B_SYMBOL_TYPE_TEXT, &cgc);
			if (status < B_NO_ERROR) {
				printf("Couldn't get symbol from add-on! %d\n", status);
				return;
			}
		}
	}

	if( TST_OpenGraphicsCard() ) {
		if( TST_SetMode() ) {
			TST_PerformTests();
		}
	}
	
	// You ALWAYS get a close message, even if
	// you refuse to open.  Return value is ignored;
	// you're going away no matter what.
	CallDriver( B_CLOSE_GRAPHICS_CARD, 0L );
	
	printf( "\ndone.\n" );
}

