/* ++++++++++
	fx_driver.c

This is a small driver to return information on the PCI bus and allow
an app to find memory mappings for devices.

+++++ */

#include <OS.h>
#include <Drivers.h>
#include <KernelExport.h>
#include <PCI.h>
#include <stdlib.h>
#include <string.h>
#include <xprintf.h>
#include <stdio.h>

#define THDFX_VID 0x121A
#define THDFX_DID 0x0001

#include "3dfx_driver.h"

#define MAX_PCI_DEVICES 32

long init_driver (void);
static long fx_open (const char* name, ulong flags, void** cookie);
static long fx_close (void* cookie);
static long fx_free(void* cookie);
static long fx_read (void* cookie, off_t pos, void* buf, size_t* nbytes);
static long fx_write (void* cookie, off_t pos, const void* buf, size_t* nbytes);
static long fx_control (void* cookie, ulong msg, void* buf, size_t len);
static const char **add_device( const char *name );
device_hooks* find_device(const char* name);
const char** publish_devices( void );

static int		numDevices=0;
static pci_info gInfo[15];
static area_id  gID[15];


long
init_driver (void)
{
	dprintf (("PCI fx-driver: %s %s\n", __DATE__, __TIME__));
	return B_NO_ERROR;
}

static long
fx_open(const char* name, ulong flags, void** cookie)
{
	int			i,err;
	int			devNum;
	char 		areaName[100];
	area_id 	id;
  	
  
	dprintf("fx_open - ENTER, name is %s\n", name);
  
	for (i = 0; get_nth_pci_info(i, &gInfo[i]) == B_NO_ERROR ; i++) 
	{

		xprintf("Checking against %04lX/%04lX\n", gInfo[i].vendor_id,
	      			gInfo[i].device_id);
		if ((gInfo[i].vendor_id == THDFX_VID) && (gInfo[i].device_id == THDFX_DID)) 
		{
			xprintf("Found match.\n");
			*cookie = (void*)&gInfo[i];
			xprintf("Base register is %ld (%lX)\n", gInfo[i].u.h0.base_registers[0], 
						gInfo[i].u.h0.base_registers[0]);
			xprintf("Base register size is %ld (%lX)\n", gInfo[i].u.h0.base_register_sizes[0], 
						gInfo[i].u.h0.base_register_sizes[0]);
			sprintf(areaName, "pci_bus%d_dev%d_func%d_reg0", 
						gInfo[i].bus, gInfo[i].device, gInfo[i].function );
			xprintf("Looking for %s\n", areaName);
			id = find_area(areaName);
			if (id < 0)
			{
				void* addr;
				xprintf("not found, mapping\n");
				id = map_physical_memory(areaName, 
						(void*)gInfo[i].u.h0.base_registers[0],  
						  gInfo[i].u.h0.base_register_sizes[0],
						  B_ANY_KERNEL_ADDRESS,
						  B_WRITE_AREA + B_READ_AREA,
						  &addr);
			
			
	    	}
	    	#if 0
			{
				unsigned short cmd;
				cmd = read_pci_config(pci.bus, pci.device, pci.function, PCI_command, 2);
				cmd |= PCI_command_io | PCI_command_memory;
				write_pci_config(pci.bus, pci.device, pci.function, PCI_command, 2, cmd);		
			}
			#endif
	  	xprintf("id is %ld for area name %s\n", id, areaName);
	  	break;
		};
    }
  return B_NO_ERROR;        
}

static long
fx_close (void* _cookie)
{
	xprintf("fx_close - ENTER\n");

	return B_NO_ERROR;
}

static long
fx_free(void* cookie)
{
	xprintf("fx_free - ENTER\n");
}

static long
fx_read (void* cookie, off_t pos, void* buf, size_t* nbytes)
{
	xprintf("fx_read - ENTER\n");
	return 0;
}

static long
fx_write (void* cookie, off_t pos, const void* buf, size_t* nbytes)
{
	xprintf("fx_write - ENTER\n");
	return 0;
}

static long
fx_control (void* _cookie, ulong msg, void* buf, size_t len)
{	
	pci_info* info = (pci_info*)_cookie;
	xprintf("fx_control - ENTER\n");
  
	switch (msg) 
	{
		case THDFX_GET_AREA_ID:
		{
			char areaName[100];
			sprintf(areaName, "pci_bus%d_dev%d_func%d_reg0", 
						info->bus, info->device, info->function );
			xprintf("fx_control - looking for area %s\n", areaName);
			*(area_id*)buf = find_area(areaName);
		}
		break;
		
    	default:
      		return B_ERROR;
      	break;
    }
  
	xprintf("fx_control - EXIT\n");
	return B_NO_ERROR;
}

static const char **add_device( const char *name )
{
  // arbitrary constants, grrr...
	// This MAY be improved before we ship DR9.
	
	enum {
		max_name = 2048,
		max_dev = 32
	};

	static char bbuf[ max_name ];
	static char *b = bbuf;
	static char *pbuf[ max_dev ];
	static char **p = pbuf; 
	int l;

	if( !name ) {
		goto bail;
	}
	
	l = 1 + strlen( name );	
	if( p + 1 > pbuf + max_dev || b + l > bbuf + max_name ) {
		goto bail;
	}
	
	*p = b;
	p += 1;

	memcpy( b, name, l );
	b += l;	
	
bail:
	return (const char**)pbuf;
}

#pragma export on
device_hooks* find_device(const char* name)
{
	static device_hooks fx_hooks = 
	{
		fx_open,
		fx_close,
		fx_free,
		fx_control,
		fx_read,
		fx_write
	};
	xprintf("find_device - ENTER\n");
	return &fx_hooks;
}


// Graphics Device Conventions:
// - Even though you can, do not put your drivers in subdirectories.
//   I don't want to have to climb trees to find graphics drivers.
// - Try to make graphics device names persistent.  bus.def

const char *devname = "3dfx";

const char** publish_devices( void )
{
	pci_info h;
	long i;
	
	xprintf( "3dfx graphics driver: publishing devices.\n");
	for (i = 0; get_nth_pci_info(i, &h) == B_NO_ERROR; i++)
	{
			if (( h.vendor_id == 0x121A ) && (h.device_id == 0x0001))
			{
				char* s = (char*)malloc(strlen(devname) + i/10 + 1);
				sprintf(s, "%s_%d", devname, i);
				add_device( s );
				xprintf("Device  added\n");
			} 
			else 
			{
				xprintf( "3dfx: Not a 3dfx chip, vid = %04lX,  did=%04lX\n",  h.vendor_id, h.device_id );
			}
		}
	xprintf("public_devices - EXIT\n");	
	return add_device( NULL );
}

#pragma export reset

