/*
 * Ethernet Driver Template - PCI
 * CopyRight (c) 1999 Be Inc., All Rights Reserved.
 * This file may be used under the terms of the Be Sample Code License.
 */

#include <OS.h>
#include <KernelExport.h>
#include <Drivers.h>
#include <stdarg.h>
#include <PCI.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <SupportDefs.h>
#include <ByteOrder.h>
#include "ether_driver.h"

/* PCI vendor and device ID's */
#define VENDOR_ID        0x1234
#define DEVICE_ID        0x4567

/* globals for driver instances */ 

#define kDevName "AcmeEthernet"
#define kDevDir "net/" kDevName "/"
#define DEVNAME_LENGTH		64			
#define MAX_CARDS 			 4			/* maximum number of driver instances */



/*  Choose "memory mapped" or "io port" pci bus access. */
#define IO_PORT_PCI_ACCESS true
//#define MEMORY_MAPPED_PCI_ACCESS true

#if IO_PORT_PCI_ACCESS
#define write8(address, value)			(*gPCIModInfo->write_io_8)((address), (value))
#define write16(address, value)			(*gPCIModInfo->write_io_16)((address), (value))
#define write32(address, value)			(*gPCIModInfo->write_io_32)((address), (value))
#define read8(address)					((*gPCIModInfo->read_io_8)(address))
#define read16(address)					((*gPCIModInfo->read_io_16)(address))
#define read32(address)					((*gPCIModInfo->read_io_32)(address))

#else /*  MEMORY_MAPPED_PCI_ACCESS */
#define read8(address)   				(*((volatile uint8*)(address))); __eieio()
#define read16(address)  				(*((volatile uint16*)(address))); __eieio()
#define read32(address) 				(*((volatile uint32*)(address))); __eieio()
#define write8(address, data)  			(*((volatile uint8 *)(address)) = data); __eieio()
#define write16(address, data) 			(*((volatile uint16 *)(address)) = (data)); __eieio()
#define write32(address, data) 			(*((volatile uint32 *)(address)) = (data)); __eieio()
#endif 


/* debug flags */
#define ERR       0x0001
#define INFO      0x0002
#define RX        0x0004		/* dump received frames */
#define TX        0x0008		/* dump transmitted frames */
#define INTERRUPT 0x0010		/* interrupt calls */
#define FUNCTION  0x0020		/* function calls */
#define PCI_IO    0x0040		/* pci reads and writes */
#define SEQ		  0x0080		/* trasnmit & receive TCP/IP sequence sequence numbers */
#define WARN	  0x0100		/* Warnings - off on final release */

/* diagnostic debug flags - compile in here or set while running with debugger "AcmeRoadRunner" command */
#define DEBUG_FLAGS ( ERR | INFO | WARN )


#define BUFFER_SIZE			2048L           /* B_PAGE_SIZE divided into even amounts that will hold a 1518 frame */
#define MAX_FRAME_SIZE		1514			/* 1514 + 4 bytes checksum */
static pci_module_info		*gPCIModInfo;
static char 				*gDevNameList[MAX_CARDS+1];
static pci_info 			*gDevList[MAX_CARDS+1];
static int32 				gOpenMask = 0;


const char** publish_devices(void ) {
	dprintf(kDevName ": publish_devices()\n" );
	return (const char **)gDevNameList;
}


/* Driver Entry Points */
__declspec(dllexport) status_t init_hardware(void );
__declspec(dllexport) status_t init_driver(void );
__declspec(dllexport) void uninit_driver(void );
__declspec(dllexport) const char** publish_devices(void );
__declspec(dllexport) device_hooks *find_device(const char *name );


/* Hardware specific definitions */
/* ring buffer sizes  */
#define TX_BUFFERS             4
#define RX_BUFFERS             4        /* Must be a power of 2 */

/* Transmit and receive frame descriptor */
typedef struct frame_descriptor {
    uint32 reserved;						/* hardware specfic frame descriptor */
} frame_desc_t;


/* per driver intance globals */
typedef struct {
	ether_address_t mac_address;								/* my ethernet address */
	int32			devID; 										/* device identifier: 0-n */
	pci_info		*pciInfo;
	uint16			irq;										/* our IRQ line */
	sem_id			ilock, olock;								/* i/o semaphores */
	int32			readLock, writeLock;						/* reentrant reads/writes not allowed */
	int32			blockFlag;									/* for blocking or nonblocking reads */
	area_id			tx_desc_area, tx_buf_area; 					/* transmit descriptor and buffer areas */
	area_id			rx_desc_area, rx_buf_area; 					/* receive descriptor and buffer areas */
	spinlock		rx_spinlock;								/* for read_hook() and rx_interrupt() */
	uchar			*tx_buf[TX_BUFFERS], *rx_buf[RX_BUFFERS];	/* tx and rx buffers */
	frame_desc_t	*tx_desc[TX_BUFFERS], *rx_desc[RX_BUFFERS];	/* tx and rx frame descriptors */
	int16			rx_received, rx_acked;						/* index to in use tx & rx ring buffers */
	int16			tx_sent, tx_acked;							/* index to oldest used tx & rx ring buffers */
	uint32			reg_base; 									/* Base address of hostRegs */
	uint32			media;										/* speed duplex media settings */
	uint32			debug;										/* debuging level */
	/*  statistics */
	uint32			handled_int, unhandled_int;					/* interrupts rececived & handled */
	uint32			stats_rx, stats_tx;							/* receive & transmit frames counts */
	uint32			stats_rx_err, stats_tx_err;
	uint32			stats_rx_overrun, stats_tx_underrun;
	uint32			stats_collisions;  							/* error stats */
} dev_info_t;


void DEBUG(dev_info_t * device, int32 flags, char * format, ...) {
	if (device->debug & flags) {
		va_list		args;
		char		s[4096];
		va_start(args, format);
		vsprintf( s, format, args );
		va_end(args);
		dprintf("%s",s);
	}
}

/* for serial debug command*/
#define DEBUGGER_COMMAND true
#if DEBUGGER_COMMAND
dev_info_t * gdev;
static int 		AcmeEthernet(int argc, char **argv);					   /* serial debug command */
#endif


/* prototypes */
static status_t open_hook(const char *name, uint32 flags, void **_cookie);
static status_t close_hook(void *_device);
static status_t free_hook(void *_device);
static status_t control_hook(void * cookie,uint32 msg,void *buf,size_t len);
static status_t read_hook(void *_device,off_t pos, void *buf,size_t *len);
static status_t write_hook(void *_device, off_t pos, const void *buf, size_t *len);
static int32    interrupt_hook(void *_device);                	   /* interrupt handler */

static int32    get_pci_list(pci_info *info[], int32 maxEntries);  /* Get pci_info for each device */
static status_t free_pci_list(pci_info *info[]);                   /* Free storage used by pci_info list */
static status_t enable_addressing(dev_info_t *device);             /* enable pci io address space for device */
static status_t init_ring_buffers(dev_info_t *device);             /* allocate and initialize frame buffer rings */
static void     free_ring_buffers(dev_info_t *device);             /* allocate and initialize frame buffer rings */
static void     get_mac_address(dev_info_t *device);               /* get ethernet address */
static status_t setpromisc(dev_info_t * device);                   /* set hardware to receive all packets */
static status_t domulti(dev_info_t *device, uint8 *addr);		   /* add multicast address to hardware filter list */
static void     reset_device(dev_info_t *device);                  /* reset the lan controller (NIC) hardware */
static void 	dump_packet(const char * msg, unsigned char * buf, uint16 size); /* diagnostic packet trace */
static void 	dump_rx_desc(dev_info_t *device);				   /* dump hardware receive descriptor structures */
static void 	dump_tx_desc(dev_info_t *device);				   /* dump hardware transmit descriptor structures */
static void		clear_statistics(dev_info_t *device);			   /* clear statistics counters */
static status_t allocate_resources(dev_info_t *device);     	   /* allocate semaphores & spinlocks */
static void     free_resources(dev_info_t *device);                /* deallocate semaphores & spinlocks */


static device_hooks gDeviceHooks =  {
	open_hook, 			/* -> open entry point */
	close_hook,         /* -> close entry point */
	free_hook,          /* -> free entry point */
	control_hook,       /* -> control entry point */
	read_hook,          /* -> read entry point */
	write_hook,         /* -> write entry point */
	NULL,               /* -> select entry point */
	NULL,                /* -> deselect entry point */
	NULL,               /* -> readv */
	NULL                /* -> writev */
};


/* Driver Entry Points */
status_t init_hardware(void )
{
	status_t		status;

	dprintf(kDevName ": init hardware ");	
	/* Check for hardware with init_driver() */
	if ((status = init_driver()) == B_OK )
		uninit_driver();
	return status;
}


status_t init_driver()
{
	status_t 		status;
	int32			entries;
	char			devName[64];
	int32			i;
	
	dprintf(kDevName ": init_driver ");	

	if((status = get_module( B_PCI_MODULE_NAME, (module_info **)&gPCIModInfo )) != B_OK) {
		dprintf(kDevName " Get module failed! %s\n", strerror(status ));
		return status;
	}
		
	/* Find Lan cards*/
	if ((entries = get_pci_list(gDevList, MAX_CARDS )) == 0) {
		dprintf("init_driver: " kDevName " not found\n");
		free_pci_list(gDevList);
		put_module(B_PCI_MODULE_NAME );
		return B_ERROR;
	}
	
	/* Create device name list*/
	for (i=0; i<entries; i++ )
	{
		sprintf(devName, "%s%ld", kDevDir, i );
		gDevNameList[i] = (char *)malloc(strlen(devName)+1);
		strcpy(gDevNameList[i], devName);
	}
	gDevNameList[i] = NULL;
	
	return B_OK;
}

void uninit_driver(void)
{
	int32 	i;
	void 	*item;

	/*Free device name list*/
	for (i=0; (item=gDevNameList[i]); i++)
		free(item);
	
	/*Free device list*/
	free_pci_list(gDevList);
	put_module(B_PCI_MODULE_NAME);
}


device_hooks *find_device(const char *name)
{
	int32 	i;
	char 	*item;

	/* Find device name */
	for (i=0; (item=gDevNameList[i]); i++)
	{
		if (strcmp(name, item) == 0)
		{
			return &gDeviceHooks;
		}
	}
	return NULL; /*Device not found */
}


static status_t open_hook(const char *name, uint32 flags, void **cookie) {

	int32				devID;
	int32				mask;
	status_t			status;
	char 				*devName;
	dev_info_t 		*device;

	/*	Find device name*/
	for (devID=0; (devName=gDevNameList[devID]); devID++) {
		if (strcmp(name, devName) == 0)
			break;
	}
	if (!devName)
		return EINVAL;
	
	/* Check if the device is busy and set in-use flag if not */
	mask = 1 << devID;
	if (atomic_or(&gOpenMask, mask) &mask) {
		return B_BUSY;
	}

	/*	Allocate storage for the cookie*/
	if (!(*cookie = device = (dev_info_t *)malloc(sizeof(dev_info_t)))) {
		status = B_NO_MEMORY;
		goto err0;
	}
	memset(device, 0, sizeof(dev_info_t));
	
	/* Setup the cookie */
	device->pciInfo = gDevList[devID];
	device->devID = devID;

	
	device->debug = DEBUG_FLAGS;
	
	DEBUG(device, FUNCTION, kDevName ": open %s device=%x\n", name, device);
	
#if DEBUGGER_COMMAND
	gdev = device;
	add_debugger_command (kDevName, AcmeEthernet, "Ethernet driver Info");
#endif

	/* enable access to the cards address space */
	if ((status = enable_addressing(device)) != B_OK)
		goto err1;

	if (allocate_resources(device) != B_OK) {
		goto err1;
	}	
	
	clear_statistics(device);
	
	/* Init Device */
	reset_device(device);
	
	get_mac_address(device);

	/* allocate and initialize frame buffer rings & descriptors */
	if (init_ring_buffers(device) != B_OK)
		goto err2;
	
	/* Setup interrupts */
	install_io_interrupt_handler( device->pciInfo->u.h0.interrupt_line, interrupt_hook, *cookie, 0 );
		
	DEBUG(device, INFO, kDevName ": add hardware specific init here\n");	

	return B_NO_ERROR;

	err2:
		free_ring_buffers(device);
		
	err1:
		free_resources(device);
		free(device);	
	
	err0:
		atomic_and(&gOpenMask, ~mask);
		DEBUG(device, ERR, kDevName ": open failed!\n");
		return status;

}

static status_t close_hook(void *_device) {
	dev_info_t *device = (dev_info_t *) _device;

	DEBUG(device, FUNCTION, kDevName ": close_hook");

	DEBUG(device, INFO, kDevName ": add hardware specific code to halt NIC and DMA activity\n");	
		
	/* Release resources we are holding */
	

#if DEBUGGER_COMMAND
	remove_debugger_command (kDevName, AcmeEthernet);
#endif
	
	free_resources(device);
			
	return (B_NO_ERROR);
}


static status_t free_hook(void * cookie) {
	dev_info_t *device = (dev_info_t *) cookie;

	DEBUG(device, FUNCTION, kDevName": free %x\n",device);

    /* Remove Interrupt Handler */
	remove_io_interrupt_handler( device->pciInfo->u.h0.interrupt_line, interrupt_hook, cookie );
	
	free_ring_buffers(device);
	
    /* Free Areas */
#if MEMORY_MAPPED_PCI_ACCESS
	delete_area(device->reg_area);
#endif

	/* Device is now available again */
	atomic_and(&gOpenMask, ~(1 << device->devID));

	free(device);
	return 0;
}


/* Standard driver control function */
static status_t control_hook(void * cookie,uint32 msg,void *buf,size_t len)
{
	dev_info_t *device = (dev_info_t *) cookie;

	switch (msg) {
		case ETHER_GETADDR: {
			uint8 i;
			DEBUG(device, FUNCTION, kDevName ": control %x ether_getaddr\n",msg);
			for (i=0; i<6; i++) {
				((uint8 *)buf)[i] = device->mac_address.ebyte[i];
			}
			return B_NO_ERROR;
		}
		case ETHER_INIT:
			DEBUG(device, FUNCTION, kDevName ": control %x init\n",msg);
			return B_NO_ERROR;
			
		case ETHER_GETFRAMESIZE:
			DEBUG(device, FUNCTION, kDevName ": control %x get_framesize\n",msg);
			*(uint32 *)buf = MAX_FRAME_SIZE;
			return B_NO_ERROR;
			
		case ETHER_ADDMULTI:
			DEBUG(device, FUNCTION, kDevName ": control %x add multi\n",msg);
			return (domulti(device, (unsigned char *)buf));
		
		case ETHER_SETPROMISC:
			DEBUG(device, FUNCTION, kDevName ": control %x set promiscuous\n",msg);
			return B_ERROR;
		
		case ETHER_NONBLOCK:
			DEBUG(device, FUNCTION, kDevName ": control %x blocking\n",msg);

			if (*((int32 *)buf))
				device->blockFlag = B_TIMEOUT;
			else
				device->blockFlag = 0;
			return B_NO_ERROR;
			
		default:
			DEBUG(device, ERR, kDevName ": unknown iocontrol %x\n",msg);
			return B_ERROR;
	}

}


/* The read() system call - upper layer interface to the ethernet driver */
static status_t  read_hook(void *_device,off_t pos, void *buf,size_t *len)
{
	dev_info_t  *device = (dev_info_t *) _device;
	status_t    status;
		
	*len = 0;
	
	/* Block until data is available */
	if((status = acquire_sem_etc( device->ilock, 1, B_CAN_INTERRUPT | device->blockFlag, 0 )) != B_NO_ERROR) {
		*len = 0;
		return status;
	}
	/* Prevent reentrant read */
	if( atomic_or( &device->readLock, 1 ) ) {
		release_sem_etc( device->ilock, 1, 0 );
		*len = 0;
		return B_ERROR;
	}

	DEBUG(device, INFO, kDevName "Add hardware specific code to transfer data from NIC into buf\n");
	/* Add hardware specific code to copy data from the NIC into buf.
	   If the frame is good copy the data into "buf" and set *len.
	   If the frame had an error increment the stats counters and receive the
	   next good frame or return a *len=0.
	   
	   Also dumping the recevied TCP/IP sequence number and doing a ping flood
	   will tell how well the driver receives packets in sequence.
	   For example, 
	   { unsigned short  *seq = (unsigned short *) device->rx_buf[rx_now];
		 DEBUG(device, SEQ, " R%4.4x ", seq[20]);
	   }
	*/
	
	/* update acked index for next read call, and release reantrant lock */
	device->readLock = 0;
	return B_OK;
}


/*
 * The write() system call - upper layer interface to the ethernet driver
 */
static status_t write_hook(void *_device, off_t pos, const void *buf, size_t *len)
{
	dev_info_t  *device = (dev_info_t *) _device;
	int16        frame_size;
	status_t      status;
	int16         tx_now;
	
	DEBUG(device, FUNCTION, kDevName ":write buf %x len %d (decimal)\n",buf,*len);
 
	if( *len > MAX_FRAME_SIZE ) {
		DEBUG(device, ERR, kDevName ": write %d > 1514 tooo long\n",*len);
		*len = MAX_FRAME_SIZE;
	}
	frame_size = *len;
	
	if( (status = acquire_sem_etc( device->olock, 1, B_CAN_INTERRUPT, 0 )) != B_NO_ERROR ) {
		*len = 0;
		return status;
	}

	/* Prevent reentrant write */
	if( atomic_or( &device->writeLock, 1 ) ) {
		release_sem_etc( device->olock, 1, 0 );
		*len = 0;
		return B_ERROR;
	}

	if (device->debug & TX) {
		dump_packet("tx", (unsigned char *) buf, frame_size);
	}

/* ------- */	
	DEBUG(device, INFO, "Add hardware specific code to transmit buff out the wire here\n");
/* ------- */	

	/* Another write may now take place */
	device->writeLock = 0;
	return B_OK;
}


/* service interrupts generated by the Lan controller (card) hardware */
static int32
interrupt_hook(void *_device)
{
	dev_info_t *device = (dev_info_t *) _device;
	int32 handled = B_UNHANDLED_INTERRUPT;
	int16 worklimit = 20;					/* process as many frames as reasonable, but limit time in interrupt */
	int32 AcmeInterrupRegister;
		
	DEBUG(device, INTERRUPT, kDevName ": ISR_ENTRY\n");	
	
	while ((AcmeInterrupRegister = 0/*read NIC hardware interrupt reg*/)  && (worklimit-- > 0)) {
		
        /* ack (clear) and re-enable interrupts */

		handled = B_INVOKE_SCHEDULER;   /* Set because the interrupt was from the NIC, not some other device sharing the interrupt line */
		device->handled_int++;  		/* update stats */
		
		DEBUG(device, INTERRUPT, kDevName ": ISR - its ours\n");
	}
		
	if (handled == B_UNHANDLED_INTERRUPT) {
		device->unhandled_int++;
	}
	return handled;
}


static int32 get_pci_list(pci_info *info[], int32 maxEntries)
{
	status_t status;
	int32 i, entries;
	pci_info	*item;
	
	item = (pci_info *)malloc(sizeof(pci_info));
	
	for (i=0, entries=0; entries<maxEntries; i++) {
		if ((status = gPCIModInfo->get_nth_pci_info(i, item)) != B_OK)
			break;
		dprintf(kDevName ": Check Vendor and Device ID here\n");
		if ( 0 || (item->vendor_id == VENDOR_ID)&& (item->device_id == DEVICE_ID)) {
			/* check if the device really has an IRQ */
			if ((item->u.h0.interrupt_line == 0) || (item->u.h0.interrupt_line == 0xFF)) {
				dprintf(kDevName " found with invalid IRQ - check IRQ assignement");
				continue;
			}
			dprintf(kDevName " found at IRQ %x ", item->u.h0.interrupt_line);
			info[entries++] = item;
			item = (pci_info *)malloc(sizeof(pci_info));
		}
	}
	info[entries] = NULL;
	free(item);
	return entries;
}

static status_t free_pci_list(pci_info *info[])
{
	int32 		i;
	pci_info	*item;
	
	for (i=0; (item=info[i]); i++)
		free(item);
	return B_OK;
}


static status_t enable_addressing(dev_info_t *device)
{

	/* Turn on I/O port decode, Memory Address Decode, and Bus Mastering - this may or may not be necessary for your hardware */
	gPCIModInfo->write_pci_config(device->pciInfo->bus, device->pciInfo->device, 
		device->pciInfo->function, PCI_command, 2,
		PCI_command_io | PCI_command_memory | PCI_command_master |
		gPCIModInfo->read_pci_config( device->pciInfo->bus, device->pciInfo->device, 
		device->pciInfo->function, PCI_command, 2));

#if IO_PORT_PCI_ACCESS
	device->reg_base = device->pciInfo->u.h0.base_registers[0];
#endif /* IO_PORT_PCI_ACCESS */

#if MEMORY_MAPPED_PCI_ACCESS
	{
	int32		base, size, offset, i;
	int32		*ptr;
	const		MemoryReg = 1;
	
	base = device->pciInfo->u.h0.base_registers[MemoryReg];
	size = device->pciInfo->u.h0.base_register_sizes[MemoryReg];
	
	/* use "poke" with the "pci" command from a terminal session to check these values */
	DEBUG(device, PCI_IO,  kDevName ": PCI base=%x size=%x\n", base, size);

	/* Round down to nearest page boundary */
	base = base & ~(B_PAGE_SIZE-1);
	
	/* Adjust the size */
	offset = device->pciInfo->u.h0.base_registers[MemoryReg] - base;
	size += offset;
	size = (size +(B_PAGE_SIZE-1)) & ~(B_PAGE_SIZE-1);

	DEBUG(device, PCI_IO,  kDevName ": Now PCI base=%x size=%x offset=%x\n", base, size, offset);
		
	if ((device->reg_area = map_physical_memory(kDevName " Regs", (void *)base, size,
		B_ANY_KERNEL_ADDRESS, B_READ_AREA | B_WRITE_AREA, &device->reg_base)) < 0)
		return B_ERROR;

	device->reg_base = device->reg_base + offset;
	}
#endif /*MEMORY_MAPPED_PCI_ACCESS */
	
	DEBUG(device, PCI_IO,  kDevName ": reg_base=%x\n", device->reg_base);

	return B_OK;
}


#define RNDUP(x, y) (((x) + (y) - 1) & ~((y) - 1))

static status_t init_ring_buffers(dev_info_t *device)
{
	uint32 	size;
	physical_entry		entry;
	uint16 i;
		
	/* create transmit buffer area */
	size = RNDUP(BUFFER_SIZE * TX_BUFFERS, B_PAGE_SIZE);
	if ((device->tx_buf_area = create_area( kDevName " tx buffers", (void **) device->tx_buf,
		B_ANY_KERNEL_ADDRESS, size, B_FULL_LOCK, B_READ_AREA | B_WRITE_AREA )) < 0) {
		DEBUG(device, ERR, kDevName " create tx buffer area failed %x \n", device->tx_buf_area);
		return device->tx_buf_area;
	}
	/* create tx descriptor area */
	size = RNDUP( sizeof(frame_desc_t) * TX_BUFFERS, B_PAGE_SIZE);
	if ((device->tx_desc_area = create_area( kDevName " tx descriptors", (void **) device->tx_desc,
		B_ANY_KERNEL_ADDRESS, size, B_FULL_LOCK, B_READ_AREA | B_WRITE_AREA )) < 0) {
		DEBUG(device, ERR, kDevName " create tx descriptor area failed %x \n", device->tx_desc_area);
		delete_area(device->tx_buf_area);
		return device->tx_desc_area;
	}


	/* initilize descriptors */
	for ( i = 0; i < TX_BUFFERS; i++) {
		device->tx_desc[i]->reserved = 0;
	}

	/* create rx buffer area */
	size = RNDUP(BUFFER_SIZE * RX_BUFFERS, B_PAGE_SIZE);
	if ((device->rx_buf_area = create_area( kDevName " rx buffers", (void **) device->rx_buf,
		B_ANY_KERNEL_ADDRESS, size, B_FULL_LOCK, B_READ_AREA | B_WRITE_AREA )) < 0) {
		DEBUG(device, ERR, kDevName " create rx buffer area failed %x \n", device->rx_buf_area);
		delete_area(device->tx_buf_area);
		delete_area(device->tx_desc_area);
		return device->rx_buf_area;
	}
	/* create rx descriptor area */
	size = RNDUP( sizeof(frame_desc_t) * RX_BUFFERS, B_PAGE_SIZE);
	if ((device->rx_desc_area = create_area( kDevName " rx descriptors", (void **) device->rx_desc,
		B_ANY_KERNEL_ADDRESS, size, B_FULL_LOCK, B_READ_AREA | B_WRITE_AREA )) < 0) {
		DEBUG(device, ERR, kDevName " create rx descriptor area failed %x \n", device->rx_desc_area);
		delete_area(device->tx_buf_area);
		delete_area(device->tx_desc_area);
		delete_area(device->rx_buf_area);
		return device->rx_desc_area;
	}

	/* init rx buffer descriptors */
	for ( i = 0; i < RX_BUFFERS; i++) {
		device->rx_desc[i]->reserved = 0;
	}

	/* initialize frame indexes */
	device->tx_sent = device->tx_acked = device->rx_received = device->rx_acked = 0;

	return B_OK;
}

static void free_ring_buffers(dev_info_t *device) {

		delete_area(device->tx_buf_area);
		delete_area(device->tx_desc_area);
		delete_area(device->rx_buf_area);
		delete_area(device->rx_desc_area);

}


static void get_mac_address(dev_info_t *device) {

	int j;
	DEBUG(device, INFO, kDevName ": Mac address: ");
	for (j=0; j<6; j++) {
		dprintf("Get Ethernet Mac address from SROM or hardware registers\n");
		device->mac_address.ebyte[j] = j;
		DEBUG(device, INFO," %x", device->mac_address.ebyte[j]);
	}
}


/* set hardware so all packets are received. */
static status_t setpromisc(dev_info_t * device) {
	DEBUG(device, FUNCTION, kDevName ":setpormisc\n");
	dprintf("Set promiscous mode here\n");
	return(B_NO_ERROR);
}

static status_t domulti(dev_info_t *device, uint8 *addr) {
	
	DEBUG(device, FUNCTION,kDevName ": domulti %2.2x:%2.2x:%2.2x:%2.2x:%2.2x:%2.2x\n", 
					addr[0],addr[1],addr[2],addr[3],addr[4],addr[5]);
	dprintf("Set up multicast filter here\n");
	return (B_NO_ERROR);
}

static void reset_device(dev_info_t *device) {
	DEBUG(device, FUNCTION,kDevName ": reset_device reset the NIC hardware\n"); 
};

static void dump_packet( const char * msg, unsigned char * buf, uint16 size) {

	uint16 j;
	
	dprintf("%s dumping %x size %d \n", msg, buf, size);
	for (j=0; j<size; j++) {
		if ((j & 0xF) == 0) dprintf("\n");
		dprintf("%2.2x ", buf[j]);
	}
}


static void dump_rx_desc(dev_info_t *device) {
	uint16 j;
	/* kprintf is used here since this is called from the serial debug command */
	kprintf("rx desc %8.8x \n", device->rx_desc);
	for (j=0; j < RX_BUFFERS; j++ ) {
		kprintf("rx_desc[%2.2d]=...\n", j);
	}
}

static void dump_tx_desc(dev_info_t *device) {
	uint16 j;
	/* kprintf is used here since this is called from the serial debug command */
	kprintf("tx desc %8.8x \n", device->tx_desc);
	
	for (j=0; j < TX_BUFFERS; j++ ) {
		kprintf("tx_desc[%2.2d]...\n",j);
	}
}

/* Serial Debugger command
   Connect a terminal emulator to the serial port at 19.2 8-1-None
   Press the keys ( alt-sysreq on Intel) or (Clover-leaf Power on Mac ) to enter the debugger
   At the kdebug> prompt enter "AcmeEthernet arg...",
   for example "AcmeEthernet R" to enable a received packet trace.
*/
#if DEBUGGER_COMMAND
static int
AcmeEthernet(int argc, char **argv) {
	uint16 i,j;
	const char * usage = "usage: AcmeEthernet { Function_calls | Interrupts | Number_frames | PCI_IO, Stats | Rx_trace | Tx_trace }\n";	
	

	if (argc < 2) {
		kprintf("%s",usage);	return 0;
	}

	for (i= argc, j= 1; i > 1; i--, j++) {
		switch (*argv[j]) {
		case 'F':
		case 'f':
			gdev->debug ^= FUNCTION;
			if (gdev->debug & FUNCTION) 
				kprintf("Function() call trace Enabled\n");
			else 			
				kprintf("Function() call trace Disabled\n");
			break; 
		case 'N':
		case 'n':
			gdev->debug ^= SEQ;
			if (gdev->debug & SEQ) 
				kprintf("Sequence numbers packet trace Enabled\n");
			else 			
				kprintf("Sequence numbers packet trace Disabled\n");
			break; 
		case 'R':
		case 'r':
			gdev->debug ^= RX;
			if (gdev->debug & RX) 
				kprintf("Receive packet trace Enabled\n");
			else 			
				kprintf("Receive packet trace Disabled\n");
			break;
		case 'T':
		case 't':
			gdev->debug ^= TX;
			if (gdev->debug & TX) 
				kprintf("Transmit packet trace Enabled\n");
			else 			
				kprintf("Transmit packet trace Disabled\n");
			break; 
		case 'S':
		case 's':
			kprintf(kDevName " statistics\n");			
			kprintf("received %d,  transmitted %d\n", gdev->stats_rx, gdev->stats_tx);
			kprintf("rx err %d,  tx err %d\n", gdev->stats_rx_err, gdev->stats_tx_err);
			kprintf("rx overrun %d,  tx underrun %d\n", gdev->stats_rx_overrun, gdev->stats_tx_underrun);
			kprintf("collisions %d\n", gdev->stats_collisions);
			kprintf("handled %d and unhandled %d interrupts\n", gdev->handled_int, gdev->unhandled_int);
			break; 
		case 'P':
		case 'p':
			gdev->debug ^= PCI_IO;
			if (gdev->debug & PCI_IO) 
				kprintf("PCI IO trace Enabled\n");
			else 			
				kprintf("PCI IO trace Disabled\n");
			break; 
		default:
			kprintf("%s",usage);
			return 0;
		}
	}
	
	return 0;
}
#endif /* DEBUGGER_COMMAND */

static void clear_statistics(dev_info_t *device) {
	
	device->handled_int = device->unhandled_int = 0;
	device->stats_rx = device->stats_tx = 0;
	device->stats_rx_err = device->stats_tx_err = 0;
	device->stats_rx_overrun = device->stats_tx_underrun = 0;
	device->stats_collisions = 0;
}

/*
 * Allocate and initialize semaphores and spinlocks.
 */
static status_t allocate_resources(dev_info_t *device) {
		
	/* Setup Semaphores */
	if ((device->ilock = create_sem(0, kDevName " rx")) < 0) {
		dprintf(kDevName " create rx sem failed %x \n", device->ilock);
		return (device->ilock);
	}
	set_sem_owner(device->ilock, B_SYSTEM_TEAM);
	
	/* intialize tx semaphore with the number of free tx buffers */
	if ((device->olock = create_sem(TX_BUFFERS, kDevName " tx")) < 0) {
		delete_sem(device->ilock);
		dprintf(kDevName " create tx sem failed %x \n", device->olock);
		return (device->olock);
	}
	set_sem_owner(device->olock, B_SYSTEM_TEAM);

	device->readLock = device->writeLock = 0;

	device->rx_spinlock = 0;

	device->blockFlag = 0;
		
	return (B_OK);
}

static void free_resources(dev_info_t *device) {
		delete_sem(device->ilock);
		delete_sem(device->olock);
}

int32	api_version = B_CUR_DRIVER_API_VERSION;
