/* Copyright 2000, Be Incorporated. All Rights Reserved.
** This file may be used under the terms of the Be Sample Code License.
*/

#include "CPiA_priv.h"
#include "CPiA.h"

#include <KernelExport.h>

#include <malloc.h>
#include <string.h>

int32 api_version = B_CUR_DRIVER_API_VERSION;

static char* usb_name = B_USB_MODULE_NAME;
static usb_module_info *usb;

static device* connected_dev = NULL; /* the one connected device we allow */
static int32 dev_count = 0;	/* number of connected devices (only one allowed) */

static sem_id dev_table_lock = -1; /* lock to synchronize access to device table */

static char** dev_names = NULL; /* /dev-relative path of each connected device */

/* ----------------------------------------------------------------------------
** Prototypes
*/

static void
cpia_command_callback(void* cookie, uint32 status, void* data, uint32 actual_len);					

static device*
add_device(const usb_device* dev);

static void
remove_device(device* dev);

static status_t
init_device(device* dev, const usb_device* usb_dev, const int32 num);

static void
deinit_device(device* dev);

/* ----------------------------------------------------------------------------
** Functions
*/

/* send a CPiA command to the camera */
static status_t
issue_cpia_command(	device* dev,
					uint8 request_type,
					uint8 request,
					uint16 lo_value,
					uint16 hi_value,
					uint16 lo_index,
					uint16 hi_index,
					uint16 lo_length,
					uint16 hi_length,
					void* data)
{
	status_t err;
	
	dprintf(ID "issue_cpia_command(%p, 0x%02x, 0x%02x, 0x%02x, 0x%02x, "
		"0x%02x, 0x%02x, 0x%02x, 0x%02x, %p)\n", dev, request_type, request,
		lo_value, hi_value, lo_index, hi_index, lo_length, hi_length, data);
	dprintf(ID "dev->dev is %p\n", dev->dev);
	
	/* CPiA commands use the USB SETUP packet definition */
	err = usb->queue_request(dev->dev, request_type, request,
		hi_value << 8 | lo_value, hi_index << 8 | lo_index,
		hi_length << 8 | lo_length, dev->buf, hi_length << 8 | lo_length,
		cpia_command_callback, dev);
	if (err == B_OK) {
		/* wait for queued request to complete */
		err = acquire_sem(dev->cmd_complete);
		if (err != B_OK) {
			goto exit;
		}
		err = dev->cmd_status;
		if (data != NULL) {
			memcpy(data, dev->buf, dev->actual_len);
		}
	}
		
exit:
	return err;
}

/* callback for issue_cpia_command() */
static void
cpia_command_callback(void* cookie, uint32 status, void* data, uint32 actual_len)
{
	device* dev = (device*)cookie;
	if (dev != NULL) {
		dev->cmd_status = status;
		if (data != NULL) {
			memcpy(dev->buf, data, actual_len);
		}
		dev->actual_len = actual_len;
		/* indicate queued request is complete */
		release_sem(dev->cmd_complete);
	}
	else {
		/* Oops. No device cookie. */
		delete_sem(dev->cmd_complete);
	}
}

/* initialize device-specific data and resources */
static status_t
init_device(device* dev, const usb_device* usb_dev, const int32 num)
{
	status_t err;
	
	dev->dev = usb_dev;
	dev->connected = true;
	dev->open_count = 0;
	dev->num = num;

	/* a self-aligned buffer that is smaller than
	** B_PAGE_SIZE will be contiguous. Not sure if control
	** transfers need to made into contiguous buffers
	** (isochronous transfers do), but I'm being conservative.
	*/
	dev->buf = (uchar*)memalign(64, 64);
	if (dev->buf == NULL) {
		err = B_NO_MEMORY;
		goto exit;
	}
	dprintf("dev->buf at %p\n", dev->buf);

	dev->lock = create_sem(1, ID "lock");
	if (dev->lock < 0) {
		err = dev->lock;
		goto exit;
	}

	/* Init cmd_complete semaphore with 0 so that 1st
	** acquire_sem() blocks. release_sem() will be called
	** on it in cpia_command_callback function.
	*/
	dev->cmd_complete = create_sem(0, ID "cmd_complete");
	if (dev->cmd_complete < 0) {
		err = dev->cmd_complete;
		goto exit;
	}
	
	/* temp */
	dev->video_in = NULL;
	dev->max_packet_size = 0;

exit:
	return err;
}

/* free device-specific resources */
static void
deinit_device(device* dev)
{
	delete_sem(dev->lock);
	delete_sem(dev->cmd_complete);
	if (dev->buf != NULL) {
		free(dev->buf);
	}
}

/* add a device to the list of connected devices */
/* ||| currently only one device supported */
static device*
add_device(const usb_device* dev)
{
	int32 i;
	device* new_dev = NULL;
	
	dprintf(ID "add_device()\n");
	
	if (dev_count >= MAX_NUM_DEVS) {
		dprintf(ID "device table full\n");
		goto exit;
	}
	
	acquire_sem(dev_table_lock);
	if (new_dev = (device*)malloc(sizeof(device))) {
		dprintf(ID "adding device %p\n", new_dev);
		connected_dev = new_dev;
		init_device(new_dev, dev, dev_count);
		dev_count++;
	}
	
	release_sem(dev_table_lock);
	

exit:
	return new_dev;
}

/* called by USB Manager when device is added to the USB */
static status_t
device_added(const usb_device* dev, void** cookie)
{
	device* new_dev = NULL;
	status_t err = B_ERROR;
	uchar data[8]; /* store bytes returned from CPiA commands */
	
	dprintf(ID "device_added(%p)\n", dev);

	new_dev = add_device(dev);
	if (new_dev != NULL) {
		/* set the cookie that will be passed to other USB
		** hook functions (currently device_removed() is the
		** only other)
		*/
		*cookie = new_dev;
		
		/* get the version info via GetCPIAVersion command */
		err = issue_cpia_command(new_dev, 0xC0, 0x01, 0, 0, 0, 0, 0x04, 0x00,
			new_dev->vers);
		if (err != B_OK) {
			dprintf(ID "issue_cpia_command() returned %ld\n", err);
		}
		else {
			dprintf(ID "GetCPIAVersion returned 0x%02x, 0x%02x, 0x%02x, 0x%02x\n",
				new_dev->vers[0], new_dev->vers[1], new_dev->vers[2],
				new_dev->vers[3]);
		}
		
		/* check for camera self test errors via GetCameraStatus command */
		err = issue_cpia_command(new_dev, 0xC0, 0x03, 0, 0, 0, 0, 0x08, 0x00, data);
		if (err != B_OK) {
			dprintf(ID "issue_cpia_command() returned %ld\n", err);
		}
		else {
			dprintf(ID "GetCameraStatus returned 0x%02x, 0x%02x, 0x%02x, 0x%02x\n, "
				"0x%02x, 0x%02x, 0x%02x, 0x%02x\n",
				data[0], data[1], data[2], data[3],
				data[4], data[5], data[6], data[7]);
			if (data[3]/* fatal error bit mask */ != 0) {
				dprintf(ID "fatal error 0x%02x. Removing device.\n", data[3]);
				remove_device(new_dev);
			}	
		}		
	}
	else {	
		dprintf(ID "Couldn't allocate device record.\n");
	}
	
exit:
	return err;
}

/* remove a device from the list of connected devices */
/* ||| currently only one device supported */
static void
remove_device(device* dev)
{
	dprintf(ID "remove_device(%p)\n", dev);

	deinit_device(dev);
	free(dev);
	if (dev != connected_dev) {
		dprintf(ID "got a weird device* in remove_device()\n");
	}
	connected_dev = NULL;
	dev_count--;
}	

/* called by USB Manager when device is removed from the USB */
static status_t
device_removed(void* cookie)
{
	device* dev = (device*)cookie;
	
	dprintf(ID "device_removed(%p)\n", dev);

	acquire_sem(dev_table_lock);

	dev->connected = false;
	dev_count--;
	/* kill the device's lock, so no one else tries to use it */
	delete_sem(dev->lock);
	if (dev->open_count == 0) {
		remove_device(dev);
	}
	
	release_sem(dev_table_lock);
	
	return B_OK;
}

static usb_notify_hooks notify_hooks = 
{
	&device_added,
	&device_removed
};

/* implements the POSIX open() */
static status_t
device_open(const char *name, uint32 flags, void **cookie)
{
	status_t err = ENODEV;
	device* dev = connected_dev;
	uchar data[8]; /* bytes returned from CPiA commands */
	const usb_configuration_info* config;
	const usb_interface_info* interface;
	size_t alt_count;
	
	dprintf(ID "device_open()\n");
	
	acquire_sem(dev_table_lock);

	if (dev->connected) {
		dev->open_count++;
		/* set the cookie that will be passed to other
		** device_*() functions
		*/
		*cookie = dev;
		err = B_OK;
	}
	else {
		dprintf(ID "Going offline -- can't be opened.\n");
	}
	
	release_sem(dev_table_lock);
	
	acquire_sem(dev->lock);
	
	/* issue GotoHiPower */
	err = issue_cpia_command(dev, 0x40, 0x04, 0, 0, 0, 0, 0, 0, NULL);
	
	/* check again for errors */
	err = issue_cpia_command(dev, 0xC0, 0x03, 0, 0, 0, 0, 0x08, 0x00, data);
	if (err != B_OK) {
		dprintf(ID "issue_cpia_command() returned %ld\n", err);
	}
	else {
		dprintf(ID "GetCameraStatus returned 0x%02x, 0x%02x, 0x%02x, 0x%02x\n, "
			"0x%02x, 0x%02x, 0x%02x, 0x%02x\n",
			data[0], data[1], data[2], data[3],
			data[4], data[5], data[6], data[7]);
		if (data[3]/* fatal error bit mask */ != 0) {
			dprintf(ID "fatal error 0x%02x.\n", data[3]);
			err = B_ERROR;
			goto exit;
		}	
	}
	
	/* select highest Alternate of Interface 1 */
	config = usb->get_nth_configuration(dev->dev, 0);
	if (config != NULL) {
		alt_count = config->interface->alt_count;
		dprintf(ID "found %ld alt interfaces.\n", alt_count);
		
		interface = &config->interface->alt[alt_count-1];
		err = usb->set_alt_interface(dev->dev, interface);
		if (err != B_OK) {
			dprintf(ID "set_alt_interface() returned %x.\n", err);
			goto exit;
		}
		
		/* call set_configuration() only after calling
		** set_alt_interface()
		*/
		err = usb->set_configuration(dev->dev, config);
		if (err != B_OK) {
			dprintf(ID "set_configuration() returned %x.\n", err);
			goto exit;
		}
		
		/* stash away a reference to this endpoint's pipe and its maximum
		** USB packet payload
		*/
		dev->video_in = config->interface->active->endpoint[0].handle;
		dev->max_packet_size =
			config->interface->active->endpoint[0].descr->max_packet_size;
		dprintf(ID "dev->max_packet_size set to %d.\n", dev->max_packet_size);
	}
	else {
		dprintf(ID "couldn't get default config.\n");
	}

exit:
	release_sem(dev->lock);
	
	return err;
}

/* called when a client calls POSIX close() on the driver, but I/O
** requests may still be pending
*/
static status_t
device_close(void *cookie)
{
	device* dev = (device*)cookie;
	
	dprintf(ID "device_close() called on \"%s%d\"\n",DEVICE_PATH,
		dev->num);
		
	return B_OK;
}

/* called after device_close(), when all pending I/O requests have
** returned
*/
static status_t
device_free (void *cookie)
{
	status_t err;
	device* dev = (device*)cookie;
	
	dprintf(ID "device_free() called on \"%s%d\"\n",DEVICE_PATH,
		dev->num);
		
	acquire_sem(dev_table_lock);

	if (--dev->open_count == 0) {
		if (dev->connected) {
			/* issue GotoLowPower */
			err = issue_cpia_command(dev, 0x40, 0x05, 0, 0, 0, 0, 0, 0, NULL);
		}
		else {
			/* The last client has closed, and the device is no longer
			** connected, so remove it from the list.
			*/
			remove_device(dev);
		}
	}
	
	release_sem(dev_table_lock);

	return err;
}

/* implements the POSIX ioctl() */
static status_t
device_control(void *cookie, uint32 msg, void *params, size_t size)
{
	status_t err = B_OK;
	device* dev = (device*)cookie;
	cpia_capture_info* info = (cpia_capture_info*)params;
	uchar data[8];
	rlea* rle_array;

	dprintf(ID "device_control()\n");
	
	if (dev == NULL) {
		dprintf(ID "Bad cookie.\n");
		return ENODEV;
	}
	
	err = acquire_sem(dev->lock);
	if (err != B_OK) {
		dprintf(ID "Couldn't acquire device lock.\n");
		goto exit;
	}
	
	switch (msg) {
	case CPIA_GRAB_1_FRAME:
		dprintf(ID "case %d\n", msg);
		/* make sure we got a valid buffer */
		if (info->buf == NULL) {
			dprintf(ID "Invalid buffer from user space.\n");
			err = ENOMEM;
			goto cleanup1;
		}
		
		/* issue SetFormat command */
		err = issue_cpia_command(dev, 0x40, 0xC8, info->video_size,
			info->subsampling, info->yuv_order, 0, 0, 0, NULL);
		
		usb->set_pipe_policy(dev->video_in, 1, 1000/15/* 1000 ms/15 frames*/,
			dev->max_packet_size);
		
		/* issue SetGrabMode */
		err = issue_cpia_command(dev, 0x40, 0xC3, 0/* single grab */, 0, 0,
			0, 0, 0, NULL);
		if (err != B_OK) {
			goto cleanup1;
		}

		/* check status again, just for grins */
		err = issue_cpia_command(dev, 0xC0, 0x03, 0, 0, 0, 0, 0x08, 0x00, data);
		if (err != B_OK) {
			dprintf(ID "issue_cpia_command() returned %ld\n", err);
		}
		else {
			dprintf(ID "GetCameraStatus returned 0x%02x, 0x%02x, 0x%02x, 0x%02x\n, "
				"0x%02x, 0x%02x, 0x%02x, 0x%02x\n",
				data[0], data[1], data[2], data[3],
				data[4], data[5], data[6], data[7]);
			if (data[3]/* fatal error bit mask */ != 0) {
				dprintf(ID "fatal error 0x%02x.\n", data[3]);
				err = B_ERROR;
				goto cleanup1;
			}	
		}		
		
/* Not quite ready to take this code live yet; leave for reference. */
#if 0
		/* issue GrabFrame command */
		err = issue_cpia_command(dev, 0x40, 0xC1, 0, (info->video_size ==
			VIDEO_SIZE_CIF) ? 120 : 60/* 0.5 * line to start streaming */,
				0, 0, 0, 0, NULL);
			
		/* queue an isochronous transaction */
		rle_array = (rlea*)malloc(sizeof (rlea) + (4-1)*sizeof (rle));
		rle_array->length = 4;
		err = usb->queue_isochronous(dev->video_in, info->buf, info->buf_size,
			rle_array, 1000/15/* 1000 ms/15 frames*/, iso_in_callback, dev);
#endif			
		break;
		
	default:
		dprintf(ID "Invalid ioctl() opcode %ld.\n", msg);
		err = B_DEV_INVALID_IOCTL;
		break;
	}
	
cleanup1:
	release_sem(dev->lock);
	
exit:
	return err;
}

/* implements the POSIX read() */
static status_t
device_read(void *cookie, off_t pos, void *buf, size_t *count)
{
	dprintf(ID "device_read()\n");
	
	return B_ERROR;
}

/* implements the POSIX write() */
static status_t
device_write(void *cookie, off_t pos, const void *buf, size_t *count)
{
	dprintf(ID "device_write()\n");
	
	return B_ERROR;
}

usb_support_descriptor supported_devices[] =
{
	{ 0, 0, 0, VENDOR_ID, PRODUCT_ID }
};	

/* called each time the driver is loaded by the kernel */
status_t
init_driver(void)
{
	dprintf(ID "init_driver()\n");
	
	if (get_module(usb_name,(module_info**)&usb) != B_OK) {
		dprintf(ID "cannot get module \"%s\"\n", usb_name);
		return B_ERROR;
	}
	dprintf(ID "usb module at %p\n", usb);
	
	usb->register_driver(DEVICE_NAME, supported_devices, 1, NULL);
	usb->install_notify(DEVICE_NAME, &notify_hooks);

	dev_table_lock = create_sem(1, ID "dev_table_lock");
	if (dev_table_lock < 0) {
		goto err;
	}
	
	return B_OK;
	
err:
	put_module(usb_name);
	return B_ERROR;
}

/* called just before the kernel unloads the driver */
void
uninit_driver(void)
{
	int32 i;
	
	dprintf(ID "uninit_driver()\n");
	
	if (dev_names != NULL) {
		for (i = 0; dev_names[i] != NULL; i++) {
			dprintf(ID "i: %ld\n", i);
			free(dev_names[i]);
		}
		free(dev_names);
		dev_names = NULL;
	}
	
	usb->uninstall_notify(DEVICE_NAME);
	
	delete_sem(dev_table_lock);
	
	if (connected_dev != NULL) {
		dprintf(ID "Device %p still exists at teardown.\n",
			connected_dev);
	}
	
	put_module(usb_name);
}

const char**
publish_devices(void)
{
	int32 i;
	device* dev = connected_dev;
	
	dprintf(ID "publish_devices()\n");
	
	if (dev_names != NULL) {
		for (i = 0; dev_names[i] != NULL; i++) {
			dprintf(ID "i: %ld\n", i);
			free(dev_names[i]);
		}
		free(dev_names);
		dev_names = NULL;
	}
	
	if (dev == NULL) {
		dprintf(ID "No devices connected.\n");
		return NULL;
	}
	
	acquire_sem(dev_table_lock);

	dev_names = (char**)malloc(sizeof (char*) * (dev_count+1));
	if (dev_names) {
		for (i = 0; i < MAX_NUM_DEVS; i++) {
			if ((dev != NULL) &&
					(dev_names[i] = (char*)malloc(strlen(DEVICE_PATH)+2/* num + \n */))) {
				sprintf(dev_names[i], "%s%d", DEVICE_PATH, dev->num);
				dprintf(ID "publishing \"%s\"\n", dev_names[i]);
			}
		}
		/* list must be NULL-terminated */
		dev_names[i] = NULL;
	}

	release_sem(dev_table_lock);
	
	return (const char**)dev_names;
}

static device_hooks hooks = {
	device_open, 			
	device_close, 			
	device_free,			
	device_control, 		
	device_read,			
	device_write,
	NULL,
	NULL,
	NULL,
	NULL		 
};

device_hooks*
find_device(const char* name)
{
	dprintf(ID "find_device(%s)\n", name);
	
	return &hooks;
}
