/*
 * Copyright (c) 1992 The Regents of the University of California.
 * All rights reserved.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice and the following
 * two paragraphs appear in all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include "24bit.h"

/*
 * We'll define the "ConvertColor" macro here to do fixed point arithmetic
 * that'll convert from YCrCb to RGB using:
 *	R = L + 1.40200*Cr;
 *	G = L - 0.34414*Cb - 0.71414*Cr
 *	B = L + 1.77200*Cb;
 *
 * We'll use fixed point by adding two extra bits after the decimal.
 */

#define BITS	8
#define ONE     ((int) 1)
#define CONST_SCALE	(ONE << BITS)
#define ROUND_FACTOR	(ONE << (BITS-1))

/* Macro to convert integer to fixed. */
#define UP(x)	(((int)(x)) << BITS)

#define RED_SHIFT	(0)
#define GREEN_SHIFT	(8)
#define BLUE_SHIFT	(16)

#define RED_MASK	(0xff00)
#define GREEN_MASK	(0xff0000)
#define BLUE_MASK	(0xff000000)

/* Macro to convert fixed to integer (with rounding). */
#define DOWN(x)	(((x) + ROUND_FACTOR) >> BITS)

/* Macro to convert a float to a fixed */
#define FIX(x)  ((int) ((x)*CONST_SCALE + 0.5))

#define CLAMP(ll,x,ul)	( ((x)<(ll)) ?(ll):( ((x)>(ul)) ?(ul):(x)))

ADither24::ADither24()
{
    int CR, CB, i;

	// Initialize color lookup tables
	for (i=0; i<256; i++) 
	{
		CB = CR = i;

		CB -= 128; 
		CR -= 128;

		Cb_r_tab[i] = FIX(1.40200) * CB;
		Cr_g_tab[i] = -FIX(0.34414) * CR;
		Cb_g_tab[i] = -FIX(0.71414) * CB;   
		Cr_b_tab[i] = FIX(1.77200) * CR;
	}
}

void
ADither24::DitherImage(unsigned char *lum, unsigned char *cr, unsigned char *cb, 
	unsigned char *out, int rows, int cols)
{
    register int Lum, CR, CB;
    register unsigned int *row1, *row2;
    register unsigned char *lum2;
    register int x, y;
    register unsigned int r, b, g;
    register int cb_r;
    register int cr_g;
    register int cb_g;
    register int cr_b;

    row1 = (unsigned int *)out;
    row2 = row1 + cols;
    lum2 = lum + cols;
    for (y=0; y<rows; y+=2) {
	for (x=0; x<cols; x+=2) {
	    register int R, G, B;

	    CR = *cr++;
	    CB = *cb++;
	    cb_r = Cb_r_tab[CB];
	    cr_g = Cr_g_tab[CR];
	    cb_g = Cb_g_tab[CB];
	    cr_b = Cr_b_tab[CR];

	    Lum = *lum++;
	    Lum = UP(Lum);
	    R = Lum + cb_r;
	    G = Lum + cr_g + cb_g;
	    B = Lum + cr_b;

	    b = (CLAMP(0,B,UP(255))	<<	BLUE_SHIFT) & BLUE_MASK;
	    g = (CLAMP(0,G,UP(255)) << GREEN_SHIFT) & GREEN_MASK;
	    r = (CLAMP(0,R,UP(255)) << RED_SHIFT) & RED_MASK;

	    *row1++ = r | g | b;

	    Lum = *lum++;
	    Lum = UP(Lum);
	    R = Lum + cb_r;
	    G = Lum + cr_g + cb_g;
	    B = Lum + cr_b;

	    b = (CLAMP(0,B,UP(255)) <<	BLUE_SHIFT) & BLUE_MASK;
	    g = (CLAMP(0,G,UP(255)) << GREEN_SHIFT) & GREEN_MASK;
	    r = (CLAMP(0,R,UP(255)) << RED_SHIFT) & RED_MASK;

	    *row1++ = r | g | b;

	    Lum = *lum2++;
	    Lum = UP(Lum);
	    R = Lum + cb_r;
	    G = Lum + cr_g + cb_g;
	    B = Lum + cr_b;

	    b = (CLAMP(0,B,UP(255))	<<	BLUE_SHIFT) & BLUE_MASK;
	    g = (CLAMP(0,G,UP(255)) << GREEN_SHIFT) & GREEN_MASK;
	    r = (CLAMP(0,R,UP(255)) << RED_SHIFT) & RED_MASK;

	    *row2++ = r | g | b;

	    Lum = *lum2++;
	    Lum = UP(Lum);
	    R = Lum + cb_r;
	    G = Lum + cr_g + cb_g;
	    B = Lum + cr_b;

	    b = (CLAMP(0,B,UP(255))	<<	BLUE_SHIFT) & BLUE_MASK;
	    g = (CLAMP(0,G,UP(255)) << GREEN_SHIFT) & GREEN_MASK;
	    r = (CLAMP(0,R,UP(255)) << RED_SHIFT) & RED_MASK;

	    *row2++ = r | g | b;
	}
	lum += cols;
	lum2 += cols;
	row1 += cols;
	row2 += cols;
    }

}

/*
 *--------------------------------------------------------------
 *
 * InitColorDither --
 *
 *	To get rid of the multiply and other conversions in color
 *	dither, we use a lookup table.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The lookup tables are initialized.
 *
 *--------------------------------------------------------------
 */
void
InitColorDither()
{
/*
    int CR, CB, i;

    Cr_b_tab = (int *)malloc(256*sizeof(int));
    Cr_g_tab = (int *)malloc(256*sizeof(int));
    Cb_g_tab = (int *)malloc(256*sizeof(int));
    Cb_r_tab = (int *)malloc(256*sizeof(int));

    for (i=0; i<256; i++) {
	CB = CR = i;

	CB -= 128; CR -= 128;

	Cb_r_tab[i] = FIX(1.40200) * CB;
	Cr_g_tab[i] = -FIX(0.34414) * CR;
	Cb_g_tab[i] = -FIX(0.71414) * CB;   
	Cr_b_tab[i] = FIX(1.77200) * CR;
    }
*/
}


/*
 *--------------------------------------------------------------
 *
 * ColorDitherImage --
 *
 *	Converts image into 24 bit color.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

void
ColorDitherImage(unsigned char *lum, unsigned char *cr, unsigned char *cb, 
	unsigned char *out, int rows, int cols)
{
/*
    register int L, CR, CB;
    register unsigned int *row1, *row2;
    register unsigned char *lum2;
    register int x, y;
    register unsigned int r, b, g;
    register int cb_r;
    register int cr_g;
    register int cb_g;
    register int cr_b;

    row1 = (unsigned int *)out;
    row2 = row1 + cols;
    lum2 = lum + cols;
    for (y=0; y<rows; y+=2) {
	for (x=0; x<cols; x+=2) {
	    register int R, G, B;

	    CR = *cr++;
	    CB = *cb++;
	    cb_r = Cb_r_tab[CB];
	    cr_g = Cr_g_tab[CR];
	    cb_g = Cb_g_tab[CB];
	    cr_b = Cr_b_tab[CR];

	    L = *lum++;
	    L = UP(L);
	    R = L + cb_r;
	    G = L + cr_g + cb_g;
	    B = L + cr_b;

	    b = (CLAMP(0,B,UP(255))	<<	BLUE_SHIFT) & BLUE_MASK;
	    g = (CLAMP(0,G,UP(255)) << GREEN_SHIFT) & GREEN_MASK;
	    r = (CLAMP(0,R,UP(255)) << RED_SHIFT) & RED_MASK;

	    *row1++ = r | g | b;

	    L = *lum++;
	    L = UP(L);
	    R = L + cb_r;
	    G = L + cr_g + cb_g;
	    B = L + cr_b;

	    b = (CLAMP(0,B,UP(255)) <<	BLUE_SHIFT) & BLUE_MASK;
	    g = (CLAMP(0,G,UP(255)) << GREEN_SHIFT) & GREEN_MASK;
	    r = (CLAMP(0,R,UP(255)) << RED_SHIFT) & RED_MASK;

	    *row1++ = r | g | b;

	    L = *lum2++;
	    L = UP(L);
	    R = L + cb_r;
	    G = L + cr_g + cb_g;
	    B = L + cr_b;

	    b = (CLAMP(0,B,UP(255))	<<	BLUE_SHIFT) & BLUE_MASK;
	    g = (CLAMP(0,G,UP(255)) << GREEN_SHIFT) & GREEN_MASK;
	    r = (CLAMP(0,R,UP(255)) << RED_SHIFT) & RED_MASK;

	    *row2++ = r | g | b;

	    L = *lum2++;
	    L = UP(L);
	    R = L + cb_r;
	    G = L + cr_g + cb_g;
	    B = L + cr_b;

	    b = (CLAMP(0,B,UP(255))	<<	BLUE_SHIFT) & BLUE_MASK;
	    g = (CLAMP(0,G,UP(255)) << GREEN_SHIFT) & GREEN_MASK;
	    r = (CLAMP(0,R,UP(255)) << RED_SHIFT) & RED_MASK;

	    *row2++ = r | g | b;
	}
	lum += cols;
	lum2 += cols;
	row1 += cols;
	row2 += cols;
    }
*/
}


// Dynamically loaded structure
#ifdef DITHER_ADDON

DITHERER dither_module = {
	NULL,
	"24-Bit Color",
	"1.0",
	InitColorDither,
	ColorDitherImage
};

#endif

