#ifndef BOLTS_SEQUENCE
#define BOLTS_SEQUENCE

#include <assert.h>

template <class AType>
class ASequenceOf
{
public:
		ASequenceOf(const unsigned long nElements);
		ASequenceOf(const unsigned long nElements, const AType& initialValue);
		ASequenceOf(const ASequenceOf&);
		~ASequenceOf();
		
		unsigned long	Length() const;
		unsigned long	SetSize(const unsigned long nElements);
		unsigned long	SetSize(const unsigned long nElements, const AType& newValue);
		
		void	SetAllToValue(const AType& newValue);
		
		AType& operator[](const unsigned long idx) const;

protected:
	AType*			fData;
	unsigned long	fSize;
};


//==============================================
// Implementation
//==============================================

template<class AType> ASequenceOf<AType>::ASequenceOf(const unsigned long nElements)
	: fData(0),
	fSize(0)
{
	fData = new AType[nElements];
	// Make sure the buffer is actually allocated
	// Really, this should throw an exception, but we'll
	// let the underlying allocation code do that for us.
	// even this assertion is a bit nasty in that your progam
	// will typically exit on an assertion, and that may not be what you want.
	assert(fData != 0);
	fSize = nElements;
}

template<class AType> ASequenceOf<AType>::ASequenceOf(const unsigned long nElements, const AType& initialValue)
	: fData(0),
	fSize(0)
{
	fData = new AType[nElements];
	assert(fData!=0);
	fSize = nElements;

	// Assign the initial value to all the cells
	for (long counter = 0; counter < nElements; counter++)
		fData[counter] = initialValue;
}

template<class AType> ASequenceOf<AType>::ASequenceOf(const ASequenceOf<AType>& other)
	: fData(0),
	fSize(0)
{
	fData = new AType[other.fSize];
	assert(fData != 0);

	fSize = other.fSize;

	for (long counter = 0; counter < fSize; counter++)
   	fData[counter] = other.fData[counter];
}

template<class AType> ASequenceOf<AType>::~ASequenceOf()
{
	fSize =0;
	delete [] fData;
	fData = 0;
}

template<class AType> AType& ASequenceOf<AType>::operator[](const unsigned long idx) const
{
	assert(idx < fSize);

	return fData[idx];
}

template<class AType> unsigned long
ASequenceOf<AType>::Length() const
{
	return fSize;
}

template<class AType> unsigned long
ASequenceOf<AType>::SetSize(const unsigned long nElements, const AType& initialValue)
{
	AType* newData = new AType[nElements];
	assert(newData != 0);
	long counterLimit;

	if (nElements <= fSize)
	{
		counterLimit = nElements;
	} else
	{
		counterLimit = fSize;

		// Set the initial values for the new cells
		for (long ctr3=fSize; ctr3<nElements; ctr3++)
		{
			newData[ctr3] = initialValue;
		}
	}

	// Copy the existing values into the new data area
	for (long ctr1 =0; ctr1 < counterLimit; ctr1++)
	{
			newData[ctr1] = fData[ctr1];
	}

	delete [] fData;
	fData = newData;
	fSize = nElements;

	return fSize;
}

template<class AType> unsigned long
ASequenceOf<AType>::SetSize(const unsigned long nElements)
{
	AType* newData = new AType[nElements];
	assert(newData != 0);

	if (nElements <= fSize)
	{
		// Shrinking down from our current size
		for (long ctr1 =0; ctr1 < nElements; ctr1++)
		{
			newData[ctr1] = fData[ctr1];
		}
	} else
	{
		for (long ctr2 = 0; ctr2 < fSize; ctr2++)
		{
			newData[ctr2] = fData[ctr2];
		}

	}

	delete [] fData;
	fData = newData;
	fSize = nElements;

	return fSize;

}

template<class AType> void
ASequenceOf<AType>::SetAllToValue(const AType& newValue)
{
	// Shrinking down from our current size
	for (long ctr1 =0; ctr1 < fSize; ctr1++)
	{
		fData[ctr1] = newValue;
	}
}



// Utility functions for sequences and others
template <class AType>
AType Max(AType a, AType b)
{
	if (a<b)
		return b;
	return a;
}

template <class AType>
void Swap(ASequenceOf<AType> theSeq, unsigned long idx1, unsigned long idx2)
{
	AType tempValue = theSeq[idx1];
	theSeq[idx1] = theSeq[idx2];
	theSeq[idx2] = tempValue;
}



#endif
