#include <stdio.h>
#include <string.h>

#include "rraster.h"
#include <File.h>

/* Header definition. */
struct ImageHeader {
    unsigned char IDLength;		/* length of Identifier String */
    unsigned char CoMapType;		/* 0 = no map */
    unsigned char ImgType;		/* image type (see below for values) */
    unsigned char Index_lo, Index_hi;	/* index of first color map entry */
    unsigned char Length_lo, Length_hi;	/* number of entries in color map */
    unsigned char CoSize;		/* size of color map entry (15,16,24,32) */
    unsigned char X_org_lo, X_org_hi;	/* x origin of image */
    unsigned char Y_org_lo, Y_org_hi;	/* y origin of image */
    unsigned char Width_lo, Width_hi;	/* width of image */
    unsigned char Height_lo, Height_hi;	/* height of image */
    unsigned char PixelSize;		/* pixel size (8,16,24,32) */
    unsigned char AttBits;		/* 4 bits, number of attribute bits per pixel */
    unsigned char Rsrvd;		/* 1 bit, reserved */
    unsigned char OrgBit;		/* 1 bit, origin: 0=lower left, 1=upper left */
    unsigned char IntrLve;		/* 2 bits, interleaving flag */
    };

typedef char ImageIDField[256];

/* Definitions for image types. */
#define TGA_Null 0
#define TGA_Map 1
#define TGA_RGB 2
#define TGA_Mono 3
#define TGA_RLEMap 9
#define TGA_RLERGB 10
#define TGA_RLEMono 11
#define TGA_CompMap 32
#define TGA_CompMap4 33

/* Definitions for interleave flag. */
#define TGA_IL_None 0
#define TGA_IL_Two 1
#define TGA_IL_Four 2


typedef struct
{
	unsigned char b, g, r;
} pixel;

typedef struct _AMemoryStream
{
	unsigned char *data;
	long dataLength;
	long currentPtr;
} AMemoryStream;


#ifndef pm_message
#define pm_message printf
#define pm_error printf
#endif



#define PPM_ASSIGN(p,r,g,b) do { p[0]=b;p[1]=g;p[2]=r;} while(0)
#define PPM_EQUAL(p,q) ( (p).r == (q).r && (p).g == (q).g && (p).b == (q).b )

#define MAXCOLORS 16384

static int mapped, rlencoded;

static pixel ColorMap[MAXCOLORS];
static int RLE_count = 0, RLE_flag = 0;

static void readtga ( AMemoryStream * ifp, struct ImageHeader* tgaP );
static void get_map_entry ( AMemoryStream * ifp, char * Value, int Size );
static void get_pixel ( AMemoryStream* ifp, unsigned char* dest, int Size );
static unsigned char getbyte ( AMemoryStream* ifp );






static AMemoryStream *
CreateMemoryStream(const char *filename)
{
	BFile *ifp = new BFile;
	record_ref ifp_ref;
	long error = get_ref_for_path(filename, &ifp_ref);
	ifp->SetRef(ifp_ref);
	error = ifp->Open(B_READ_ONLY);	

	AMemoryStream *newStream;
	
	newStream = new AMemoryStream;
	newStream->currentPtr = 0;
	newStream->data = new unsigned char [ifp->Size()];
	newStream->dataLength = ifp->Size();
	ifp->Read(newStream->data, ifp->Size());
	ifp->Close();
	
		
	return newStream;
}

static void
DestroyMemoryStream(AMemoryStream *aStream)
{
	delete [] aStream->data;
	delete aStream;
}

static GfxImage *	
TargaCreateImage(const char *filename)
{
	// Read a targa file
    struct ImageHeader tga_head;
	int rows = 0;
	int cols = 0;
	int maxval = 0;
    int i;
    unsigned int temp1, temp2;
    int argn, debug, row, col, realrow, truerow, baserow;
	
	AMemoryStream *ifp = CreateMemoryStream(filename);
	
    /* Read the Targa file header. */
    readtga( ifp, &tga_head );

#ifdef DEBUG
#define pm_message printf
	{
	pm_message( "IDLength = %d\n", (int) tga_head.IDLength );
	pm_message( "CoMapType = %d\n", (int) tga_head.CoMapType );
	pm_message( "ImgType = %d\n", (int) tga_head.ImgType );
	pm_message( "Index_lo = %d\n", (int) tga_head.Index_lo );
	pm_message( "Index_hi = %d\n", (int) tga_head.Index_hi );
	pm_message( "Length_lo = %d\n", (int) tga_head.Length_lo );
	pm_message( "Length_hi = %d\n", (int) tga_head.Length_hi );
	pm_message( "CoSize = %d\n", (int) tga_head.CoSize );
	pm_message( "X_org_lo = %d\n", (int) tga_head.X_org_lo );
	pm_message( "X_org_hi = %d\n", (int) tga_head.X_org_hi );
	pm_message( "Y_org_lo = %d\n", (int) tga_head.Y_org_lo );
	pm_message( "Y_org_hi = %d\n", (int) tga_head.Y_org_hi );
	pm_message( "Width_lo = %d\n", (int) tga_head.Width_lo );
	pm_message( "Width_hi = %d\n", (int) tga_head.Width_hi );
	pm_message( "Height_lo = %d\n", (int) tga_head.Height_lo );
	pm_message( "Height_hi = %d\n", (int) tga_head.Height_hi );
	pm_message( "PixelSize = %d\n", (int) tga_head.PixelSize );
	pm_message( "AttBits = %d\n", (int) tga_head.AttBits );
	pm_message( "Rsrvd = %d\n", (int) tga_head.Rsrvd );
	pm_message( "OrgBit = %d\n", (int) tga_head.OrgBit );
	pm_message( "IntrLve = %d\n", (int) tga_head.IntrLve );
	}
#endif
	rows = ( (int) tga_head.Height_lo ) + ( (int) tga_head.Height_hi ) * 256;
	cols = ( (int) tga_head.Width_lo ) + ( (int) tga_head.Width_hi ) * 256;

	switch ( tga_head.ImgType )
	{
		case TGA_Map:
		case TGA_RGB:
		case TGA_Mono:
		case TGA_RLEMap:
		case TGA_RLERGB:
		case TGA_RLEMono:
		break;

		default:
		pm_error( "unknown Targa image type %d", tga_head.ImgType );
	}

	if ( tga_head.ImgType == TGA_Map ||
	 tga_head.ImgType == TGA_RLEMap ||
	 tga_head.ImgType == TGA_CompMap ||
	 tga_head.ImgType == TGA_CompMap4 )
	{ 
		/* Color-mapped image */
		if ( tga_head.CoMapType != 1 )
		pm_error("mapped image (type %d) with color map type != 1",
			tga_head.ImgType );
		mapped = 1;
		
		/* Figure maxval from CoSize. */
		switch ( tga_head.CoSize )
		{
			case 8:
			case 24:
			case 32:
				maxval = 255;
			break;

			case 15:
			case 16:
				maxval = 31;
			break;

			default:
				pm_error("unknown colormap pixel size - %d", tga_head.CoSize );
		}
	} else
	{ 
		/* Not colormap, so figure maxval from PixelSize. */
		mapped = 0;
		switch ( tga_head.PixelSize )
		{
			case 8:
			case 24:
			case 32:
				maxval = 255;
			break;

			case 15:
			case 16:
				maxval = 31;
			break;

			default:
				pm_error( "unknown pixel size - %d", tga_head.PixelSize );
		}
	}

    /* If required, read the color map information. */
	if ( tga_head.CoMapType != 0 )
	{
		temp1 = tga_head.Index_lo + tga_head.Index_hi * 256;
		temp2 = tga_head.Length_lo + tga_head.Length_hi * 256;
	
		if ( ( temp1 + temp2 + 1 ) >= MAXCOLORS )
			pm_error( "too many colors - %d", ( temp1 + temp2 + 1 ) );
		
		for ( i = temp1; i < ( temp1 + temp2 ); ++i )
			get_map_entry( ifp, (char *)&ColorMap[i], (int) tga_head.CoSize );
	}

	/* Check run-length encoding. */
	if ( tga_head.ImgType == TGA_RLEMap ||
	 tga_head.ImgType == TGA_RLERGB ||
	 tga_head.ImgType == TGA_RLEMono )
		rlencoded = 1;
    else
		rlencoded = 0;

	GfxImage *newImage = (GfxImage *)malloc(sizeof(GfxImage));
	// Assign values to image
	newImage->XOffset = 0;
	newImage->YOffset = 0;
	newImage->width = cols;
	newImage->height = rows;
	newImage->type = B_RGB_32_BIT;
	newImage->bytes_per_row = cols * 4;
	newImage->data = (unsigned char *)malloc(cols*rows*sizeof(ulong));

	unsigned char *pixels = newImage->data;
	truerow = 0;
	baserow = 0;
	for ( row = 0; row < rows; ++row )
	{
		realrow = truerow;
		if ( tga_head.OrgBit == 0 )
			realrow = rows - realrow - 1;

		for ( col = 0; col < cols; ++col )
			get_pixel( ifp, &(pixels[realrow*newImage->bytes_per_row+col*4]), (int) tga_head.PixelSize );
		
		if ( tga_head.IntrLve == TGA_IL_Four )
			truerow += 4;
		else if ( tga_head.IntrLve == TGA_IL_Two )
			truerow += 2;
		else
			++truerow;
		if ( truerow >= rows )
			truerow = ++baserow;
	}

	DestroyMemoryStream(ifp);

	return newImage;
}



static unsigned char
getbyte(AMemoryStream *ifp )
{
	unsigned char c = 0;

	if (ifp->currentPtr >= ifp->dataLength)
		pm_error( "EOF / read error" );

	c = ifp->data[ifp->currentPtr];
	ifp->currentPtr++;

	return c;
}

//======================================================================
//
//======================================================================
static void
readtga(AMemoryStream *ifp, struct ImageHeader* tgaP )
{
    unsigned char flags;

    tgaP->IDLength = getbyte( ifp );
    tgaP->CoMapType = getbyte( ifp );
    tgaP->ImgType = getbyte( ifp );
    tgaP->Index_lo = getbyte( ifp );
    tgaP->Index_hi = getbyte( ifp );
    tgaP->Length_lo = getbyte( ifp );
    tgaP->Length_hi = getbyte( ifp );
    tgaP->CoSize = getbyte( ifp );
    tgaP->X_org_lo = getbyte( ifp );
    tgaP->X_org_hi = getbyte( ifp );
    tgaP->Y_org_lo = getbyte( ifp );
    tgaP->Y_org_hi = getbyte( ifp );
    tgaP->Width_lo = getbyte( ifp );
    tgaP->Width_hi = getbyte( ifp );
    tgaP->Height_lo = getbyte( ifp );
    tgaP->Height_hi = getbyte( ifp );
    tgaP->PixelSize = getbyte( ifp );
    flags = getbyte( ifp );
    tgaP->AttBits = flags & 0xf;
    tgaP->Rsrvd = ( flags & 0x10 ) >> 4;
    tgaP->OrgBit = ( flags & 0x20 ) >> 5;
    tgaP->IntrLve = ( flags & 0xc0 ) >> 6;

    if ( tgaP->IDLength != 0 )
	{
		ifp->currentPtr += tgaP->IDLength;
	}
}

static void
get_map_entry( AMemoryStream *ifp, char *Value, int Size )
{
	unsigned char j, k, r, g, b;

	/* Read appropriate number of bytes, break into rgb & put in map. */
	switch ( Size )
	{
		case 8:				/* Grey scale, read and triplicate. */
			r = g = b = getbyte( ifp );
		break;

		case 16:			/* 5 bits each of red green and blue. */
		case 15:			/* Watch for byte order. */
			j = getbyte( ifp );
			k = getbyte( ifp );
			r = ( k & 0x7C ) >> 2;
			g = ( ( k & 0x03 ) << 3 ) + ( ( j & 0xE0 ) >> 5 );
			b = j & 0x1F;
		break;

		case 32:
		case 24:			/* 8 bits each of blue green and red. */
			b = getbyte( ifp );
			g = getbyte( ifp );
			r = getbyte( ifp );
			if ( Size == 32 )
				getbyte( ifp );	/* Read alpha byte & throw away. */
		break;

		default:
			pm_error( "unknown colormap pixel size (#2) - %d", Size );
	}
    PPM_ASSIGN( Value, r, g, b );
}

static void
get_pixel( AMemoryStream *ifp, unsigned char *dest, int Size )
{
	static unsigned char Red, Grn, Blu;
	unsigned char j, k;
	static unsigned int l;

	/* Check if run length encoded. */
	if ( rlencoded )
	{
		if ( RLE_count == 0 )
		{ /* Have to restart run. */
			unsigned char i;
			i = getbyte( ifp );
			RLE_flag = ( i & 0x80 );
			if ( RLE_flag == 0 )
				/* Stream of unencoded pixels. */
				RLE_count = i + 1;
			else
				/* Single pixel replicated. */
				RLE_count = i - 127;
			/* Decrement count & get pixel. */
			--RLE_count;
	    } else
	    { /* Have already read count & (at least) first pixel. */
			--RLE_count;
			if ( RLE_flag != 0 )
			/* Replicated pixels. */
				goto PixEncode;
	    }
	}
	
	/* Read appropriate number of bytes, break into RGB. */
	switch ( Size )
	{
		case 8:				/* Grey scale, read and triplicate. */
			Red = Grn = Blu = l = getbyte( ifp );
		break;

		case 16:			/* 5 bits each of red green and blue. */
		case 15:			/* Watch byte order. */
			j = getbyte( ifp );
			k = getbyte( ifp );
			l = ( (unsigned int) k << 8 ) + j;
			Red = ( k & 0x7C ) >> 2;
			Grn = ( ( k & 0x03 ) << 3 ) + ( ( j & 0xE0 ) >> 5 );
			Blu = j & 0x1F;
		break;

		case 32:
			dest[0] = getbyte(ifp);
			dest[1] = getbyte(ifp);
			dest[2] = getbyte(ifp);
			dest[3] = getbyte(ifp);
			return;
		break;
		case 24:			/* 8 bits each of blue green and red. */
			dest[0] = getbyte(ifp);
			dest[1] = getbyte(ifp);
			dest[2] = getbyte(ifp);
			return ;
		break;

		default:
			pm_error( "unknown pixel size (#2) - %d", Size );
	}

PixEncode:
    if ( mapped )
	{	
    	PPM_ASSIGN( dest, ColorMap[l].r, ColorMap[l].g, ColorMap[l].b);
	}else
		PPM_ASSIGN( dest, Red, Grn, Blu );
}




#pragma export on
char *rrasaddon_IDName();
char *rrasaddon_IDAuthor();
char *rrasaddon_IDNotice();
char *rrasaddon_IDEncoder();
char *rrasaddon_IDDecoder();
float CanCreateImage(void *bytes, long byteLen);
GfxImage *CreateImage(BFile *file);
GfxImage *CreateImage(char *file);
#pragma export off

char *IDName = "TARGA Codec";
char *IDAuthor = "William Adams";
char *IDNotice = "Copyright Be Inc. 1996, All Rights reserved.";
char *IDEncoder = "IDtga";
char *IDDecoder = "IDtga";


char *rrasaddon_IDName()
{
	return IDName;
}

char *rrasaddon_IDAuthor()
{
	return IDAuthor;
}

char *rrasaddon_IDNotice()
{
	return IDNotice;
}

char *rrasaddon_IDEncoder()
{
	return IDEncoder;
}

char *rrasaddon_IDDecoder()
{
	return IDDecoder;
}


//====================================================
// Function: CanReadImage
// 
// Returns a float value representing the degree to
// which this module is capable of reading the image.
// The closer it returns to 1.0, the more confident
// it is that the image can be read successfully.
//
// If the image can be read at all, then a value of 0.8
// should be returned.  If it can't be read at all, then
// a value of 0.0 should be returned.
//====================================================

float
CanCreateImage(void *data, long dataLen)
{
	if (dataLen < 8)
		return 0.0;
		
	unsigned char *dataPtr = (unsigned char *)data;
	
	if (dataPtr[0]==0 && dataPtr[1]==0 &&
	   dataPtr[2]==2 && dataPtr[3]==0 &&
	   dataPtr[4]==0 && dataPtr[5]==0 &&
	   dataPtr[6]==0 && dataPtr[7]==0)
		return 1.0;

	return 0.0;
}


//====================================================
// Function: CreateImage
//
// Create a GfxImage based on a file of GIF data
//====================================================

GfxImage *
CreateImage(BFile *fp)
{
	char pathname[1024]={'\0'};
	char fname[1024];
	
	fp->GetPath(pathname,1023);
	printf("%s - Creating: %s\n", IDName, pathname);
	 
	return CreateImage(pathname);
}


GfxImage *
CreateImage(char *fname)
{
	GfxImage *tmpImage = TargaCreateImage(fname);
	
	return tmpImage;
}

#pragma export off
