
#include <string.h>
#include <stdio.h>
#include "rraster.h"


/*
 * comments on error handling:
 * a truncated file is not considered a Major Error.  The file is loaded, the
 * rest of the pic is filled with 0's.
 *
 * a file with garbage characters in it is an unloadable file.  All allocated
 * stuff is tossed, and LoadPBM returns non-zero
 *
 * not being able to malloc is a Fatal Error.  The program is aborted.
 */


#define TRUNCSTR "File appears to be truncated."

static int garbage;
static long numgot, filesize;

static int loadpbm  (FILE *, GfxImage *, int);
static int loadpgm  (FILE *, GfxImage *, int, int);
static int loadppm  (FILE *, GfxImage *, int, int);
static int getint   (FILE *, GfxImage *);
static int getbit   (FILE *, GfxImage *);
static int getshort (FILE *);
static int pbmError (char *, char *);

static char *bname="some.pbm";

/*******************************************/
static int 
pbmError(char *fname, char *st)
{
  fprintf(stderr, "%s:  %s", fname, st);
  return 0;
}

static int 
LoadPBM(char *fname, GfxImage *img)
{
  /* returns '1' on success */

  FILE  *fp;
  int    c, c1;
  int    maxv, rv;

  garbage = maxv = rv = 0;
  //bname = imglib_basename(fname);

	img->XOffset = 0;
	img->YOffset = 0;
	img->data     = (byte *) NULL;
	img->comment = (char *) NULL;


	/* open the file */
	fp = fopen(fname,"r");
	if (!fp) 
		return (pbmError(bname, "can't open file"));

	/* compute file length */
	fseek(fp, 0L, 2);
	filesize = ftell(fp);
	fseek(fp, 0L, 0);


	/* 
	 read the first two bytes of the file to determine which format
     this file is.  "P1" = ascii bitmap, "P2" = ascii greymap,
     "P3" = ascii pixmap, "P4" = raw bitmap, "P5" = raw greymap,
     "P6" = raw pixmap 
	*/

	c = getc(fp);  
	c1 = getc(fp);
	if (c!='P' || c1<'1' || c1>'6') 
		return(pbmError(bname, "unknown format"));

	/* read in header information */
	img->width = getint(fp, img);  
	img->height = getint(fp, img);

	/* if we're not reading a bitmap, read the 'max value' */
	if ( !(c1=='1' || c1=='4')) 
	{
    	maxv = getint(fp, img);
    	if (maxv < 1) 
			garbage=1;    /* to avoid 'div by zero' probs */
	}


	if (garbage) 
	{
    	fclose(fp);
    	if (img->comment) 
			free(img->comment);
    	img->comment = (char *) NULL;
    	return (pbmError(bname, "Garbage characters in header."));
	}


	//if (c1=='1' || c1=='2' || c1=='3') 
	//	img->file_format = F_PBMASCII;
	//else 
	//	img->file_format = F_PBMRAW;

	/* note:  pic, type, r,g,b, frmInfo, shortFrm, and colorType fields of
     picinfo struct are filled in in the format-specific loaders */

	/* call the appropriate subroutine to handle format-specific stuff */
	if (c1=='1' || c1=='4') 
		rv = loadpbm(fp, img, c1=='4' ? 1 : 0);
  	else if (c1=='2' || c1=='5') 
		rv = loadpgm(fp, img, c1=='5' ? 1 : 0, maxv);
	else if (c1=='3' || c1=='6') 
		rv = loadppm(fp, img, c1=='6' ? 1 : 0, maxv);

	fclose(fp);

	if (!rv) 
	{
		if (img->data) 
			free(img->data);
		if (img->comment) 
			free(img->comment);
		img->data     = (byte *) NULL;
		img->comment = (char *) NULL;
	}

	return rv;
}  



/*******************************************/
static int 
loadpbm(FILE *fp, GfxImage *img, int raw)
{
  byte *pic8;
  byte *pix;
  int   i,j,bit,w,h;

	w = img->width;  
	h = img->height;
	pic8 = (byte *) calloc((size_t) w * h, (size_t) 1);
	if (!pic8) 
		return pbmError(bname, "couldn't malloc 'pic8'");

	img->data  = pic8;
	img->type = B_COLOR_8_BIT;
	sprintf(img->full_info, "PBM, %s format.  (%ld bytes)", 
	  (raw) ? "raw" : "ascii", filesize);
  sprintf(img->short_info, "%dx%d PBM.", w, h);
  img->color_format = F_BWDITHER;


  /* B/W bitmaps have a two entry colormap */
  img->palette[0].red = img->palette[0].green = img->palette[0].blue = 255;   /* entry #0 = white */
  img->palette[1].red = img->palette[1].green = img->palette[1].blue = 0;     /* entry #1 = black */


  if (!raw) {
    numgot = 0;
    for (i=0, pix=pic8; i<h; i++) {
      for (j=0; j<w; j++, pix++) *pix = getbit(fp, img);
    }

    if (numgot != w*h) pbmError(bname, TRUNCSTR);
    if (garbage) {
      return(pbmError(bname, "Garbage characters in image data."));
    }
  }


  else {   /* read raw bits */
    int trunc = 0, k = 0;

    for (i=0, pix=pic8; i<h; i++) {
      for (j=0,bit=0; j<w; j++, pix++, bit++) {

	bit &= 7;
	if (!bit) {
	  k = getc(fp);
	  if (k==EOF) { trunc=1; k=0; }
	}

	*pix = (k&0x80) ? 1 : 0;
	k = k << 1;
      }
    }

    if (trunc) pbmError(bname, TRUNCSTR);
  }

  return 1;
}


/*******************************************/
static int loadpgm(FILE *fp, GfxImage *img, int raw, int maxv)
{
  byte *pix, *pic8;
  int   i,j,bitshift,w,h, holdmaxv;


  w = img->width;  h = img->height;
  pic8 = (byte *) calloc((size_t) w*h, (size_t) 1);
  if (!pic8) return(pbmError(bname, "couldn't malloc 'pic8'"));


  img->data  = pic8;
  img->type = B_COLOR_8_BIT;
  sprintf(img->full_info, "PGM, %s format.  (%ld bytes)", 
	  (raw) ? "raw" : "ascii", filesize);
  sprintf(img->short_info, "%dx%d PGM.", img->width, img->height);
  img->color_format = F_GREYSCALE;


  /* if maxv>255, keep dropping bits until it's reasonable */
  holdmaxv = maxv;
  bitshift = 0;
  while (maxv>255) { maxv = maxv>>1;  bitshift++; }

  /* fill in a greyscale colormap where maxv maps to 255 */
  for (i=0; i<=maxv; i++)
    img->palette[i].red = img->palette[i].green = img->palette[i].blue = (i*255)/maxv;


  numgot = 0;

  if (!raw) {
    for (i=0, pix=pic8; i<h; i++) {
      for (j=0; j<w; j++, pix++)
	*pix = (byte) (getint(fp, img) >> bitshift);
    }
  }
  else { /* raw */
    if (holdmaxv>255) {
      for (i=0, pix=pic8; i<h; i++) {
	for (j=0; j<w; j++, pix++)
	  *pix = (byte) (getshort(fp) >> bitshift);
      }
    }
    else {
      numgot = fread(pic8, (size_t) 1, (size_t) w*h, fp);  /* read raw data */
    }
  }

  if (numgot != w*h) pbmError(bname, TRUNCSTR);   /* warning only */

  if (garbage) {
    return (pbmError(bname, "Garbage characters in image data."));
  }

  return 1;
}


/*******************************************/
static int loadppm(FILE *fp, GfxImage *img, int raw, int maxv)
{
	byte *pix, *pic24, scale[256], *pic8;
	int   i,j,bitshift, w, h, holdmaxv;

	w = img->width;  
	h = img->height;

	/* allocate 24-bit image */
	pic24 = (byte *) calloc((size_t) w*h*3, (size_t) 1);
	if (!pic24) 
		return pbmError(bname,"couldn't malloc 'pic24'");

	img->data  = (byte *) calloc((size_t) w*h*4, (size_t) 1);
	img->bytes_per_row = w*4;
	img->type = B_RGB_32_BIT;
	
	sprintf(img->full_info, "PPM, %s format.  (%ld bytes)", 
	  (raw) ? "raw" : "ascii", filesize);
	sprintf(img->short_info, "%dx%d PPM.", w, h);
	img->color_format = B_RGB_32_BIT;


	/* if maxv>255, keep dropping bits until it's reasonable */
	holdmaxv = maxv;
	bitshift = 0;
	while (maxv>255) 
	{
		maxv = maxv>>1;  
		bitshift++;
	}

	numgot = 0;

	if (!raw) 
	{
		for (i=0, pix=pic24; i<h; i++) 
		{
			for (j=0; j<w*3; j++, pix++)
			{
				*pix = (byte) (getint(fp, img) >> bitshift);
			}
		}
	} else 
	{ /* raw */
		if (holdmaxv>255) 
		{
			for (i=0, pix=pic24; i<h; i++) 
			{
				for (j=0; j<w*3; j++,pix++)
	  				*pix = (byte) (getshort(fp) >> bitshift);
      		}
		} else 
		{
			numgot = fread(pic24, (size_t) 1, (size_t) w*h*3, fp);  /* read data */
		}
	}
  
	if (numgot != w*h*3) 
		pbmError(bname, TRUNCSTR);

	if (garbage)
		return(pbmError(bname, "Garbage characters in image data."));


	/* have to scale all RGB values up (Conv24to8 expects RGB values to
     range from 0-255 */

	if (maxv<255) 
	{ 
		for (i=0; i<=maxv; i++) 
			scale[i] = (i * 255) / maxv;

		for (i=0, pix=pic24; i<h; i++) 
		{
			for (j=0; j<w*3; j++, pix++) 
			{
				*pix = scale[*pix];
			}
		}
	}

	// Copy over to image buffer
	long w3 = w*3;
	for (i=0; i<h; i++) 
	{
		for (j=0; j<w; j++) 
		{
			int j4 = j*4;
			int j3 = j*3;
			long dstoffset = i*w3;
			img->data[i*img->bytes_per_row+j4] = pic24[dstoffset+j3+2];
			img->data[i*img->bytes_per_row+j4+1] = pic24[dstoffset+j3+1];
			img->data[i*img->bytes_per_row+j4+2] = pic24[dstoffset+j3];
			img->data[i*img->bytes_per_row+j4+3] = 255;			
		}
	}
	
	return 1;
}



/*******************************************/
static int getint(FILE *fp, GfxImage *img)
{
  int c, i, firstchar;

  /* note:  if it sees a '#' character, all characters from there to end of
     line are appended to the comment string */

  /* skip forward to start of next number */
  c = getc(fp);
  while (1) {
    /* eat comments */
    if (c=='#') {   /* if we're at a comment, read to end of line */
      char cmt[256], *sp, *tmpptr;

      sp = cmt;  firstchar = 1;
      while (1) {
	c=getc(fp);
	if (firstchar && c == ' ') firstchar = 0;  /* lop off 1 sp after # */
	else {
	  if (c == '\n' || c == EOF) break;
	  if ((sp-cmt)<250) *sp++ = c;
	}
      }
      *sp++ = '\n';
      *sp   = '\0';

      if (strlen(cmt) > 0) {    /* add to img->comment */
	if (!img->comment) {
	  img->comment = (char *) malloc(strlen(cmt)+1);
	  if (!img->comment) 
		return pbmError(bname, "malloc failure in xvpbm.c getint");
	  img->comment[0] = '\0';
	}
	else {
	  tmpptr = (char *) realloc(img->comment, 
		      strlen(img->comment) + strlen(cmt) + 1); 
	  if (!tmpptr) 
		pbmError(bname,"realloc failure in xvpbm.c getint");
	  img->comment = tmpptr;
	}
	strcat(img->comment, cmt);
      }
    }

    if (c==EOF) return 0;
    if (c>='0' && c<='9') break;   /* we've found what we were looking for */

    /* see if we are getting garbage (non-whitespace) */
    if (c!=' ' && c!='\t' && c!='\r' && c!='\n' && c!=',') garbage=1;

    c = getc(fp);
  }


  /* we're at the start of a number, continue until we hit a non-number */
  i = 0;
  while (1) {
    i = (i*10) + (c - '0');
    c = getc(fp);
    if (c==EOF) return i;
    if (c<'0' || c>'9') break;
  }

  numgot++;
  return i;
}



/*******************************************/
static int getshort(FILE *fp)
{
  /* used in RAW mode to read 16-bit values */

  int c1, c2;

  c1 = getc(fp);
  if (c1 == EOF) return 0;
  c2 = getc(fp);
  if (c2 == EOF) return 0;

  numgot++;

  return (c2 << 8) | c1;
}



/*******************************************/
static int getbit(FILE *fp, GfxImage *img)
{
  int c;

  /* skip forward to start of next number */
  c = getc(fp);
  while (1) {
    /* eat comments */
    if (c=='#') {   /* if we're at a comment, read to end of line */
      char cmt[256], *sp, *tmpptr;

      sp = cmt;
      while (1) {
	c=getc(fp);
	if (c == '\n' || c == EOF) break;

	if ((sp-cmt)<250) *sp++ = c;
      }
      *sp++ = '\n';
      *sp = '\0';

      if (strlen(cmt) > 0) {    /* add to img->comment */
	if (!img->comment) {
	  img->comment = (char *) malloc(strlen(cmt)+1);
	  if (!img->comment) 
		return pbmError(bname,"malloc failure in xvpbm.c getint");
	  img->comment[0] = '\0';
	}
	else {
	  tmpptr = (char *) realloc(img->comment, 
		      strlen(img->comment) + strlen(cmt) + 1); 
	  if (!tmpptr) 
		return pbmError(bname,"realloc failure in xvpbm.c getint");
	  img->comment = tmpptr;
	}
	strcat(img->comment, cmt);
      }
    }
    if (c==EOF) return 0;
    if (c=='0' || c=='1') break;   /* we've found what we were looking for */

    /* see if we are getting garbage (non-whitespace) */
    if (c!=' ' && c!='\t' && c!='\r' && c!='\n' && c!=',') garbage=1;

    c = getc(fp);
  }


  numgot++;
  return(c-'0');
}







/*******************************************/
static int 
WritePBM(FILE *fp, byte *pic, int ptype,
	int w, int h, byte *rmap, byte *gmap, byte *bmap,
	int num_colors, int colorstyle, int raw,
	char *comment)
{
  /* writes a PBM/PGM/PPM file to the already open stream
     if (raw), writes as RAW bytes, otherwise writes as ASCII 
     'colorstyle' single-handedly determines the type of file written
     if colorstyle==0, (Full Color) a PPM file is written
     if colorstyle==1, (Greyscale)  a PGM file is written
     if colorstyle==2, (B/W stipple) a PBM file is written */

  int   magic;
  byte *pix;
  int   i,j,len;

  /* calc the appropriate magic number for this file type */
  magic = 0;
  if      (colorstyle==0) magic = 3;
  else if (colorstyle==1) magic = 2;
  else if (colorstyle==2) magic = 1;

  if (raw && magic) magic+=3;


  /* write the header info */
  fprintf(fp,"P%d\n",magic);
  fprintf(fp,"# CREATOR: BeOS Imglib %s\n", "11/7/96");

  if (comment) {      /* write comment lines */
    char *sp;

    sp = comment;
    while (*sp) {
      fprintf(fp, "# ");
      while (*sp && *sp != '\n') fputc(*sp++, fp);
      if (*sp == '\n') sp++;
      fputc('\n', fp);
    }
  }


  fprintf(fp,"%d %d\n",w,h);
  if (colorstyle!=2) fprintf(fp,"255\n");

  if (ferror(fp)) return -1;

  /* write the image data */

  if (colorstyle==0) {                  /* 24bit RGB, 3 bytes per pixel */
    for (i=0, pix=pic, len=0; i<h; i++) {
      for (j=0; j<w; j++) {
	if (raw) {
	  if (ptype==B_COLOR_8_BIT) {
	    putc(rmap[*pix],fp);  putc(gmap[*pix],fp);  putc(bmap[*pix],fp);
	  }
	  else {  /* B_RGB_32_BIT */
	    putc(pix[0],fp);  putc(pix[1],fp);  putc(pix[2],fp);
	  }
	}
	else {
	  if (ptype==B_COLOR_8_BIT) 
	    fprintf(fp,"%3d %3d %3d ",rmap[*pix], gmap[*pix], bmap[*pix]);
	  else
	    fprintf(fp,"%3d %3d %3d ",pix[0], pix[1], pix[2]);

	  len+=12;
	  if (len>58) { fprintf(fp,"\n");  len=0; }
	}
	
	pix += (ptype==B_RGB_32_BIT) ? 3 : 1;
      }
    }
  }


  else if (colorstyle==1) {             /* 8-bit greyscale */
    byte rgb[256];
    if (ptype==B_COLOR_8_BIT)
      for (i=0; i<num_colors; i++) rgb[i] = MONO(rmap[i],gmap[i],bmap[i]);

    for (i=0, pix=pic, len=0; i<w*h; i++) {

      if (raw) putc((ptype==B_COLOR_8_BIT) ? rgb[*pix] : MONO(pix[0],pix[1],pix[2]),fp);

      else {
	if (ptype==B_COLOR_8_BIT) fprintf(fp,"%3d ",rgb[*pix]);
	            else fprintf(fp,"%3d ",MONO(pix[0],pix[1],pix[2]));
	len += 4;
	if (len>66) { fprintf(fp,"\n");  len=0; }
      }

      pix += (ptype==B_RGB_32_BIT) ? 3 : 1;
    }
  }

  else if (colorstyle==2) {             /* 1-bit B/W stipple */
    int bit,k,flipbw;
    char *str0, *str1;

    /* shouldn't happen */
    if (ptype == B_RGB_32_BIT) 
		return pbmError(bname, "B_RGB_32_BIT and B/W Stipple in WritePBM()\n");

    /* if '0' is black, set flipbw */
    flipbw = (MONO(rmap[0],gmap[0],bmap[0]) < MONO(rmap[1],gmap[1],bmap[1]));

    str0 = (flipbw) ? "1 " : "0 ";
    str1 = (flipbw) ? "0 " : "1 ";

    for (i=0, pix=pic, len=0; i<h; i++) {
      for (j=0, bit=0, k=0; j<w; j++, pix++) {
	if (raw) {
	  k = (k << 1) | *pix;
	  bit++;
	  if (bit==8) {
	    if (flipbw) k = ~k;
	    fputc(k,fp);
	    bit = k = 0;
	  }
	}
	else {
	  if (*pix) fprintf(fp,str1);
	       else fprintf(fp,str0);
	  len+=2;
	  if (len>68) { fprintf(fp,"\n"); len=0; }
	}
      } /* j */
      if (raw && bit) {
	k = k << (8-bit);
	if (flipbw) k = ~k;
	fputc(k,fp);
      }
    }
  }

  if (ferror(fp)) return -1;

  return 0;
}


	  
	  





#pragma export on
char *rrasaddon_IDName();
char *rrasaddon_IDAuthor();
char *rrasaddon_IDNotice();
char *rrasaddon_IDEncoder();
char *rrasaddon_IDDecoder();
float CanCreateImage(void *bytes, long byteLen);
GfxImage *CreateImage(BFile *file);
GfxImage *CreateImage(char *file);
#pragma export off

char *IDName = "PBM Codec";
char *IDAuthor = "William Adams";
char *IDNotice = "Copyright Be Inc. 1996, All Rights reserved.";
char *IDEncoder = "IDpbm";
char *IDDecoder = "IDpbm";

char *rrasaddon_IDName()
{
	return IDName;
}

char *rrasaddon_IDAuthor()
{
	return IDAuthor;
}

char *rrasaddon_IDNotice()
{
	return IDNotice;
}

char *rrasaddon_IDEncoder()
{
	return IDEncoder;
}

char *rrasaddon_IDDecoder()
{
	return IDDecoder;
}


//====================================================
// Function: CanReadImage
// 
// Returns a float value representing the degree to
// which this module is capable of reading the image.
// The closer it returns to 1.0, the more confident
// it is that the image can be read successfully.
//
// If the image can be read at all, then a value of 0.8
// should be returned.  If it can't be read at all, then
// a value of 0.0 should be returned.
//====================================================

float
CanCreateImage(void *data, long dataLen)
{
	if (dataLen < 2)
		return 0.0;
	
	unsigned char *dataPtr = (unsigned char *)data;	
	if ((dataPtr[0] == 'P') && (dataPtr[1]>='1') && (dataPtr[1]<='6'))
		return 1.0;

	return 0.0;
}



GfxImage *
CreateImage(BFile *fp)
{
	char pathname[1024]={'\0'};
	
	fp->GetPath(pathname,1023);
	printf("%s - Creating: %s\n", IDName, pathname);
	 
	return CreateImage(pathname);
}


GfxImage *
CreateImage(char *fname)
{
	GfxImage *newImage = (GfxImage *)malloc(sizeof(GfxImage));
	
	if (LoadPBM(fname, newImage))
		return newImage;
	else
		free(newImage);
		
	return 0;
}

