
#include <string.h>
#include <stdio.h>
#include "rraster.h"


/*
 * File Format:
 *   (format identifier:  "#define" as first couple chars in file)
 *
 * looks for first line beginning with '#define'
 *   reads "#define identifier width"  (identifier is ignored)
 * looks for next line beginning with '#define'
 *   reads "#define identifier height" (identifier is ignored)
 * looks for next occurence of characters '0x'
 *   read next two chars as two hex digits
 *   move forward to next occurence of '0x'
 *   repeat
 */
 

static int xbmError (char *, char *);


static int 
LoadXBM(char *fname, GfxImage *img)
{
  /* returns '1' on success */

  FILE  *fp;
  int    c, c1;
  int    i, j, k, bit, w, h;
  byte  *pix, *pic8;
  long   filesize;
  char   line[256], name[256];
  byte   hex[256];
  char  *bname = fname;

	k = 0;

	fp = fopen(fname,"r");
	if (!fp) 
		return (xbmError(bname, "couldn't open file"));

	fseek(fp, 0L, 2);
	filesize = ftell(fp);
	fseek(fp, 0L, 0);


	/* read width:  skip lines until we hit a #define */
	while (1) 
	{
		if (!fgets(line,256,fp)) 
			return(xbmError(bname, "EOF reached in header info."));

		if (strncmp(line,"#define",    (size_t) 7)==0 &&
			sscanf(line,"#define %s %d", name, &w)==2 &&
			strstr(name, "_width") != NULL) break;
	}


	/* read height:  skip lines until we hit another #define */
	while (1) {
    if (!fgets(line,256,fp)) 
      return(xbmError(bname, "EOF reached in header info."));
    
    if (strncmp(line,"#define",    (size_t) 7)==0 &&
		sscanf(line,"#define %s %d", name, &h)==2 &&
		strstr(name, "_height") != NULL) 
		break;
	}



	/* scan forward until we see the first '0x' */
	c = getc(fp);  
	c1 = getc(fp);
	while (c1!=EOF && !(c=='0' && c1=='x') ) 
	{ 
		c = c1;  
		c1 = getc(fp); 
	}

	if (c1==EOF) 
    	return(xbmError(bname, "No bitmap data found"));

	if (w<1 || h<1 || w>10000 || h>10000) 
		return(xbmError(bname, "not an XBM file"));
  
	pic8 = (byte *) calloc((size_t) w*h, (size_t) 1);
	if (!pic8) 
		return(xbmError(bname, "couldn't malloc 'pic8'"));

	/* load up the img structure */
	img->data = pic8;
	img->XOffset = 0;
	img->YOffset = 0;
	img->width = w;     
	img->height = h;
	img->bytes_per_row = w;
	img->type = B_COLOR_8_BIT;
	img->file_format = F_XBM;
	img->color_format = F_BWDITHER;
	sprintf(img->full_info, "X11 Bitmap  (%ld bytes)", filesize);
	sprintf(img->short_info, "%dx%d X11 Bitmap.",w,h);
	img->comment = (char *) NULL;

	/* B/W bitmaps have a two entry colormap */
	img->palette[0].red = img->palette[0].green = img->palette[0].blue = 255;     /* 0 = white */
	img->palette[1].red = img->palette[1].green = img->palette[1].blue = 0;       /* 1 = black */


	/* initialize the 'hex' array for zippy ASCII-hex -> int conversion */

	for (i=0; i<256; i++) 
		hex[i] = 255;   /* flag 'undefined' chars */
	for (i='0'; i<='9'; i++) 
		hex[i] = i - '0';
	for (i='a'; i<='f'; i++) 
		hex[i] = i + 10 - 'a';
	for (i='A'; i<='F'; i++) 
		hex[i] = i + 10 - 'A';

	/* read/convert the image data */

	for (i=0, pix=pic8; i<h; i++)
	{
		for (j=0,bit=0; j<w; j++, pix++, bit = (++bit)&7) 
		{
			if (!bit) 
			{
				/* get next byte from file.  we're already positioned at it */
				c = getc(fp);  
				c1 = getc(fp);
				if (c<0 || c1<0) 
				{ 
					/* EOF: break out of loop */	  
					c=c1='0'; 
					i=h; 
					j=w;
					xbmError(bname, "The file would appear to be truncated.");
				}

				if (hex[c1] == 255) 
				{  
					if (hex[c] == 255) 
						k = 0;   /* no digits after the '0x' ... */
					else 
						k = hex[c];
				} else 
					k = (hex[c] << 4) + hex[c1];

				/* advance to next '0x' */
				c = getc(fp);  
				c1 = getc(fp);
				while (c1!=EOF && !(c=='0' && c1=='x') ) 
				{ 
					c = c1;  
					c1 = getc(fp); 
				}
			}

			*pix = (k&1) ? 1 : 0;
			
				
			k = k >> 1;
		}
	}

	fclose(fp);

	return 1;
}  



/*******************************************/
static int xbmError(char *fname, char *st)
{
  printf("%s:  %s", fname, st);
  return 0;
}


#if 0
/*******************************************/
int WriteXBM(FILE *fp, byte *pic, int w, int h, 
	byte *rmap, byte *gmap, byte *bmap, char *fname)
{
  /* pic is expected to be an array of w*h bytes, each of which is either
     '0' or '1'.
     The 'darker' of {rmap,gmap,bmap}[0] and {rmap,gmap,bmap}[1] is 
     considered black, and the other one, white.
     Some sort of stippling algorithm should've
     been called already to produce pic, otherwise the output won't be at all
     useful */

  int   i,j,k,bit,len,nbytes,flipbw;
  byte *pix;
  char name[256], *foo;

  //foo = imglib_basename(fname);
  strcpy(name, foo);

  foo = (char *) index(name,'.');
  if (foo) *foo='\0';                 /* truncated name at first '.' */

  fprintf(fp,"#define %s_width %d\n",name,w);  
  fprintf(fp,"#define %s_height %d\n",name,h);
  fprintf(fp,"static char %s_bits[] = {\n",name);

  fprintf(fp," ");

  /* set flipbw if color#0 is black */
  flipbw = (MONO(rmap[0],gmap[0],bmap[0]) < MONO(rmap[1],gmap[1],bmap[1]));

  nbytes = h * ((w+7)/8);   /* # of bytes to write */

  for (i=0, len=1, pix=pic; i<h; i++) {
    for (j=bit=k=0; j<w; j++,pix++) {
      k = (k>>1);
      if (*pix) k |= 0x80;
      bit++;
      if (bit==8) {
	if (flipbw) k = ~k;
	fprintf(fp,"0x%02x",(byte) k&0xff);
	nbytes--;  len += 4;
	if (nbytes) { fprintf(fp,",");  len++; }
	if (len>72) { fprintf(fp,"\n ");  len=1; }
	bit = k = 0;
      }
    }

    if (bit) {
      k = k >> (8-bit);
      if (flipbw) k = ~k;
      fprintf(fp,"0x%02x",(byte) k&0xff);
      nbytes--;  len += 4;
      if (nbytes) { fprintf(fp,",");  len++; }
      if (len>72) { fprintf(fp,"\n ");  len=1; }
    }
  }

  fprintf(fp,"};\n");

  if (ferror(fp)) return -1;
  return 0;
}

#endif







#pragma export on
char *rrasaddon_IDName();
char *rrasaddon_IDAuthor();
char *rrasaddon_IDNotice();
char *rrasaddon_IDEncoder();
char *rrasaddon_IDDecoder();
float CanCreateImage(void *bytes, long byteLen);
GfxImage *CreateImage(BFile *file);
GfxImage *CreateImage(char *file);
#pragma export off

char *IDName = "XBM Codec";
char *IDAuthor = "William Adams";
char *IDNotice = "Copyright Be Inc. 1996, All Rights reserved.";
char *IDEncoder = "IDxbm";
char *IDDecoder = "IDxbm";

char *rrasaddon_IDName()
{
	return IDName;
}

char *rrasaddon_IDAuthor()
{
	return IDAuthor;
}

char *rrasaddon_IDNotice()
{
	return IDNotice;
}

char *rrasaddon_IDEncoder()
{
	return IDEncoder;
}

char *rrasaddon_IDDecoder()
{
	return IDDecoder;
}


//====================================================
// Function: CanReadImage
// 
// Returns a float value representing the degree to
// which this module is capable of reading the image.
// The closer it returns to 1.0, the more confident
// it is that the image can be read successfully.
//
// If the image can be read at all, then a value of 0.8
// should be returned.  If it can't be read at all, then
// a value of 0.0 should be returned.
//====================================================

float
CanCreateImage(void *data, long dataLen)
{
	if (dataLen < 2)
		return 0.0;
	
	char *dataPtr = (char *)data;	
	if (strncmp((char *) dataPtr, "/* XBM */", (size_t) 9)==0)
		return 1.0;

	if (strncmp((char *) dataPtr,"#define", (size_t) 7)==0)
		return 1.0;

	return 0.0;
}



GfxImage *
CreateImage(BFile *fp)
{
	char pathname[1024]={'\0'};
	
	fp->GetPath(pathname,1023);
	printf("%s - Creating: %s\n", IDName, pathname);
	 
	return CreateImage(pathname);
}


GfxImage *
CreateImage(char *fname)
{
	GfxImage *newImage = (GfxImage *)malloc(sizeof(GfxImage));
	
	if (LoadXBM(fname, newImage))
		return newImage;
	else
		free(newImage);
		
	return 0;
}

