/* +-------------------------------------------------------------------+ */
/* | Copyright 1990, 1991, 1993, David Koblas.  (koblas@netcom.com)    | */
/* |   Permission to use, copy, modify, and distribute this software   | */
/* |   and its documentation for any purpose and without fee is hereby | */
/* |   granted, provided that the above copyright notice appear in all | */
/* |   copies and that both that copyright notice and this permission  | */
/* |   notice appear in supporting documentation.  This software is    | */
/* |   provided "as is" without express or implied warranty.           | */
/* +-------------------------------------------------------------------+ */


#include <string.h>
#include <stdio.h>
#include "rraster.h"
#include "png.h"



static int 
LoadPNG(char *fname, GfxImage *img)
{
	/* returns '1' on success */

	FILE  *fp;
	png_structp png_ptr;
	png_infop info_ptr;
	png_infop end_info;
	
	printf("LoadPNG - BEGIN\n");
	/* open the file */
	fp = fopen(fname,"r");
	if (!fp) 
	{
		printf("LoadPNG - couldn't open file: %s\n", fname);
		return 0;
	}
	
    png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING,
       NULL, NULL, NULL);
    if (!png_ptr)
    {
		printf("LoadPNG - couldn't create read struct\n");
	    return 0;
    }

	info_ptr = png_create_info_struct(png_ptr);
    if (!info_ptr)
    {
		printf("LoadPNG - couldn't create info struct\n");
        png_destroy_read_struct(&png_ptr, (png_infopp)NULL, (png_infopp)NULL);
        return 0;
    }
    
	end_info = png_create_info_struct(png_ptr);
    if (!end_info)
    {
		printf("LoadPNG - couldn't create end info struct\n");
        png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
        return 0;
    }


	img->XOffset = 0;
	img->YOffset = 0;
	img->data     = (byte *) NULL;
	img->comment = (char *) NULL;


	// Initialize the io routines
	png_init_io(png_ptr, fp);
	png_read_info(png_ptr, info_ptr);


	/* read in header information */
	img->width = info_ptr->width;  
	img->height = info_ptr->height;
	img->data = (unsigned char *)malloc(img->width * img->height * 4);
	img->type = B_RGB_32_BIT;

   /* expand paletted colors into true RGB triplets */
   if (info_ptr->color_type == PNG_COLOR_TYPE_PALETTE)
      png_set_expand(png_ptr);

   /* expand paletted or RGB images with transparency to full alpha channels
    * so the data will be available as RGBA quartets */
   if (info_ptr->valid & PNG_INFO_tRNS)
      png_set_expand(png_ptr);

   /* tell libpng to strip 16 bit/color files down to 8 bits/color */
   if (info_ptr->bit_depth == 16)
      png_set_strip_16(png_ptr);

   /* flip the rgb pixels to bgr */
   if (info_ptr->color_type == PNG_COLOR_TYPE_RGB ||
      info_ptr->color_type == PNG_COLOR_TYPE_RGB_ALPHA)
      png_set_bgr(png_ptr);

   /* add a filler byte to RGB files (before or after each RGB triplet) */
   if (info_ptr->bit_depth == 8 && info_ptr->color_type == PNG_COLOR_TYPE_RGB)
      png_set_filler(png_ptr, 0xff, PNG_FILLER_AFTER);

	/* allocate the memory to hold the image using the fields
		of png_info. 
	*/

	int channels = 0;
	int rowbytes = 0;
	img->bytes_per_row = rowbytes;
	
	if ((info_ptr->color_type & PNG_COLOR_TYPE_PALETTE)==PNG_COLOR_TYPE_PALETTE)
      channels = 1;
	else
		channels = 3;
	if (info_ptr->color_type & PNG_COLOR_MASK_ALPHA)
		channels++;

	rowbytes = ((info_ptr->width * info_ptr->bit_depth * channels + 7) >> 3);

	
	printf("LoadPNG - channels: %d\n",channels);
	printf("LoadPNG - rowbytes: %d\n",rowbytes);
	
	/* the easiest way to read the image */
	png_bytep *row_pointers = (png_bytep *)malloc(img->height*sizeof(png_bytep*));

	for (int row = 0; row < img->height; row++)
	{
		row_pointers[row] = &img->data[row*rowbytes];
		//(unsigned char *)malloc(info_ptr->rowbytes);
	}

	png_read_image(png_ptr, row_pointers);

	/* read the rest of the file, getting any additional chunks in info_ptr */
	png_read_end(png_ptr, info_ptr);

	/* clean up after the read, and free any memory allocated */
	png_destroy_read_struct(&png_ptr, &info_ptr, &end_info);

	fclose(fp);

	return 1;
}  



#pragma export on
char *rrasaddon_IDName();
char *rrasaddon_IDAuthor();
char *rrasaddon_IDNotice();
char *rrasaddon_IDEncoder();
char *rrasaddon_IDDecoder();
float CanCreateImage(void *bytes, long byteLen);
GfxImage *CreateImage(BFile *file);
GfxImage *CreateImage(char *file);
#pragma export off

char *IDName = "PNG Codec";
char *IDAuthor = "William Adams";
char *IDNotice = "Copyright Be Inc. 1996, All Rights reserved.";
char *IDEncoder = "IDpng";
char *IDDecoder = "IDpng";

char *rrasaddon_IDName()
{
	return IDName;
}

char *rrasaddon_IDAuthor()
{
	return IDAuthor;
}

char *rrasaddon_IDNotice()
{
	return IDNotice;
}

char *rrasaddon_IDEncoder()
{
	return IDEncoder;
}

char *rrasaddon_IDDecoder()
{
	return IDDecoder;
}


//====================================================
// Function: CanReadImage
// 
// Returns a float value representing the degree to
// which this module is capable of reading the image.
// The closer it returns to 1.0, the more confident
// it is that the image can be read successfully.
//
// If the image can be read at all, then a value of 0.8
// should be returned.  If it can't be read at all, then
// a value of 0.0 should be returned.
//====================================================

float
CanCreateImage(void *data, long dataLen)
{
	if (dataLen < 6)
		return 0.0;
		
    int is_png = png_check_sig((unsigned char *)data, dataLen);
    if (is_png)
    {
        return 1.0;
    }

	return 0.0;
}



//====================================================
// Function: CreateImage
//
// Create a GfxImage based on a file of GIF data
//====================================================

GfxImage *
CreateImage(BFile *fp)
{
	char pathname[1024]={'\0'};
	
	fp->GetPath(pathname,1023);
	printf("%s - Creating: %s\n", IDName, pathname);
	 
	return CreateImage(pathname);
}


GfxImage *
CreateImage(char *fname)
{
	GfxImage *newImage = (GfxImage *)malloc(sizeof(GfxImage));
	
	if (LoadPNG(fname, newImage))
		return newImage;
	else
		free(newImage);
		
	return 0;
}

