/*
 * xvpcx.c - load routine for PCX format pictures
 *
 * LoadPCX(fname, img)  -  loads a PCX file
 */


/*
 * the following code has been derived from code written by
 *  Eckhard Rueggeberg  (Eckhard.Rueggeberg@ts.go.dlr.de)
 */



#include <File.h>
#include "rraster.h"

typedef unsigned char byte;

/* offsets into PCX header */
#define PCX_ID      0
#define PCX_VER     1
#define PCX_ENC     2
#define PCX_BPP     3
#define PCX_XMINL   4
#define PCX_XMINH   5
#define PCX_YMINL   6
#define PCX_YMINH   7
#define PCX_XMAXL   8
#define PCX_XMAXH   9
#define PCX_YMAXL   10
#define PCX_YMAXH   11
                          /* hres (12,13) and vres (14,15) not used */
#define PCX_CMAP    16    /* start of 16*3 colormap data */
#define PCX_PLANES  65 
#define PCX_BPRL    66
#define PCX_BPRH    67

#define PCX_MAPSTART 0x0c	/* Start of appended colormap	*/

static int MY_DEBUG = 1;
static int  pcxLoadImage  (BFile *, byte *, byte *, int, int);
static void pcxLoadRaster (BFile *, byte *, int, byte *, int, int);
static int  pcxError      (char *, char *);





/*******************************************/
static int 
pcxError(char *st)
{
	printf("CODECpcx:  %s", st);
	fflush(stdout);
	
	return 0;
}


/*****************************/
static void 
pcxLoadRaster(BFile *fp, byte *image, 
	int depth, byte *hdr, int w, int h)
{
	printf("pcxLoadRaster - 1.0\n");
	/* supported:  8 bits per pixel, 1 plane, or 1 bit per pixel, 1-8 planes */
	int row, bcnt, bperlin, pad;
	int i, j, cnt, mask, plane, pmask;
	byte *oldimage;
	byte aChar;
	int b;
	
	bperlin = hdr[PCX_BPRL] + ((int) hdr[PCX_BPRH]<<8);
	printf("pcxLoadRaster - bperlin: %d\n", bperlin);
	
	if (depth == 1) 
		pad = (bperlin * 8) - w;
	else 
		pad = bperlin - w;

	row = bcnt = 0;

	plane = 0;  
	pmask = 1;  
	oldimage = image;

	printf("pcxLoadRaster - 1.2\n");
	while (fp->Read(&aChar,1) != B_ERROR) 
	{
		b = aChar;
		if ((b & 0xC0) == 0xC0) 
		{   /* have a rep. count */
			cnt = b & 0x3F;
			fp->Read(&aChar,1);
			b = aChar;
			if (b == EOF) 
			{ 
				printf("pcxLoadRaster - returning early.\n");
				fflush(stdout);
				fp->Read(&b,1);
				return; 
			}
    	} else 
			cnt = 1;
    
		for (i=0; i<cnt; i++) 
		{
			if (depth == 1) 
			{
				for (j=0, mask=0x80; j<8; j++) 
				{
					//printf(".");
					*image++ |= ((b & mask) ? pmask : 0);
					mask = mask >> 1;
				}
			} else 
			{
				//char newVal = 65 + floor((((float)b/255.0) * 64.0));
				//printf("%c",newVal);
				*image++ = b;
      		}

			bcnt++;
	
			if (bcnt == bperlin) 
			{     /* end of a line reached */
				//printf("\n");
				bcnt = 0;
				plane++;  

				if (plane >= (int) hdr[PCX_PLANES]) 
				{   /* moved to next row */
					plane = 0;
					image -= pad;
					oldimage = image;
					row++;
					if (row >= h) 
						return;   /* done */
				} else 
				{   /* next plane, same row */
					image = oldimage;
				}	

				pmask = 1 << plane;
			}
		}
	}
}    

/*****************************/
static int pcxLoadImage(BFile *fp, 
	byte *image, byte *hdr, 
	int w, int h)
{
	switch (hdr[PCX_BPP]) 
	{
		case 1:   pcxLoadRaster(fp, image, 1, hdr, w, h);   break;
		case 8:   pcxLoadRaster(fp, image, 8, hdr, w, h);   break;
		default:
			pcxError("Unsupported # of bits per plane.");
			return (0);
	}

	return 1;
}



//============ BEGIN ADD-ON INTERFACE ================
#pragma export on
char *rrasaddon_IDName();
char *rrasaddon_IDAuthor();
char *rrasaddon_IDNotice();
char *rrasaddon_IDEncoder();
char *rrasaddon_IDDecoder();
float CanCreateImage(void *bytes, long byteLen);
GfxImage *CreateImage(BFile *file);
#pragma export off

char *IDName = "PCX Codec";
char *IDAuthor = "William Adams";
char *IDNotice = "Copyright Be Inc. 1996, All Rights reserved.";
char *IDEncoder = "IDpcx";
char *IDDecoder = "IDpcx";

char *rrasaddon_IDName()
{
	return IDName;
}

char *rrasaddon_IDAuthor()
{
	return IDAuthor;
}

char *rrasaddon_IDNotice()
{
	return IDNotice;
}

char *rrasaddon_IDEncoder()
{
	return IDEncoder;
}

char *rrasaddon_IDDecoder()
{
	return IDDecoder;
}


//====================================================
// Function: CanReadImage
// 
// Returns a float value representing the degree to
// which this module is capable of reading the image.
// The closer it returns to 1.0, the more confident
// it is that the image can be read successfully.
//
// If the image can be read at all, then a value of 0.8
// should be returned.  If it can't be read at all, then
// a value of 0.0 should be returned.
//====================================================

float
CanCreateImage(void *data, long dataLen)
{
	if (dataLen < 6)
		return 0.0;
		
	byte *ptr = (byte *)data;
	
	if ((ptr[0]==0x0a) && (ptr[1] <= 5))
		return 1.0;

	return 0.0;
}



//====================================================
// Function: CreateImage
//
// Create a GfxImage based on a file of GIF data
//====================================================

GfxImage *
CreateImage(BFile *fp)
{
	printf("CODECpcx - attempting to create image\n");

	if (!fp)
	{
		pcxError("CODECpcx - early return, lack of file\n");
		return 0;
	}
		
	fp->Open(B_READ_ONLY);
	fp->Seek(0, B_SEEK_TOP);
		
	// Allocate space for the image
	GfxImage *img = (GfxImage *)malloc(sizeof(GfxImage));
	img->type = B_COLOR_8_BIT;
	img->data  = (byte *) NULL;
	img->comment = (char *) NULL;


	long   filesize = 0;
	char  *bname, *errstr;
	byte   hdr[128];
	int    i, colors, gray;


	filesize = fp->Size();
	fp->Open(B_READ_ONLY);
	fp->Seek(0, B_SEEK_TOP);


	/* read the PCX header */
	if ((fp->Read(hdr, (size_t) 128)==B_ERROR)) 
	{
		fp->Close();
		pcxError("EOF reached in PCX header.\n");
		return 0;
	}

	if (hdr[PCX_ID] != 0x0a || hdr[PCX_VER] > 5) 
	{
		fp->Close();
		pcxError("unrecognized magic number");
		return 0;
	}

	img->width = (hdr[PCX_XMAXL] + ((int) hdr[PCX_XMAXH]<<8)) 
           - (hdr[PCX_XMINL] + ((int) hdr[PCX_XMINH]<<8));

	img->height = (hdr[PCX_YMAXL] + ((int) hdr[PCX_YMAXH]<<8)) 
           - (hdr[PCX_YMINL] + ((int) hdr[PCX_YMINH]<<8));

	img->width++;  
	img->height++;
	img->bytes_per_row = hdr[PCX_BPRL] + ((int) hdr[PCX_BPRH]<<8);
	
	colors = 1 << (hdr[PCX_BPP] * hdr[PCX_PLANES]);

	if (MY_DEBUG) {
    fprintf(stderr,"PCX: %dx%d image, version=%d, encoding=%d\n", 
	    img->width, img->height, hdr[PCX_VER], hdr[PCX_ENC]);
    fprintf(stderr,"   BitsPerPixel=%d, planes=%d, BytePerRow=%d, colors=%d\n",
	    hdr[PCX_BPP], hdr[PCX_PLANES], 
	    hdr[PCX_BPRL] + ((int) hdr[PCX_BPRH]<<8),
	    colors);
	}

	if (colors>256) 
	{
		fp->Close();
		pcxError("No more than 256 colors allowed in PCX file.");
		return 0;  
	}

  if (hdr[PCX_ENC] != 1) {
    fp->Close();
    pcxError("Unsupported PCX encoding format.");
	return 0;
  }

	/* note:  overallocation to make life easier... */
	img->data = (byte *) malloc((size_t) (img->height + 1) * img->width + 16);
	if (!img->data) 
	{
		pcxError("Can't alloc 'image' in CreateImage(PCX)");
		return 0;
	}
	
	memset((char *) img->data, 0x3f, (size_t) ((img->height+1) * img->width + 16));

	printf("CODECpcx - 1.8 - image: 0x%x\n", img->data);
	if (!pcxLoadImage(fp, img->data, hdr, img->width, img->height)) 
	{
		pcxError("pcxLoadImage returned null");
		free(img->data);
		fp->Close();
		return 0;
	}


	if ((fp->Error() == B_ERROR))    /* just a warning */
		pcxError("PCX file appears to be truncated.");

	if (colors>16) 
	{       /* handle trailing colormap */
		while (1) 
		{
			byte aChar;
			fp->Read(&aChar,1);
			if ((fp->Error() == B_ERROR) || 
				(aChar==PCX_MAPSTART)) 
				break;
		}

		for (i=0; i<colors; i++) 
		{
			fp->Read(&img->palette[i].red,1);
			fp->Read(&img->palette[i].green,1);
			fp->Read(&img->palette[i].blue,1);
		}

		if (fp->Error() == B_ERROR)
		{
			pcxError("Error reading PCX colormap.  Using grayscale.");
			for (i=0; i<256; i++) 
			{
				img->palette[i].red = i;
				img->palette[i].green = i;
				img->palette[i].blue = i;
			}
		}
	} else if (colors<=16) 
	{   /* internal colormap */
		for (i=0; i<colors; i++) 
		{
			img->palette[i].red = hdr[PCX_CMAP + i*3];
			img->palette[i].green = hdr[PCX_CMAP + i*3 + 1];
			img->palette[i].blue = hdr[PCX_CMAP + i*3 + 2];
		}
	}

	if (colors == 2) 
	{    /* b&w */
    	if (MONO(img->palette[0].red, img->palette[0].green, img->palette[0].blue) ==
			MONO(img->palette[1].red, img->palette[1].green, img->palette[1].blue)) 
		{
      /* create cmap */
      img->palette[0].red = img->palette[0].green = img->palette[0].blue = 255;
      img->palette[1].red = img->palette[1].green = img->palette[1].blue = 0;
      if (MY_DEBUG) 
		fprintf(stderr,"PCX: no cmap:  using 0=white,1=black\n");
    }
  }


	fp->Close();

	/* finally, convert into XV internal format */


	img->file_format = -1;    /* no default format to save in */

	/* check for grayscaleitude */
	for (i=0; i<colors; i++) 
	{
		if ((img->palette[i].red != img->palette[i].green) || 
			(img->palette[i].red != img->palette[i].blue)) 
			break;
	}
	gray = (i==colors) ? 1 : 0;


  if (colors > 2 || (colors==2 && !gray)) {  /* grayscale or PseudoColor */
    img->color_format = (gray) ? F_GREYSCALE : F_FULLCOLOR;
    sprintf(img->full_info, 
	    "%s PCX, %d plane%s, %d bit%s per pixel.  (%ld bytes)", 
	    (gray) ? "Greyscale" : "Color", 
	    hdr[PCX_PLANES], (hdr[PCX_PLANES]==1) ? "" : "s",
	    hdr[PCX_BPP],    (hdr[PCX_BPP]==1) ? "" : "s",
	    filesize);
  }
  else {
    img->color_format = F_BWDITHER;
    sprintf(img->full_info, "B&W PCX.  (%ld bytes)", filesize);
  }

  sprintf(img->short_info, "%dx%d PCX.", img->width, img->height);
	
  return img;
}

#pragma export off
