/*
 * xvjpeg.c - i/o routines for 'jpeg' format pictures
 */

/* based in part on 'example.c' from the IJG JPEG distribution */


#include "imglib_private.h"

#ifdef HAVE_JPEG

#include <setjmp.h>

#include "jpeglib.h"
#include "jerror.h"

#define CREATOR_STR "CREATOR: "

#if BITS_IN_JSAMPLE != 8
  Sorry, this code only copes with 8-bit JSAMPLEs. /* deliberate syntax err */
#endif


/*** Stuff for JPEG Dialog box ***/
#define JWIDE 400
#define JHIGH 200
#define J_NBUTTS 2
#define J_BOK    0
#define J_BCANC  1
#define BUTTH    24

/* minimum size compression when doing a 'quick' image load.  (of course, if
   the image *is* smaller than this, you'll get whatever size it actually is.
   This is currently hardcoded to be twice the size of a schnauzer icon, as
   the schnauzer's the only thing that does a quick load... */

#define QUICKWIDE 160    
#define QUICKHIGH 120

struct my_error_mgr {
  struct jpeg_error_mgr pub;
  jmp_buf               setjmp_buffer;
};

typedef struct my_error_mgr *my_error_ptr;


/*** local functions ***/
static    void         drawJD             (int, int, int, int);
static    void         clickJD            (int, int);
static    void         doCmd              (int);
static    void         writeJPEG          (void);
METHODDEF void         xv_error_exit      (j_common_ptr);
METHODDEF void         xv_error_output    (j_common_ptr);
METHODDEF void         xv_prog_meter      (j_common_ptr);
static    unsigned int j_getc             (j_decompress_ptr);
METHODDEF boolean      xv_process_comment (j_decompress_ptr);
static    int          writeJFIF          (FILE *, byte *, int,int,int);



/*** local variables ***/
static char *filename;
static char *fbasename;
static char *comment;



void
WriteJPEG(FILE *fp, byte *inpix, int ptype, int w, int h,
	  byte *rmap, byte *gmap, byte *bmap,
	  int nc, int colorType,
	  char *comments)
{
  int            i, rv, pfree;
  register byte *ip, *ep;
  byte          *image8, *image24;


/* inpix = GenSavePic(&ptype, &w, &h, &pfree, &nc, &rmap, &gmap, &bmap); */

  image8 = image24 = (byte *) NULL;


  /* monocity: see if the image is mono, save it that way to save space */
  if (colorType != F_GREYSCALE) {
    if (ptype == IMAGE_8_BIT) {
      for (i=0; i<nc && rmap[i]==gmap[i] && rmap[i]==bmap[i]; i++);
      if (i==nc) colorType = F_GREYSCALE;    /* made it all the way through */
    }
    else {  /* IMAGE_24_BIT */
      for (i=0,ip=inpix; i<w*h && ip[0]==ip[1] && ip[1]==ip[2]; i++,ip+=3);
      if (i==w*h) colorType = F_GREYSCALE;  /* all the way through */
    }
  }
  
  
  /* first thing to do is build an 8/24-bit Greyscale/TrueColor image
     (meaning: non-colormapped) */
  
  if (colorType == F_GREYSCALE) {   /* build an 8-bit Greyscale image */
    image8 = (byte *) malloc((size_t) w * h);
    if (!image8) FatalError("writeJPEG: unable to malloc image8\n");
    
    if (ptype == IMAGE_8_BIT) {
      for (i=0,ip=image8,ep=inpix; i<w * h; i++, ip++, ep++)
	*ip = MONO(rmap[*ep], gmap[*ep], bmap[*ep]);
    }
    else {  /* IMAGE_24_BIT */
      for (i=0,ip=image8,ep=inpix; i<w*h; i++, ip++, ep+=3)
	*ip = MONO(ep[0],ep[1],ep[2]);
    }
  }

  else {    /* *not* F_GREYSCALE */
    if (ptype == IMAGE_8_BIT) {
      image24 = (byte *) malloc((size_t) w * h * 3);
      if (!image24) {  /* this simply isn't going to work */
	FatalError("writeJPEG: unable to malloc image24\n");
      }

      for (i=0, ip=image24, ep=inpix; i<w*h; i++, ep++) {
	*ip++ = rmap[*ep];
	*ip++ = gmap[*ep];
	*ip++ = bmap[*ep];
      }
    }

    else {  /* IMAGE_24_BIT */
      image24 = inpix;
    }
  }

  
  /* in any event, we've got some valid image.  Do the JPEG Thing */
  rv = writeJFIF(fp, (colorType==F_GREYSCALE) ? image8 : image24,
		 w, h, colorType);
  
  if      (colorType == F_GREYSCALE) free(image8);
  else if (ptype == IMAGE_8_BIT)            free(image24);
}






/***************************************************************************/
/* JPEG INTERFACE ROUTINES *************************************************/
/***************************************************************************/



/**************************************************/
METHODDEF void xv_error_exit(cinfo) 
     j_common_ptr cinfo;
{
  my_error_ptr myerr;

  myerr = (my_error_ptr) cinfo->err;
  (*cinfo->err->output_message)(cinfo);     /* display error message */
  longjmp(myerr->setjmp_buffer, 1);         /* return from error */
}


/**************************************************/
METHODDEF void xv_error_output(cinfo) 
     j_common_ptr cinfo;
{
  my_error_ptr myerr;
  char         buffer[JMSG_LENGTH_MAX];

  myerr = (my_error_ptr) cinfo->err;
  (*cinfo->err->format_message)(cinfo, buffer);

  fprintf(stderr, "%s: %s", fbasename, buffer);   /* send it to XV */
}


/**************************************************/
METHODDEF void xv_prog_meter(cinfo)
     j_common_ptr cinfo;
{
  struct jpeg_progress_mgr *prog;

  prog = cinfo->progress;


#ifdef FOO
  fprintf(stderr,"xv_prog_meter: cnt=%ld, maxcnt=%ld, pass=%d, maxpass=%d\n",
	  prog->pass_counter, prog->pass_limit, prog->completed_passes,
	  prog->total_passes);
#endif
}



/***************************************************************************/
/* LOAD ROUTINES ***********************************************************/
/***************************************************************************/


/*******************************************/
int LoadJFIF(fname, img, quick)
     char    *fname;
     GfxImage *img;
     int      quick;
{
  /* returns '1' on success, '0' on failure */

  struct jpeg_decompress_struct    cinfo;
  struct jpeg_progress_mgr         prog;
  struct my_error_mgr              jerr;
  JSAMPROW                         rowptr[1];
  FILE                            *fp;
  static byte                     *pic;
  long                             filesize;
  int                              i,w,h,bperpix;


  fbasename = imglib_basename(fname);
  pic       = (byte *) NULL;
  comment   = (char *) NULL;

  img->type  = IMAGE_8_BIT;

  if ((fp = fopen(fname, "r")) == NULL) return 0;

  fseek(fp, 0L, 2);
  filesize = ftell(fp);
  fseek(fp, 0L, 0);


  cinfo.err = jpeg_std_error(&jerr.pub);
  jerr.pub.error_exit     = xv_error_exit;
  jerr.pub.output_message = xv_error_output;

  if (setjmp(jerr.setjmp_buffer)) {
    /* if we're here, it blowed up... */
    jpeg_destroy_decompress(&cinfo);
    fclose(fp);
    if (pic)     free(pic);
    if (comment) free(comment);

    img->data = (byte *) NULL;
    img->comment = (char *) NULL;

    return 0;
  }


  jpeg_create_decompress(&cinfo);
  jpeg_set_marker_processor(&cinfo, JPEG_COM, xv_process_comment);

  /* hook up progress meter */
  prog.progress_monitor = xv_prog_meter;
  cinfo.progress = &prog;

  jpeg_stdio_src(&cinfo, fp);
  (void) jpeg_read_header(&cinfo, TRUE);



  /* do various cleverness regarding decompression parameters & such... */



  jpeg_calc_output_dimensions(&cinfo);
  w = cinfo.output_width;
  h = cinfo.output_height;

  if (quick) {
    int wfac, hfac, fac;
    wfac = w / QUICKWIDE;
    hfac = h / QUICKHIGH;

    fac = wfac;  if (fac > hfac) fac = hfac;
    if      (fac > 8) fac = 8;
    else if (fac > 4) fac = 4;
    else if (fac > 2) fac = 2;
    else fac = 1;

    cinfo.scale_num   = 1;
    cinfo.scale_denom = fac;
    cinfo.dct_method = JDCT_FASTEST;
    cinfo.do_fancy_upsampling = FALSE;


    jpeg_calc_output_dimensions(&cinfo);
    w = cinfo.output_width;
    h = cinfo.output_height;
  }


  if (cinfo.jpeg_color_space == JCS_GRAYSCALE) {
    cinfo.out_color_space = JCS_GRAYSCALE;
    cinfo.quantize_colors = FALSE;
    
    for (i=0; i<256; i++) img->r[i] = img->g[i] = img->b[i] = i;
  }
  
  jpeg_calc_output_dimensions(&cinfo);   /* note colorspace changes... */
    

  if (cinfo.output_components != 1 && cinfo.output_components != 3) {
    fprintf(stderr, "%s:  can't read %d-plane JPEG file!",
	    fbasename, cinfo.output_components);
    jpeg_destroy_decompress(&cinfo);
    fclose(fp);
    if (comment) free(comment);
    return 0;
  }


  bperpix = cinfo.output_components;
  img->type = (bperpix == 1) ? IMAGE_8_BIT : IMAGE_24_BIT;

  pic = (byte *) malloc((size_t) (w * h * bperpix));
  if (!pic) {
    fprintf(stderr, "%s:  can't read JPEG file - out of memory",
	    fbasename);
    jpeg_destroy_decompress(&cinfo);
    fclose(fp);
    if (comment) free(comment);
    return 0;
  }
  
  jpeg_start_decompress(&cinfo);

  while (cinfo.output_scanline < cinfo.output_height) {
    rowptr[0] = (JSAMPROW) &pic[cinfo.output_scanline * w * bperpix];
    (void) jpeg_read_scanlines(&cinfo, rowptr, (JDIMENSION) 1);
  }

  
  jpeg_finish_decompress(&cinfo);


  /* return 'GfxImage' structure to XV */

  img->data = pic;
  img->width = w;
  img->height = h;
  img->file_format = F_JPEG;

  if (cinfo.out_color_space == JCS_GRAYSCALE) {
    sprintf(img->full_info, "Greyscale JPEG. (%ld bytes)", filesize);
    img->color_format = F_GREYSCALE;
    
    for (i=0; i<256; i++) img->r[i] = img->g[i] = img->b[i] = i;
  }
  else {
    sprintf(img->full_info, "Color JPEG. (%ld bytes)", filesize);
    img->color_format = F_FULLCOLOR;

    if (cinfo.quantize_colors) {
      for (i=0; i<cinfo.actual_number_of_colors; i++) {
	img->r[i] = cinfo.colormap[0][i];
	img->g[i] = cinfo.colormap[1][i];
	img->b[i] = cinfo.colormap[2][i];
      }
    }
  }
  
  sprintf(img->short_info, "%dx%d %s JPEG. ", w,h, 
	  (cinfo.out_color_space == JCS_GRAYSCALE) ? "Greyscale " : "Color ");
  
  img->comment = comment;

  jpeg_destroy_decompress(&cinfo);
  fclose(fp);

  comment = NULL;
  return 1;
}
  
  


/**************************************************/
static unsigned int j_getc(cinfo)
     j_decompress_ptr cinfo;
{
  struct jpeg_source_mgr *datasrc = cinfo->src;
  
  if (datasrc->bytes_in_buffer == 0) {
    if (! (*datasrc->fill_input_buffer) (cinfo))
      ERREXIT(cinfo, JERR_CANT_SUSPEND);
  }
  datasrc->bytes_in_buffer--;
  return GETJOCTET(*datasrc->next_input_byte++);
}


/**************************************************/
METHODDEF boolean xv_process_comment(cinfo)
     j_decompress_ptr cinfo;
{
  int          length, hasnull;
  unsigned int ch;
  char         *oldsp, *sp;

  length  = j_getc(cinfo) << 8;
  length += j_getc(cinfo);
  length -= 2;                  /* discount the length word itself */

  if (!comment) {
    comment = (char *) malloc((size_t) length + 1);
    if (comment) comment[0] = '\0';
  }
  else comment = (char *) realloc(comment, strlen(comment) + length + 1);
  if (!comment) FatalError("out of memory in xv_process_comment");
  
  oldsp = sp = comment + strlen(comment);
  hasnull = 0;

  while (length-- > 0) {
    ch = j_getc(cinfo);
    *sp++ = (char) ch;
    if (ch == 0) hasnull = 1;
  }

  if (hasnull) sp = oldsp;       /* swallow comment blocks that have nulls */
  *sp++ = '\0';

  return TRUE;
}




/***************************************************************************/
/* WRITE ROUTINES **********************************************************/
/***************************************************************************/

static int writeJFIF(fp, pic, w,h, coltype)
     FILE *fp;
     byte *pic;
     int   w,h,coltype;
{
  struct     jpeg_compress_struct cinfo;
  struct     jpeg_progress_mgr    prog;
  struct     my_error_mgr         jerr;
  JSAMPROW                        rowptr[1];
  int                             i, bperpix;
  char                            xvcmt[256];

  comment = (char *) NULL;

  cinfo.err               = jpeg_std_error(&jerr.pub);
  jerr.pub.error_exit     = xv_error_exit;
  jerr.pub.output_message = xv_error_output;

  if (setjmp(jerr.setjmp_buffer)) {
    /* if we're here, it blowed up... */
    jpeg_destroy_compress(&cinfo);
    /* XXXdbg if (picComments && comment) free(comment); */
    return 1;
  }


  jpeg_create_compress(&cinfo);
  jpeg_stdio_dest(&cinfo, fp);

  cinfo.image_width  = w;
  cinfo.image_height = h;
  if (coltype == F_GREYSCALE) {
    cinfo.input_components = 1;
    cinfo.in_color_space = JCS_GRAYSCALE;
  }
  else {
    cinfo.input_components = 3;
    cinfo.in_color_space = JCS_RGB;
  }

  bperpix = cinfo.input_components;


  prog.progress_monitor = xv_prog_meter;
  cinfo.progress = &prog;


  jpeg_set_defaults(&cinfo);
#if 0
  jpeg_set_quality(&cinfo, qDial.val, TRUE);
  cinfo.smoothing_factor = smDial.val;
#endif


  jpeg_start_compress(&cinfo, TRUE);


  /*** COMMENT HANDLING ***/

#if 0  /* XXXdbg */
  sprintf(xvcmt, "%sBeOS Imglib %s  Quality = %d, Smoothing = %d\n",
	  CREATOR_STR, REVDATE, qDial.val, smDial.val);
 
  if (picComments) {   /* append XV comment */
    char *sp, *sp1;  int done;

    i   = strlen(picComments);
    comment = (char *) malloc(i + strlen(xvcmt) + 2 + 1);
    if (!comment) FatalError("out of memory in writeJFIF()");
    
    strcpy(comment, picComments);
    
    /* see if there's a line that starts with 'CREATOR: ' in the
       comments.  If there is, rip it out. */
    
    sp = comment;  done = 0;
    while (!done && *sp) {
      if (strncmp(sp, CREATOR_STR, strlen(CREATOR_STR)) == 0) {
	sp1 = sp;
	while (*sp1 && *sp1 != '\n') sp1++;    /* find end of this line */
	if (*sp1 == '\n') sp1++;               /* move past \n */

	/* move comments from sp1 and on down to sp */
	bcopy(sp1, sp, strlen(sp1) + 1);   /* +1 to copy the trailing \0 */

	done = 1;
      }
      else {   /* skip ahead to next line */
	while (*sp && *sp != '\n') sp++;
	if (*sp == '\n') sp++;
      }
    }

    /* count # of \n's at end of comment.  
       If none, add 2.   If one, add 1.  If two or more, add none. */

    sp = comment + strlen(comment);
    for (i=0; i<3 && i<strlen(comment); i++) {
      sp--;
      if (*sp != '\n') break;
    }

    for ( ; i<2; i++) strcat(comment, "\n");
    strcat(comment, xvcmt);
  }
  else comment = xvcmt;
  
#else
  comment = "BeOS Image IO library\n";
#endif  
  
  jpeg_write_marker(&cinfo,JPEG_COM,(byte *) comment,(u_int) strlen(comment));
  
  while (cinfo.next_scanline < cinfo.image_height) {
    rowptr[0] = (JSAMPROW) &pic[cinfo.next_scanline * w * bperpix];
    (void) jpeg_write_scanlines(&cinfo, rowptr, (JDIMENSION) 1);
  }
  
  jpeg_finish_compress(&cinfo);
  jpeg_destroy_compress(&cinfo);
  return 0;
}




#endif  /* HAVE_JPEG */
