#include <stdio.h>
#include <string.h>

#include <File.h>

#include "rraster.h"

extern "C" {
#include "jinclude.h"
#include "jpeglib.h"
#include "jerror.h"		/* get library error codes too */
}

#include <setjmp.h>

struct my_error_mgr {
  struct jpeg_error_mgr pub;	/* "public" fields */

  jmp_buf setjmp_buffer;	/* for return to caller */
};

typedef struct my_error_mgr * my_error_ptr;

/*
 * Here's the routine that will replace the standard error_exit method:
 */

METHODDEF void
my_error_exit (j_common_ptr cinfo)
{
  /* cinfo->err really points to a my_error_mgr struct, so coerce pointer */
  my_error_ptr myerr = (my_error_ptr) cinfo->err;

  /* Always display the message. */
  /* We could postpone this until after returning, if we chose. */
  (*cinfo->err->output_message) (cinfo);

  /* Return control to the setjmp point */
  longjmp(myerr->setjmp_buffer, 1);
}



static GfxImage *	
LoadJFIF(const char *filename, GfxImage *newBitmap)
{
	/* This struct contains the JPEG decompression parameters and pointers to
	* working space (which is allocated as needed by the JPEG library).
	*/
	struct jpeg_decompress_struct cinfo;
  
	/* We use our private extension JPEG error handler.
	* Note that this struct must live as long as the main JPEG parameter
	* struct, to avoid dangling-pointer problems.
	*/
	struct my_error_mgr jerr;
	
	/* More stuff */
	FILE * infile;		/* source file */
	JSAMPARRAY buffer;		/* Output row buffer */
	int row_stride;		/* physical row width in output buffer */

	/* In this example we want to open the input file before doing anything else,
	* so that the setjmp() error recovery below can assume the file is open.
	* VERY IMPORTANT: use "b" option to fopen() if you are on a machine that
	* requires it in order to read binary files.
	*/

	if ((infile = fopen(filename, "r")) == NULL) 
	{
		fprintf(stderr, "can't open %s\n", filename);
		return 0;
	}

/* Step 1: allocate and initialize JPEG decompression object */

	/* We set up the normal JPEG error routines, then override error_exit. */
	cinfo.err = jpeg_std_error(&jerr.pub);
	jerr.pub.error_exit = my_error_exit;
	
	/* Establish the setjmp return context for my_error_exit to use. */
	if (setjmp(jerr.setjmp_buffer)) 
	{
    	/* If we get here, the JPEG code has signaled an error.
     	* We need to clean up the JPEG object, close the input file, and return.
     	*/
		jpeg_destroy_decompress(&cinfo);
		fclose(infile);
		return 0;
	}
	
	/* Now we can initialize the JPEG decompression object. */
	jpeg_create_decompress(&cinfo);

/* Step 2: specify data source (eg, a file) */

	jpeg_stdio_src(&cinfo, infile);

/* Step 3: read file parameters with jpeg_read_header() */

	(void) jpeg_read_header(&cinfo, TRUE);
	/* We can ignore the return value from jpeg_read_header since
	*   (a) suspension is not possible with the stdio data source, and
	*   (b) we passed TRUE to reject a tables-only JPEG file as an error.
	* See libjpeg.doc for more info.
	*/

/* Step 4: set parameters for decompression */

	/* In this example, we don't need to change any of the defaults set by
	* jpeg_read_header(), so we do nothing here.
	*/

/* Step 5: Start decompressor */

	(void) jpeg_start_decompress(&cinfo);
	/* We can ignore the return value since suspension is not possible
	* with the stdio data source.
	*/

	/* We may need to do some setup of our own at this point before reading
	* the data.  After jpeg_start_decompress() we have the correct scaled
	* output image dimensions available, as well as the output colormap
	* if we asked for color quantization.
	* In this example, we need to make an output work buffer of the right size.
	*/ 
	
	/* JSAMPLEs per row in output buffer */
	row_stride = cinfo.output_width * cinfo.output_components;
	newBitmap->width = cinfo.output_width;
	newBitmap->height = cinfo.output_height;
	newBitmap->bytes_per_row = cinfo.output_width * 4;
	newBitmap->data = (unsigned char *)malloc(cinfo.output_width*cinfo.output_height*4);
	newBitmap->type = B_RGB_32_BIT;
	
	/* Make a one-row-high sample array that will go away when done with image */
	buffer = (*cinfo.mem->alloc_sarray)
		((j_common_ptr) &cinfo, JPOOL_IMAGE, row_stride, 1);
		
	unsigned char *pixels = newBitmap->data;
	
/* Step 6: while (scan lines remain to be read) */
  /*           jpeg_read_scanlines(...); */

  /* Here we use the library's state variable cinfo.output_scanline as the
   * loop counter, so that we don't have to keep track ourselves.
   */
	while (cinfo.output_scanline < cinfo.output_height) 
	{
		/* jpeg_read_scanlines expects an array of pointers to scanlines.
		* Here the array is only one element long, but you could ask for
		* more than one scanline at a time if that's more convenient.
		*/
		int linesRead = jpeg_read_scanlines(&cinfo, buffer, 1);
		
		/* Assume put_scanline_someplace wants a pointer and sample count. */
		int dstoffset = (cinfo.output_scanline-1) * newBitmap->bytes_per_row;
		for (int ctr = 0; ctr < cinfo.output_width; ctr++)
		{
			int ctr4 = ctr*4;
			int ctr3 = ctr*3;
			
			pixels[dstoffset+(ctr4)] = buffer[0][(ctr3)+2];		// Blue
			pixels[dstoffset+(ctr4)+1] = buffer[0][(ctr3)+1];	// Green 
			pixels[dstoffset+(ctr4)+2] = buffer[0][(ctr3)]; // Red
			pixels[dstoffset+(ctr4)+3] = 255;
		}
	}

/* Step 7: Finish decompression */

	(void) jpeg_finish_decompress(&cinfo);
	/* We can ignore the return value since suspension is not possible
	* with the stdio data source.
	*/

/* Step 8: Release JPEG decompression object */

	/* This is an important step since it will release a good deal of memory. */
	jpeg_destroy_decompress(&cinfo);

	/* After finish_decompress, we can close the input file.
	* Here we postpone it until after no more JPEG errors are possible,
	* so as to simplify the setjmp error logic above.  (Actually, I don't
	* think that jpeg_destroy can do an error exit, but why assume anything...)
	*/
	fclose(infile);

  /* At this point you may want to check to see whether any corrupt-data
   * warnings occurred (test whether jerr.pub.num_warnings is nonzero).
   */

  /* And we're done! */
  return newBitmap;
}


#pragma export on
char *rrasaddon_IDName();
char *rrasaddon_IDAuthor();
char *rrasaddon_IDNotice();
char *rrasaddon_IDEncoder();
char *rrasaddon_IDDecoder();
float CanCreateImage(void *bytes, long byteLen);
GfxImage *CreateImage(BFile *file);
GfxImage *CreateImage(char *file);
#pragma export off

char *IDName = "JFIF Codec";
char *IDAuthor = "William Adams";
char *IDNotice = "Copyright Be Inc. 1996, All Rights reserved.";
char *IDEncoder = "IDjfif";
char *IDDecoder = "IDjfif";

static int LoadJFIF(char *fname, GfxImage *img, int quick);

char *rrasaddon_IDName()
{
	return IDName;
}

char *rrasaddon_IDAuthor()
{
	return IDAuthor;
}

char *rrasaddon_IDNotice()
{
	return IDNotice;
}

char *rrasaddon_IDEncoder()
{
	return IDEncoder;
}

char *rrasaddon_IDDecoder()
{
	return IDDecoder;
}


//====================================================
// Function: CanReadImage
// 
// Returns a float value representing the degree to
// which this module is capable of reading the image.
// The closer it returns to 1.0, the more confident
// it is that the image can be read successfully.
//
// If the image can be read at all, then a value of 0.8
// should be returned.  If it can't be read at all, then
// a value of 0.0 should be returned.
//====================================================

float
CanCreateImage(void *data, long dataLen)
{
	if (dataLen < 3)
		return 0.0;
		
	unsigned char *dataPtr = (unsigned char *)data;
	
	if (dataPtr[0]==0xff && dataPtr[1]==0xd8 && 
	   dataPtr[2]==0xff)
		return 1.0;

	return 0.0;
}

//====================================================
// Function: CreateImage
//
// Create a GfxImage based on a file of GIF data
//====================================================

GfxImage *
CreateImage(BFile *fp)
{
	char pathname[1024]={'\0'};
	
	fp->GetPath(pathname,1023);
	printf("%s - Creating: %s\n", IDName, pathname);
	 
	return CreateImage(pathname);
}


GfxImage *
CreateImage(char *fname)
{
	GfxImage *newImage = (GfxImage *)malloc(sizeof(GfxImage));
	
	if (LoadJFIF(fname, newImage))
		return newImage;
	else
		free(newImage);
		
	return 0;
}


#pragma export off
