#include "gameview.h"
#include "images.h"

#include <malloc.h>
#include <stdio.h>

/*
	Function:  DrawBitmap8
	
	Copies a rectangular area from the source to a specified
	offset in the destination.  The transfer mode can be used
	as well to specify whether a copy is performed or an
	sover operation instead.
*/

pixel_buffer fScreen;
GameView *gGamePresenter = 0;

static void
DrawBitmap8(const pixel_buffer &src, 
	const ushort x1, const ushort y1,
	const ushort width, const ushort height)
{
	unsigned long dstoffset = y1*(fScreen.bytes_per_row)+x1;
	unsigned long srcoffset = 0;
	
	unsigned char *dstbits = (unsigned char *)fScreen.pixels;
	unsigned char *srcbits = (unsigned char *)src.pixels;
	
	for (ushort y = 0; y < height; y++)
	{
		for (ushort x=0; x < width; x++)
		{
			//if (srcbits[srcoffset+x] != B_TRANSPARENT_8_BIT)
				dstbits[dstoffset+x] = ((unsigned char *)src.pixels)[srcoffset+x];
		}

		dstoffset += fScreen.bytes_per_row;
		srcoffset += src.bytes_per_row;
	}
}

/*

*/

GameView::GameView(GameEngine *gameEngine)
	: BWindowScreen("Multipede", B_8_BIT_640x480),
	fGameEngine(gameEngine),
	fGameState(gameEngine->fGameState)
{
	gGamePresenter = this;
	
	unsigned long width;
	unsigned long height;
	
	fGameEngine->GetSize(width, height);
	
	cdmap = 0;
	dmap=(enum mtype *) malloc(sizeof(enum mtype)*2*width*height);
	
	// unset the display map
	unset_dmap();
		
}

GameView::~GameView()
{
	free(dmap);
}

void 
GameView::Quit()
{
	Disconnect();
	
	// Quit the WindowScreen.
	BWindowScreen::Quit();
}

void 
GameView::ScreenConnected(bool active)
{
	if (active == TRUE)
	{
		fScreen.mode = B_COLOR_8_BIT;
		fScreen.pixels = (unsigned char *)CardInfo()->frame_buffer;
		fScreen.bytes_per_row = CardInfo()->bytes_per_row;
		fScreen.width = CardInfo()->width;
		fScreen.height = CardInfo()->height;
		
		#ifdef DEBUG
		printf("ABePEDEWindow::ScreenConnected\n");
		printf("ABePEDEWindow::ScreenConnected - size: %d %d\n",fScreen.width, fScreen.height);
		printf("ABePEDEWindow::ScreenConnected - bytes: %d\n",fScreen.bytes_per_row);
		#endif
		
		gGameEngine->PostMessage(MPCONTINUE);
	} else
		gGameEngine->PostMessage(MPPAUSE);
}


//=====================================================================
// The following filter routines must be implemented so that mouse
// and keyboard actions can go somewhere.  If you don't have anything
// special to do, then just leave them empty, but they must exist.
//=====================================================================

void	
GameView::MessageReceived(BMessage *message)
{
	switch (message->what) 
	{
		case B_KEY_DOWN:
		{
			long key = message->FindLong("key");
			long aChar = message->FindLong("char");
			
			switch (aChar)
			{
				case B_SPACE:
					gGameEngine->PostMessage(MPTOGGLEPAUSE);
				break;
				
				case B_LEFT_ARROW:
					//printf("GameView::MessageReceived - Left %x\n",aChar);
					gGameEngine->PostMessage(MPMOVELEFT);
				break;
				
				case B_RIGHT_ARROW:
					//printf("GameView::MessageReceived - Right %x\n",aChar);
					gGameEngine->PostMessage(MPMOVERIGHT);
				break;
				
				case B_UP_ARROW:
					//printf("GameView::MessageReceived - Up %x\n",aChar);
					gGameEngine->PostMessage(MPMOVEUP);
				break;
				
				case B_DOWN_ARROW:
					//printf("GameView::MessageReceived - Down %x\n",aChar);
					gGameEngine->PostMessage(MPMOVEDOWN);
				break;
				
				case B_ESCAPE:
					//printf("GameView::MessageReceived - posting quit message\n");
					gGameEngine->PostMessage(MPQUIT);			
				break;
				
				default:
					printf("GameView::MessageReceived - Unknown KeyDown 0x%x   Char 0x%x\n",key,aChar);
				break;
			}
		} 
		break;
		
	}	
}



/* draw the graphics from the map onto the screen */
void 
GameView::update_image_from_map() 
{
	unsigned long gwidth, gheight;
	fGameEngine->GetSize(gwidth, gheight);
	int width = gwidth *16;
	int height = gheight * 16;
	int x,y;

	#ifdef DEBUG
	printf("GamePresenter::update_image_from_map - \n");
	printf("  gwidth: %d gheight: %d\n", gwidth, gheight);
	printf("  width: %d height: %d\n", width, height);
	#endif
	
	for (y=0; y<(height); y+=16) 
	{
		for (x=0; x<(width); x+=16) 
		{
			enum mtype *offset = dmap+(cdmap*gheight*gwidth)  + ((x/16)*gheight) + (y/16);
			
			// only draw if the display map has changed since last time updated

			if (*(offset) != *(dmap+(((cdmap+1)%2)*gheight*gwidth)+((x/16)*gheight)+(y/16)) ) 
			{
				switch (*(offset))
				{
					case EMPTY : 
						DrawBitmap8(empty, x, y, 16, 16);
					break;
					
					case MUSH : 
						DrawBitmap8(mushroom, x, y, 16, 16);
					break;
					
					case SNAKE : 
						DrawBitmap8(snake, x, y, 16, 16);
					break;
					
					case KILL : 
						DrawBitmap8(killer, x, y, 16, 16);
					break;
					
					case HDUP : 
						DrawBitmap8(hdup, x, y, 16, 16);
					break;
					
					case HDRIGHT : 
						DrawBitmap8(hdright, x, y, 16, 16);
					break;
					
					case HDDOWN : 
						DrawBitmap8(hddown, x, y, 16, 16);
					break;
					
					case HDLEFT : 
						DrawBitmap8(hdleft, x, y, 16, 16);
					break;
					
					case HEART : 
						DrawBitmap8(heart, x, y, 16, 16);
					break;
				} 
			}
		}
	}
}


/* draw graphics after snake has died */
void 
GameView::update_image_dead() 
{

	int x,y,a,b;

	/* put the normal stuff on the display map */
	update_map();

	/* set size of rectangle of skulls to draw */
	a= (fGameState->dcount<(fGameState->width/2)) ? fGameState->dcount : (fGameState->width/2);
	b= (fGameState->dcount<(fGameState->height/2)) ? fGameState->dcount : (fGameState->height/2);

	/* draw the rectangle of skulls */
	for (x=(fGameState->width/2)-a; x<(fGameState->width/2)+a; x++) 
	{
		for (y=(fGameState->height/2)-b; y<(fGameState->height/2)+b; y++) 
		{
			*(dmap+(cdmap*fGameState->height*fGameState->width)+(x*fGameState->height)+y)=KILL;
		}
	}

	/* draw the display map onto the screen */
	update_image_from_map();
}

/* update the display map, and draw it */
void 
GameView::update_image() 
{
	#ifdef DEBUG
	printf("GamePresenter::update_image - BEGIN\n");
	#endif
	
	update_map();
	update_image_from_map();
	
	#ifdef DEBUG	
	printf("GamePresenter::update_image - END\n");
	#endif
}

/* draw graphics after player has won */
void 
GameView::update_image_won() 
{
	int x,y,a,b;

	/* put the normal stuff on the display map */
	update_map();

	/* set size of rectangle of skulls to draw */
	a= (fGameState->dcount<(fGameState->width/2)) ? fGameState->dcount : (fGameState->width/2);
	b= (fGameState->dcount<(fGameState->height/2)) ? fGameState->dcount : (fGameState->height/2);

	// draw the rectangle of skulls
	for (x=(fGameState->width/2)-a; x<(fGameState->width/2)+a; x++) 
	{
		for (y=(fGameState->height/2)-b; y<(fGameState->height/2)+b; y++) {
      *(dmap+(cdmap*fGameState->height*fGameState->width)+(x*fGameState->height)+y)=HEART;
		}
	}

	// draw the display map onto the screen
	update_image_from_map();
}


/* combines the snake and the underlying map onto a 'display' map */
void 
GameView::update_map() 
{
	int x,y,c;
	unsigned long width, height;
	fGameEngine->GetSize(width, height);
	
	#ifdef DEBUG
	printf("GamePresenter::update_map - width: %d  height: %d\n", width, height);
	#endif
	
	// swap the display map bank we're using
	cdmap= 1-cdmap;

	// copy the underlying map onto the display map
	for (y=0; y<height; y++) 
	{
		for (x=0; x<width; x++) 
		{
			*(dmap+(cdmap*height*width)+(x*height)+y)=*(fGameState->map+(x*height)+y);
		}
	}

	// overlay the snake onto the display map
	for (c=1; c<fGameState->slength; c++) 
	{
    	*(dmap+(cdmap*height*width) +
		(fGameState->shist[(fGameState->shistoff-c+HISTLENGTH)%HISTLENGTH].x*height) +
		fGameState->shist[(fGameState->shistoff-c+HISTLENGTH)%HISTLENGTH].y ) = SNAKE;
	}

	// copy the snake's head onto the display map
	*(dmap+(cdmap*height*width)+(fGameState->spos.x*height)+fGameState->spos.y)=(mtype)fGameState->sdir;

}

      
/* clear all the display map to UNSET */
void 
GameView::unset_dmap() 
{
	int x,y;
	unsigned long width, height;
	fGameEngine->GetSize(width, height);
	
	for (x=0; x<width; x++) 
	{
		for (y=0; y<height; y++) 
		{
			*(dmap+(cdmap*height*width)+(x*height)+y)=UNSET;
		}
	}
}

