#include "gamestick.h"
#include "gameengine.h"

#include <math.h>
#include <stdio.h>

//==================================================
// Method: GameStick
//
// Default constructor will attach to joystick1 (lower port)
// This is convenient when you don't want to specify
// a port, it uses a good default.
//==================================================

GameStick::GameStick()
{
	Initialize("joystick1");
}

//==================================================
// Method: GameStick
//
// Constructor method that takes a parameter to specify
// which game port the user wants to connect to.
// Valid values are:
//	GAMEPORT1	- The lower joystick port
//	GAMEPORT3	- The upper joystick port
//	GAMEPORT2	- Second of lower port with splitter
//	GAMEPORT4	- Second of upper port with splitter
//==================================================

GameStick::GameStick(EGamePort whichone)
{
	switch (whichone)
	{
		case GAMEPORT1:
			Initialize("joystick1");
		break;
		
		case GAMEPORT2:
			Initialize("joystick2");
		break;
		
		case GAMEPORT3:
			Initialize("joystick3");
		break;
		
		case GAMEPORT4:
			Initialize("joystick4");
		break;		
	}
}

void
GameStick::Initialize(const char *whichone)
{
	fIsValid = (fJoystick.Open(whichone) != B_ERROR);
	fZeroX = 0;
	fZeroY = 0;
	fMinX = 10000;
	fMinY = 10000;
	fMaxX = -10000;
	fMaxY = -10000;
	
	fMinXRange = 50;
	fMinYRange = 50;
	fMaxXRange = 50;
	fMaxYRange = 50;
	
	float x, y;
	FindZero(x, y);
}


//==================================================
// Method: FindZero
//
// This method tries to find a good zero value for the
// joystick.  The zero values are the x and y values 
// that are returned when the joystick is at rest.  Thus
// this routine should be called whenever the stick is sure
// to be at rest.
//
// A number of consecutive samples are taken from the 
// port and a weighted average is maintained.  At the
// end of the cycle, the zero values should be set.
//
// After the zero values are established, the gamestick
// is trained to get the current range values.
//==================================================

long
GameStick::FindZero(float &x, float &y)
{
	double totalX = 0.0;
	double totalY = 0.0;
	
	for (int counter = 1; counter <= 10; counter++)
	{
		fJoystick.Update();
		totalX = ((counter-1)*totalX + fJoystick.horizontal)/counter;
		totalY = ((counter-1)*totalY + fJoystick.vertical)/counter;
		//printf("totalX: %g  totalY: %g\n", totalX, totalY);
	}
	
	fZeroX = totalX;
	fZeroY = totalY;
	
	fLastX = fZeroX;
	fLastY = fZeroY;
	
	Train();

	x = fZeroX;
	y = fZeroY;

	
	return 0;
}

//==================================================
// Method: GetPosition
//
// Returns x and y values from -1 to 1
//
//	           1.0 y
//
//	-1         0.0      1.0 x
//     
//             -1.0
//==================================================

void
GameStick::GetPosition(float &x, float &y)
{
	fJoystick.Update();
	
	fLastX = fJoystick.horizontal;
	fLastY = fJoystick.vertical;
	Train();
	
	float xoffset = fLastX - fZeroX;
	float yoffset = fLastY - fZeroY;
	
	if (xoffset < 0.0)
		x = xoffset / fMinXRange;
	else
		x = xoffset / fMaxXRange;
	x = -x;
	
	if (yoffset > 0.0)
		y = yoffset / fMaxYRange;
	else
		y = yoffset / fMinYRange;
		
}

//===============================================
//	Method: GetAbsolutePosition
//
// This method returns the values as reported by the
// joystick port directly.  If you are going to use
// this method, then you may consider just using the
// BJoystick object directly.
//===============================================
void
GameStick::GetAbsolutePosition(short &x, short &y)
{
	fJoystick.Update();
	
	x = fJoystick.horizontal;
	y = fJoystick.vertical;

	// Record this position and train again
	fLastX = x;
	fLastY = y;
	Train();
}

void
GameStick::GetZeroPosition(short &x, short &y)
{
	x = fZeroX;
	y = fZeroY;
}

//===============================================
// Function: CLAMPLOW
//
// Make sure that the x value is not less than the
// specified low mark.  Since the joystick is self
// training, this helps to ensure that there aren't
// wild fluctuations in the range of values when the
// stick is first moved.
//===============================================
static inline void
CLAMPLOW(float &x, const float lowmark)
{
	if (x < lowmark)
		x = lowmark;
}

//===============================================
//	Method: Train
//
//	Training the joystick involves using the
//	the latest information to generate new x and
//	y min and max values.  This auto training allows
//	the joystick to learn its limits over time 
//	without having to tell the user to move the 
//	stick around to calibrate it in the beginning
//	of a game.  
//
//	Training does not find the zero value.  Here we
//	assume the zero position has already been found
//	by the constructor code, which assumes the stick 
//	is in a neutral position when it starts.
//===============================================
void
GameStick::Train()
{
	if (fLastX < fMinX)
		fMinX = fLastX;
	if (fLastX > fMaxX)
		fMaxX = fLastX;

	if (fLastY < fMinY)
		fMinY = fLastY;
	if (fLastY > fMaxY)
		fMaxY = fLastY;
		
	fMinXRange = fZeroX - fMinX;
	fMinYRange = fZeroY - fMinY;

	fMaxXRange = fMaxX - fZeroX;
	fMaxYRange = fMaxY - fZeroY;
	
	// Create a good dead buffer zone
	CLAMPLOW(fMinXRange, 50);
	CLAMPLOW(fMinYRange, 50);
	CLAMPLOW(fMaxXRange, 50);
	CLAMPLOW(fMaxYRange, 50);
}

//===============================================
//	Method: IsButtonPressed
//
//	The method tells you whether a particular button
//	is pressed or not.  The whichbutton parameter
//	is a bitfield constructed by ORing together the
//	enums for BUTTON1, and BUTTON2.
//
//	Returns:
//		This returns either true or false for the
//		buttons that were passed in the whichbutton
//		field.
//
//	Note: When you get raw joystick data, these 
//		values are the inverse of the logical button
//		state.  That is, when a button is pressed, 
//		the raw data will show a 0, when a button is
//		not pressed, it will show a 1.  This routine
//		returns the logical state, which is the opposite
//		of these raw numbers.
//===============================================

bool	
GameStick::IsButtonPressed(const long whichbutton)
{
	fJoystick.Update();
	
	if (whichbutton & BUTTON1)
	{
		if (!fJoystick.button1)	
			return 1;
	}
	
	if (whichbutton & BUTTON2)
	{
		if (!fJoystick.button2)	
			return 1;
	}
	
	return 0;
	
}



//========================================================
// Class: GameStickThread
//
// This class is meant to feed commands to the game engine.
// The joystick is sampled 15 times a second.  Whenever there
// is input that indicates the user wants to move in a particular
// direction, a command is sent to the game engine.
//
// Sampling at 15 times a second should be fast enough to be
// responsive to the fastest game play.  A better method would be
// to make the sampling double the rate of the game speed.
//========================================================
GameStickThread::GameStickThread()
	: ATicker(15)
{
}


//========================================================
// Method: Tick
//
// The Tick method is called once per time interval of 
// the clock.
//
// The idea here is to send a command to the game engine
// if the user is pressing the joystick far enough in
// a particular direction.  The threshold is set at such
// a level that the user doesn't have to push the stick all
// the way to the edge, but it will not trigger if the bump
// it just a tiny bit.
//
// The x direction is favored over the y direction if they
// are both activated.
//
// This method works particularly well with the gamepad type
// of joystick where there is a thumb control.
//========================================================
void
GameStickThread::Tick(const double tickCount)
{
	float x, y;
	float threshold = 0.6;
	//EGameCommand command = MPNONE;
	long command = MPNONE;
	
	// Get the joystick position
	fGameStick.GetPosition(x, y);
	
	// Figure out which direction to go
	if (y > threshold)
		command = MPMOVEUP;
	else if (y < -threshold)
		command = MPMOVEDOWN;
		
	if (x > threshold)
		command = MPMOVERIGHT;
	else if (x < -threshold)
		command = MPMOVELEFT;
		
	//printf("GamestickThread: %d\n",command);
	// Tell the gameengine to go in that direction
	if (command != MPNONE)
	{
			//printf("GamestickThread: %d\n",command);
			gGameEngine->PostMessage(command);
	}
}
