#include "gameengine.h"
#include "gameview.h"
#include "gamestick.h"
#include "gameclock.h"

#include <malloc.h>
#include <stdlib.h>
#include <stdio.h>
#include <Debug.h>

GameEngine *gGameEngine = NULL;

GameEngine::GameEngine(const unsigned long width, const unsigned long height)
	:BApplication ('BPED'),
	fWidth(width),
	fHeight(height)
{
	//SET_DEBUG_ENABLED(TRUE);

	fIsQuitting = FALSE;
	gGameEngine = this;
	fGameState = new GameState(fWidth, fHeight);
}


void
GameEngine::ReadyToRun()
{
	fGamePresenter = new GameView(this);
	fGameStickThread = new GameStickThread();
	fGameClock = new GameClock(this,fGameState->speed);
}

bool
GameEngine::QuitRequested()
{
	fIsQuitting = TRUE;
	
	// Pause the game
	Pause();

	// Stop ticking
	delete fGameClock;

	// Get rid of joystick input
	delete fGameStickThread;
	
	
	// Quit the presentation to give up the window
	fGamePresenter->Quit();
			
	return TRUE;
}


GameEngine::~GameEngine()
{
	// Don't delete this thing.  It's a window and will be
	// taken care of elsewise.
	//delete fGamePresenter;
}


void
GameEngine::MessageReceived(BMessage *message)
{
	#ifdef DEBUG
	printf("GameEngine::MessageReceived - BEGIN (%d)\n", message->what);
	#endif
	
	switch (message->what) 
	{
		case MPQUIT:
			//printf("GameEngine::MessageReceived - posting quit to be_app\n");
			be_app->PostMessage(B_QUIT_REQUESTED);
		break;
		
		case MPCONTINUE:
			Continue();
		break;
		
		case MPPAUSE:
			Pause();
		break;
		
		case MPTOGGLEPAUSE:
			TogglePause();
		break;
		
		case MPMOVEUP:
			MoveUp();
		break;
		
		case MPMOVEDOWN:
			MoveDown();
		break;
		
		case MPMOVELEFT:
			MoveLeft();
		break;
		
		case MPMOVERIGHT:
			MoveRight();
		break;
		
		default:
			printf("GameEngine::MessageReceived - Unknown Message Received: %d\n", message->what);
			BApplication::MessageReceived(message);
		break;
	}
	
	#ifdef DEBUG
	printf("GameEngine::MessageReceived - END\n");
	#endif
}

void
GameEngine::GetSize(unsigned long &width, unsigned long &height)
{
	width = fWidth;
	height = fHeight;
}


void 
GameEngine::Pulse()
{
	#ifdef DEBUG
	printf("GameEngine::Tick - BEGIN\n");
	printf("GameState: %d\n", fGameState->gamestate);
	#endif
	
	// If we're quitting, just return immediately
	if (fIsQuitting)
		return;
		
    // set up new game if NEWLEVEL is set in gamestate, & clear flag
    if (fGameState->gamestate & NEWLEVEL) 
	{
		fGameState->NewGame(); 
		fGameState->gamestate &= ~NEWLEVEL;
	}

		/* only do anything if PAUSED is false in gamestate */
		if (!(fGameState->gamestate & PAUSED)) 
		{
			/* if the snake has just died, do special animation */
			if (fGameState->gamestate & DYING) 
			{
				fGamePresenter->update_image_dead();
				fGameState->dcount++;

				/* if animation has reached the edge of the screen, restart game */
				if (fGameState->dcount>((fGameState->width>fGameState->height)?(fGameState->width/2):(fGameState->height/2))) 
				{
					fGameState->dcount=0; 
					fGameState->gamestate|=NEWLEVEL; 
					fGameState->gamestate&=~DYING; 
				}

      		} else if (fGameState->gamestate & WIN) 
			{
        		fGamePresenter->update_image_won();
        		fGameState->dcount++;

        		/* if animation has reached the edge of the screen, restart game */
        		if (fGameState->dcount>((fGameState->width>fGameState->height)?(fGameState->width/2):(fGameState->height/2))) 
				{
					fGameState->dcount=0; 
					fGameState->gamestate|=NEWLEVEL; 
					fGameState->gamestate&=~WIN; 
				}
			} else 
			{
				if (fGameState->gamestate & INGAME) 
				{
					fGameState->update_game();
					fGamePresenter->update_image();
       			 }
     		 }
   		 }
	#ifdef DEBUG
	printf("GameEngine::Tick - END\n");
	#endif
}

void
GameEngine::MoveLeft()
{
	fGameState->sdir=HDLEFT;
}

void
GameEngine::MoveRight()
{
	fGameState->sdir=HDRIGHT;
}

void
GameEngine::MoveUp()
{
	fGameState->sdir=HDUP;
}

void
GameEngine::MoveDown()
{
	fGameState->sdir=HDDOWN;
}

void
GameEngine::Pause()
{
	fGameState->gamestate |= PAUSED;
}

void
GameEngine::Continue()
{
	fGameState->gamestate &= ~PAUSED;
}

void
GameEngine::TogglePause()
{
	if (fGameState->gamestate & PAUSED)
		Continue();
	else
		Pause();
}

void
GameEngine::TurnClockwise()
{
	fGameState->sdir=((fGameState->sdir+1)%4)+4;
}

void
GameEngine::TurnCounterClockwise()
{
	fGameState->sdir=((fGameState->sdir-1)%4)+4;
}


//================================================
//	Class: GameState
//
// Maintains the state of the game
//================================================
GameState::GameState(const unsigned long aWidth, const unsigned long aHeight) 
{
	#ifdef DEBUG
	printf("GameState::GameState() - BEGIN\n");
	#endif
	
	width = aWidth;
	height = aHeight;
	
	/* initialise the games state */
	Initialize();

	NewGame();
		
	#ifdef DEBUG
	printf("GameState::GameState() - END\n");
	#endif
}

GameState::~GameState() 
{
	free(map);	// Free map data
}


void
GameState::Initialize()
{
	#ifdef DEBUG
	printf("GameState::Initialize() - BEGIN\n");
	#endif
	
	difficulty=0;		// difficulty level
	walls=0;			// can you wrap around?
	
	map=(enum mtype *) malloc(sizeof(enum mtype)*width*height);

	gamestate=PAUSED | NEWLEVEL | INGAME;
	
	#ifdef DEBUG
	printf("GameState::Initialize() - END\n");
	#endif
}

void 
GameState::NewGame()
{
	int c;
	
	/* set snake starting position */
	spos.x=width/2; spos.y=height/2;

	/* set snake starting direction */
	sdir=HDUP;

	/* setup snake's movement history buffer */
	shistoff=0; 
	slength=5;
	
	speed = 7;
	
	for (c=0; c<=slength; c++) 
	{
		shist[(shistoff-c+HISTLENGTH)%HISTLENGTH].x=spos.x;
		shist[(shistoff-c+HISTLENGTH)%HISTLENGTH].y=spos.y+c;
	}
	
	/* setup the map with mushrooms, skulls and hearts */
	c=((difficulty*.3)+1)*20;
	c=(c*width*height)/256;
	mushnum=c;
	
	make_map(c,c/4,c/40);
}

/* 
	function to set up the map, with numbers of each object to be placed
	passed to it 
*/

void 
GameState::make_map(int mush,int death,int hearts) 
{

  int x,y,p,freenum;
  enum mtype mval; 

	//printf("GameState::make_map() - BEGIN\n");
  /* setup random seed */
  p=rand();

  /* set initial number of free grid spaces */
  freenum=width*height; 

	//printf("GameState::make_map() - empty\n");
  /* set the whole map to be EMPTY */
  for (y=0; y<height; y++) {
    for (x=0; x<width; x++) {
    mval = EMPTY;
    *(map+(x*height)+y)=mval;
   }
  }

	//printf("GameState::make_map() - mushrooms\n");
	/* populate the world with mushrooms */
	while (mush>0 && freenum>5) 
	{
		/* set the x and y positions to random values within the map */
		x=(p<0) ? (-p)%width : p%width;
		p^=(p<<13) | (p>>19);
		p+=1;
		y=(p<0) ? (-p)%height : p%height;

		/* 
		if map point is empty, put a mushroom there, decrement the number
		still needing to be placed by one, and decrement the number of 
		free spaces 
		*/
		if (*(map+(x*height)+y)==EMPTY) 
		{
			*(map+(x*height)+y) = MUSH;
			mush--;
			freenum--;
		}

		p^=(p<<13) | (p>>19);
		p+=1;

	}

	//printf("GameState::make_map() - skulls\n");
	/* sprinkle the world with skull tokens */
	while (death>0 && freenum>5) 
	{

		/* all as above */
		x=(p<0) ? (-p)%width : p%width;
		p^=(p<<13) | (p>>19);
		p+=1;
		y=(p<0) ? (-p)%height : (p)%height;

		if (*(map+(x*height)+y)==EMPTY) 
		{
			*(map+(x*height)+y)=KILL;
			death--;
			freenum--;
		}

		p^=(p<<13) | (p>>19);
		p+=1;
	}

	//printf("GameState::make_map() - hearts\n");
	/* sprinkle the world with heart tokens */
	while (hearts>0 && freenum>5) 
	{

		/* all as above */
		x=(p<0) ? (-p)%width : p%width;
		p^=(p<<13) | (p>>19);
		p+=1;
		y=(p<0) ? (-p)%height : (p/65536)%height;

		if (*(map+(x*height)+y)==EMPTY) 
		{
			*(map+(x*height)+y)=HEART;hearts--;freenum--;
		}

		p^=(p<<13) | (p>>19);
		p+=1;

	}
	
	//printf("GameState::make_map() - END\n");
}

/* update various game things */
void 
GameState::update_game() 
{
  enum mtype newval;
  int count;

  /* change snake's postition, dependant on snakes direction */
  switch (sdir) {

    case HDUP: spos.y--; break;

    case HDRIGHT: spos.x++; break;

    case HDDOWN: spos.y++; break;

    case HDLEFT: spos.x--; break;

  }

  /* move the history buffer forward one place */
  shistoff++;
  shistoff%=HISTLENGTH;

	if (((spos.x<0)||(spos.y<0)||(spos.x>=width)||(spos.y>=height))&&walls) 
	{
		gamestate|=DYING; 
		dcount=0; 
		newval=EMPTY;
		spos.x=(spos.x>=width) ? width-1: spos.x;
		spos.x=(spos.x<0) ? 0: spos.x;
		spos.y=(spos.y>=height) ? height-1: spos.y;
		spos.y=(spos.y<0) ? 0: spos.y;
	}

	/* make sure position is positive */
	spos.x = (spos.x<0) ? spos.x+width : spos.x;
	spos.y = (spos.y<0) ? spos.y+height : spos.y;

	/* wrap around position */
	spos.x%=width;
	spos.y%=height;

	/* store position in history buffer */
	shist[shistoff]=spos;

	/* check to see what snake has landed on */
	switch (*(map+(spos.x*height)+spos.y)) 
	{

		case EMPTY: 
			newval=EMPTY; 
		break; /* do nothing */

		/* increase length, remove mushroom & decrement number of mushrooms left */
		case MUSH: 
			slength++; 
			newval=EMPTY; 
			mushnum--; 
		break;

		case KILL: 
			gamestate|=DYING; 
			dcount=0; 
			newval=EMPTY;
		break; /* kill snake */

		case HEART: 
			slength/=2; 
			newval=EMPTY; 
		break; /* reduce snake length */

		default: 
			newval=*(map+(spos.x*height)+spos.y); 
		break; /* do nothing */
	}

	*(map+(spos.x*height)+spos.y)=newval; /* update map */

	/* check history buffer to see if the snake has collided with its own tail */
	for (count=1; count<slength; count++) 
	{
		if (  ((spos.x)==(shist[(shistoff+(HISTLENGTH-count))%HISTLENGTH].x))
       		&& ((spos.y)==(shist[(shistoff+(HISTLENGTH-count))%HISTLENGTH].y)) ) 
		{ 
			gamestate|=DYING; 
			dcount=0; 
		}
	}

	if (mushnum<=0) 
	{ 
		gamestate|=WIN; 
		difficulty++; 
	}
    
}



