/*
	bone_proto.h
	
	networking protocol module template
	
	Copyright 1999-2000, Be Incorporated, All Rights Reserved.
*/

#ifndef H_BONE_PROTO
#define H_BONE_PROTO

#include <BeBuild.h>
#include <SupportDefs.h>
#include <KernelExport.h>
#include <module.h>
#include <bone_data.h>
#include <net/route.h>

#ifdef __cplusplus
extern "C" {
#endif

struct bone_endpoint;
struct bone_proto_node;

typedef struct bone_proto_info
{
	struct module_info info;
	
	/*
	 * called by API driver on a new endpoint
	 */
	status_t (*init_protocol)(struct bone_proto_node *you);
	
	/*
	 * called by API driver when deleting an endpoint (after all refs
	 * to the socket have been closed; analagous to the "free" device hook) 
	 */
	status_t (*uninit_protocol)(struct bone_proto_node *you);

	/*
	 * called by API driver when a user calls "close" on the socket
	 */
	status_t (*close)(struct bone_proto_node *you);
	
	/*
	 * called by API driver to create an end-to-end link
	 */
	status_t (*connect)(struct bone_proto_node *you, struct sockaddr *them);
	
	/*
	 * called by API driver when there is data to send.
	 * protocols below the transport layer need to be prepared for "you"
	 * to be 0, since some protocols (e.g. tcp, icmp, etc) will send data
	 * without being associated with an endpoint.  An example would
	 * be TCP reflecting a RST to an inbound SYN segment for which there
	 * is no listener.
	 *
	 * note: network layer protocols (IP, etc) free the passed-in data
	 * on success or failure.
	 */
	status_t (*send_data)(struct bone_proto_node *you, bone_data_t *data);

	/*
	 * Called by the API driver to determine how many bytes may be sent without
	 * blocking
	 */
	int32 (*send_avail)(struct bone_proto_node *you);

	/*
	 * called by the API driver to read data.  Data should be copied into the "into"
	 * iovecs accordingly.  bone_util_info_t::dequeue_fifo_data does this automatically,
	 * for example.
	 */
	status_t (*read_data)(struct bone_proto_node *you, struct iovec *into, int numiov, int32 flags, bigtime_t timeout, struct sockaddr *addr, int len);
	
	/*
	 * called by the API driver to determine how many bytes of data may be immediately
	 * read without blocking
	 */	
	int32  (*read_avail)(struct bone_proto_node *you);


	/*
	 * called by datalink layer when data arrives
	 * note: receiver (api driver on success, protocol on error) frees data 
	 */
	status_t (*receive_data)(bone_data_t *data);
	

	/*
	 * called by API driver to block for a new connection
	 */
	status_t (*accept)(struct bone_proto_node *you, struct bone_proto_node *clone);
	
	/*
	 * called by API driver on ioctl or setsockopt
	 */
	status_t (*control)(struct bone_proto_node *you, int level, int cmd, void *arg, int arglen);

	/*
	 * called from higher-level protocols to get route MTU
	 */
	int32	 (*getMTU)(struct bone_proto_node *you, struct sockaddr *sa);
	
	/*
	 * called from higher-level protocols to get route 
	 */
	route_t  (*get_route)(struct bone_proto_node *you, struct sockaddr *addr);
	
	/*
	 * called from api dricer to bind to local address
	 */
	status_t (*bind)(struct bone_proto_node *you, struct sockaddr *sa);

	/*
	 * called from api driver to unbind from local address
	 */
	status_t (*unbind)(struct bone_proto_node *you, struct sockaddr *sa);
	
	/*
	 * called from api driver to specify listen queue for ibd connections
	 */	
	status_t (*listen)(struct bone_proto_node *you, int count);
	
	/*
	 * called from api driver to close one direction of a connection (half-close)
	 */
	status_t (*shutdown)(struct bone_proto_node *you, int direction);
	
	/*
	 * interface for error codes (ICMP, etc) to be passed back to protocols
	 * from the network layer
	 */
	status_t (*error)(int32 error_code, bone_data_t *error_data);

	/*
	 * interface for error replies (ICMP, etc) to be sent from protocols
	 * to other sites.  Error codes can be passed, or a data ptr and length
	 * may be passed.  If error_data is non-0 error_code will be assumed to
	 * be a length, otherwise a code.
	 */
	status_t (*error_reply)(struct bone_proto_node *you, bone_data_t *caused_error, uint32 error_code, void *error_data);
	
} bone_proto_info_t;

/*
 *	each protocol gets passed one of these at init and for most calls
 * 	to save state and determine who to hand data off to.
 */
typedef struct bone_proto_node
{
	struct bone_endpoint 	*endpoint;
	bone_proto_info_t 		*proto;
	void			  		*proto_state;
	struct bone_proto_node	*next;
	struct bone_proto_node	*prev;
} bone_proto_node_t;

#ifdef __cplusplus
}
#endif


#endif /* H_BONE_PROTO */