/*
	bone_loop.c
	
	loopback networking interface module
	
	Copyright 1999-2000, Be Incorporated, All Rights Reserved.
*/

#include "bone_util.h"
#include "bone_data.h"
#include "bone_interface.h"
#include "bone_datalink.h"
#include "bone_frame.h"
#include <string.h>
#include <net/if.h>
#include <net/if_types.h>
#include <OS.h>
#include <errno.h>
#include <bone_endpoint.h>
#include <malloc.h>

static bone_util_info_t *gUtil;

#define LOOP_MTU 65536 
#define LOOP_FRAME_SIZE 4

typedef struct bone_loopback_interface
{
	bone_interface_info_t	interface;
	bone_benaphore_t		lock;
	bone_fifo_t				*fifo;
} bone_loopback_interface_t;

static void set_function_pointers(bone_loopback_interface_t *iface);

static ifnet_t *init_interface(bone_datalink_info_t *dli, struct bone_interface_params *params)
{
	status_t rc = B_ERROR;
	int index;
	bone_loopback_interface_t *liface = 0;
	ifnet_t *iface;
	bone_frame_info_t    *fit = 0;
	char name[255];
	
	PROF_IN
	/*
	 *  A real NIC module would loop through all the busses it could be on
	 *  (via get_nth_pci_info, etc.) and instantiate an ifnet_t for each
	 *  card it finds, calling the datalink's register_interface with each.
	 */
	
	iface = (ifnet_t *) gUtil->malloc(sizeof(ifnet_t));
	if(iface == 0)
	{
		PROF_OUT
		return 0;
	}
	/*
	 * A real interface would fill out its if_media structure here as well.
	 */
	
	liface = (bone_loopback_interface_t *) gUtil->malloc(sizeof(bone_loopback_interface_t));
		
	if(liface == 0)
		goto error0;
				
	iface->if_mtu = LOOP_MTU;
	iface->if_type = IFT_LOOP;
	
	if(gUtil->create_benaphore( &liface->lock, "loopback lock") < 0)
		goto error1;
					
	/*
	 * Add the function pointers.  Pretty durn important :-)
	 */	
	set_function_pointers(liface);
	
	iface->if_info = (bone_interface_info_t *) liface;

	/*
	 * no datalink == no framing
	 */
	if(dli != 0)
	{	
		if(get_module("network/framings/bone_loop_frame", (struct module_info **) &fit) != B_OK)
			goto error2;
		
		iface->if_frame = fit;
	}	
	
	if((iface->if_name = gUtil->malloc(16)) == 0)
		goto error4;
	
	strcpy(iface->if_name, "loop0");
	
	strcpy(name, iface->if_name);
	strcat(name, " data fifo");
	
	liface->fifo = gUtil->new_fifo(name, LOOP_MTU * 10);

	if(dli != 0)
	{
		if(dli->register_interface(iface) != B_NO_ERROR)
			goto error4;
	}

#if DEBUG > 1
	dprintf("bone_loopback: init successful\n");
#endif

	iface->if_flags |= IFF_LOOPBACK | IFF_NOARP; 
	PROF_OUT
	return iface;

error5:
	gUtil->delete_fifo(liface->fifo);
error4:
	gUtil->free(iface->if_name);
error3:
	put_module("network/framings/bone_loop_frame");
error2:
	gUtil->delete_benaphore(&liface->lock);
error1:
	gUtil->free(liface);
error0:
	gUtil->free(iface);
	PROF_OUT
	return 0;
}

/*
 * bring the interface up.
 */
static status_t up(ifnet_t *iface)
{

#if DEBUG > 1
	dprintf("bone_loopback: up\n");
#endif
	iface->if_flags |= IFF_UP;
	return B_NO_ERROR;
}


/*
 * bring the interface down.
 */
static void down(ifnet_t *iface)
{
	bone_loopback_interface_t *liface = 0;
	
	PROF_IN
	if(iface == 0)
	{
		PROF_OUT
		return;
	}	
	iface->if_flags &= ~IFF_UP;
	
	liface = (bone_loopback_interface_t *) iface->if_info;
	
	gUtil->clear_fifo(liface->fifo);
	PROF_OUT
#if DEBUG > 1
	dprintf("bone_loopback: down\n");
#endif
}


/*
 * send framed MTU-sized chunks
 */

static status_t bone_loop_send_data(ifnet_t *iface, bone_data_t *data)
{
	bone_loopback_interface_t *liface = 0;
	status_t rc = B_ERROR;
	bone_data_t *bdt = 0;
	
	PROF_IN
	if(iface == 0 || data == 0)
	{
		PROF_OUT
		return EINVAL;
	}	
	liface = (bone_loopback_interface_t *) iface->if_info;
	
	bdt = gUtil->clone_data(data);
	
	if(bdt == 0)
		goto error0;


	if(gUtil->enqueue_fifo_data(liface->fifo, bdt) != B_NO_ERROR)
		goto error1;

	iface->if_opackets++;
	iface->if_obytes+=gUtil->get_data_size(bdt);
	PROF_OUT
	return B_NO_ERROR;

error1:
	gUtil->delete_data(bdt);
error0:
	iface->if_oerrors++;
	PROF_OUT
	return B_ERROR;
}


static status_t bone_loop_receive_data(ifnet_t *iface, bone_data_t **data)
{
	bone_loopback_interface_t *liface = 0;
	status_t rc = B_ERROR;
	bone_fifo_t *fifo = 0;
	
	PROF_IN
	if(iface == 0 ||data == 0)
	{
		PROF_OUT
		return EINVAL;
	}
	if(!(iface->if_flags & IFF_UP))
	{
		PROF_OUT
		return B_ERROR;	
	}
	liface = (bone_loopback_interface_t *) iface->if_info;
	
	
	if(liface != 0)
	{
		fifo = liface->fifo;
		
		*data = 0;
		
		if(fifo != 0)
			*data = gUtil->pop_fifo_data(fifo, B_INFINITE_TIMEOUT);
			
		if(*data != 0)
		{
			rc = B_NO_ERROR;
#if DEBUG > 1
		dprintf("bone_loopback: receive successful\n");
#endif
		} else {
			rc = B_ERROR;
#if DEBUG > 1
		dprintf("bone_loopback: receive error\n");
#endif

		}
	}
		
	if(rc == B_NO_ERROR)
	{
		iface->if_ipackets++;
		iface->if_ibytes+=gUtil->get_data_size((*data));

	} else {
		iface->if_ierrors++;
	}
	PROF_OUT
	return rc;
}


static status_t bone_loop_control(ifnet_t *iface, bone_endpoint_t *sock, int cmd, void *arg, int arglen)
{
	return B_OK;
}

static status_t	 bone_loop_setMTU(ifnet_t *iface, uint32 mtu)
{
	
	iface->if_mtu = mtu;

	return B_NO_ERROR;
}


static status_t bone_loop_setPromiscuous(ifnet_t *iface, int on)
{
	return EOPNOTSUPP;
}

static status_t   bone_loop_setMedia(ifnet_t *iface, uint32 media)
{
	return EOPNOTSUPP;
}



static status_t bone_loop_getHardwareAddr(ifnet_t *iface, bone_iface_hwaddr_t *addr)
{
	PROF_IN
	if(addr != 0 && iface != 0)
	{
		memcpy(addr->hw_addr,iface, sizeof(iface));
		addr->addrlen = sizeof(iface);
	}
	PROF_OUT
	return B_OK;
}


static status_t bone_loop_getMulticastAddrs(ifnet_t *iface, bone_iface_hwaddr_t **addrs, int numaddr)
{
	return EOPNOTSUPP;
}


static status_t bone_loop_setMulticastAddrs(ifnet_t *iface, bone_iface_hwaddr_t **addrs, int numaddr)
{
	return EOPNOTSUPP;
}



static status_t std_ops(int32 op, ...)
{
	status_t err;
	
	switch(op) {		
		case B_MODULE_INIT:
#ifdef BONE_SYM
			if(load_driver_symbols("bone_loopback") == B_OK) {
				kprintf("bone_loopback: loaded symbols. The funk be with you!\n");
			} else {
				kprintf("bone_loopback: no symbols.  The funk is not on your side!\n");
			}
#endif
			if(get_module(BONE_UTIL_MOD_NAME, (struct module_info **)&gUtil) == B_OK)
			{
				err = B_NO_ERROR;
			} else {
				err = B_ERROR;
				kprintf("bone_loopback: Fatal Error: couldn't load utility module!\n");				
			}
		break;
				
		case B_MODULE_UNINIT:
			put_module(BONE_UTIL_MOD_NAME);
			err =  B_NO_ERROR;
		break;
		
		default:
			err = -1;
		break;
	}
	return err;
}



bone_interface_info_t bone_loopback_module = {
	{
		"network/interfaces/bone_loopback",
		B_KEEP_LOADED,
		&std_ops
	},
	init_interface,
	up,
	down,
	bone_loop_send_data,
	bone_loop_send_data,
	bone_loop_receive_data,
	bone_loop_control,
	bone_loop_setMTU,
	bone_loop_setPromiscuous,
	bone_loop_setMedia,
	bone_loop_getHardwareAddr,
	bone_loop_getMulticastAddrs,
	bone_loop_setMulticastAddrs
};

static void set_function_pointers(bone_loopback_interface_t *iface)
{
	memcpy(&iface->interface, &bone_loopback_module, sizeof(bone_interface_info_t));
}

_EXPORT bone_interface_info_t *modules[] = {
        &bone_loopback_module,
        NULL
};
