//******************************************************************************
// 
//	File:		pulse.cpp
//
//	Description:	Pulse(tm) shows CPU usage on BeOS systems
//
//******************************************************************************
 
/*
	Copyright 1999, Be Incorporated.   All Rights Reserved.
	This file may be used under the terms of the Be Sample Code License.
*/

#define DEBUG 1
#include <Debug.h>
#include <stdio.h>
#include <string.h>
#include <OS.h>
#include "progressbar.h"
#include "pulse.h"
#include <Dragger.h>
#include "Pictures"
#include <Screen.h>
#include <stdlib.h>

#ifndef _ALERT_H
#include <Alert.h>
#endif


//
// Globals
//

extern const char *app_signature;


BPicture *BitmapToPicture(BBitmap *bitmap);

static int CalcCPUSpeed(const system_info * sysinfo)
{
	int target = sysinfo->cpu_clock_speed / 1000000;
	int frac = target % 100;
	int delta = -frac;
	int at = 0;
	int freqs[] = { 100, 50, 25, 75, 33, 67, 20, 40, 60, 80, 10, 30, 70, 90 };
	uint32 ix;
	for (ix=0; ix<sizeof(freqs)/sizeof(freqs[0]); ix++) {
		int ndelta = freqs[ix]-frac;
		if (abs(ndelta) < abs(delta)) {
			at = freqs[ix];
			delta = ndelta;
		}
	}
	return target+delta;
} 

// --------------------------------------------------------------

TPulseWindow::TPulseWindow(BRect r)
	 :BWindow(r, "CPU Monitor", B_TITLED_WINDOW, B_NOT_RESIZABLE | B_NOT_ZOOMABLE)
{
	SetPulseRate(200000);
}

// --------------------------------------------------------------
bool TPulseWindow::QuitRequested()
{
	be_app->PostMessage(B_QUIT_REQUESTED);
	return TRUE;
}

//---------------------------------------------------------------
// T P U L S E V I E W
//---------------------------------------------------------------

TPulseView::TPulseView(BRect r)
	 :BView(r, "PulseView", B_FOLLOW_NONE, B_WILL_DRAW | B_PULSE_NEEDED)
{
	BRect		theRect;
	BBitmap*	normalBits;
	BBitmap*	pushedBits;
	BPicture	*normalPict = NULL;
	BPicture	*pushedPict = NULL;
	int			i;

	get_system_info(&fSysInfo);

	theRect.Set(0,0,63,62);
	fAnim1 = new BBitmap(theRect, B_COLOR_8_BIT);
#if __POWERPC__
	fAnim1->SetBits(Anim1,11718,0,B_COLOR_8_BIT);
#endif
#if __INTEL__
	if(fSysInfo.cpu_type < B_CPU_AMD_X86) {
		fAnim1->SetBits(IntelLogo,11718,0,B_COLOR_8_BIT);
	}
	else {
		fAnim1->SetBits(BlankLogo,11718,0,B_COLOR_8_BIT);
	}
#endif

	//
	// Add CPU activity bars
	//
	
	// allocate progress bars and button pointers...
	fCPUs = new TProgressBar*[ fSysInfo.cpu_count ];
	fButtons = new CPUButton*[ fSysInfo.cpu_count ];
	

	//
	// Set up the CPU activity bars and buttons
	//
	
	for( i=0; i<fSysInfo.cpu_count; i++ ) {
		theRect.Set( PROGRESS_MLEFT,
				PROGRESS_MTOP + ITEM_OFFSET*i,
				PROGRESS_MLEFT + TProgressBar::PROGRESS_WIDTH,
				PROGRESS_MTOP + ITEM_OFFSET*i + TProgressBar::PROGRESS_HEIGHT
			);
		fCPUs[i] = new TProgressBar( theRect, "CPU Progress Bar", 0 );
		fCPUs[i]->SetPrev( fSysInfo.cpu_infos[i].active_time );
		AddChild( fCPUs[i] );
		
		theRect.Set( 0, 0, 17, 17 );
	
		normalBits = new BBitmap(theRect, B_COLOR_8_BIT);
		normalBits->SetBits(ButUp, normalBits->BitsLength(), 0, B_COLOR_8_BIT);
		AddID((uchar *)normalBits->Bits(), normalBits->BytesPerRow(), i);
		pushedBits = new BBitmap(theRect, B_COLOR_8_BIT);
		pushedBits->SetBits(ButDown, pushedBits->BitsLength(), 0, B_COLOR_8_BIT);
		AddID((uchar *)pushedBits->Bits(), pushedBits->BytesPerRow(), i);
		
		normalPict = BitmapToPicture(normalBits);
		pushedPict = BitmapToPicture(pushedBits);
		
		delete(normalBits);
		delete(pushedBits);

		theRect.Set( CPUBUTTON_MLEFT,
				CPUBUTTON_MTOP + ITEM_OFFSET*i,
				CPUBUTTON_MLEFT + CPUButton::CPUBUTTON_WIDTH,
				CPUBUTTON_MTOP + ITEM_OFFSET*i + CPUButton::CPUBUTTON_HEIGHT
			);
		fButtons[i] = new CPUButton( theRect, "CPU Button", "", i,
				new BPicture(*normalPict),
				new BPicture(*pushedPict)
			);
		AddChild( fButtons[i] );
		//
		//	if there is only 1 cpu it will be hidden below
		//	thus, no need to add the dragger as it will still
		//	be visible when replicants are turned on
		//
		if( fSysInfo.cpu_count > 1 ) {
			BDragger	*dw;
			BRect		rrr;
#if 1
			rrr = theRect;
			rrr.top = rrr.bottom;
			rrr.left = rrr.right;
			rrr.bottom += 7;
			rrr.right += 7;
			rrr.OffsetBy(-1,-1);
			dw = new BDragger(rrr, fButtons[i], 0);
			AddChild(dw);
#else
			rrr = theRect;
			rrr.OffsetTo(B_ORIGIN);
			rrr.top = rrr.bottom - 7;
			rrr.left = rrr.right - 7;
			dw = new BDragger(rrr, fButtons[i], 0);
			fButtons[i]->AddChild(dw);
#endif
		}
	}
	
	if( fSysInfo.cpu_count == 1 ) {
		fCPUs[0]->MoveBy( -3, 12 );
		fButtons[0]->Hide();
	}

	delete(normalPict);
	delete(pushedPict);
}

// --------------------------------------------------------------
TPulseView::~TPulseView()
{
	delete fAnim1;
	delete fButtons;
	delete fCPUs;
}
	

// --------------------------------------------------------------
void TPulseView::AddID(uchar* bits, int32 row_bytes, int32 id)
{
	int		x, y;
	uchar	*num = NULL;

	switch(id + 1) {
		case 0: num = num0; break;
		case 1: num = num1; break;
		case 2: num = num2; break;
		case 3: num = num3; break;
		case 4: num = num4; break;
		case 5: num = num5; break;
		case 6: num = num6; break;
		case 7: num = num7; break;
		case 8: num = num8; break;
		case 9: num = num9; break;
	}

	for (x = 0; x < 4; x++) {
		for (y = 0; y < 6; y++) {
			if (num[y * 4 + x])
				bits[row_bytes * (5 + y) + 5 + x] = 0x0;
		}
	}
}

// --------------------------------------------------------------
BRect TPulseView::CalcViewSize()
{
	system_info si;
	get_system_info( &si );
	
	float ht = PROGRESS_MTOP + PROGRESS_MBOTTOM + si.cpu_count*ITEM_OFFSET;
	
	if( PULSEVIEW_MIN_HEIGHT > ht ) {
		ht = PULSEVIEW_MIN_HEIGHT;
	}
	
	return BRect( 0, 0, PULSEVIEW_WIDTH, ht );
}


// --------------------------------------------------------------
// Draw and label the graph. 
// --------------------------------------------------------------
void TPulseView::Draw(BRect updateRect)
{
	BRect		frame;
	char		buf[500];
	char*		processor;
	float		width;
#if __INTEL__
	char*		vendor;
#endif
	
	frame = Bounds();
	//
	// Black frame
	///
	SetHighColor(0,0,0);
	frame.right--;
	frame.bottom--;
	StrokeRect(frame);

	//
	// Bevelled edges
	//
	SetHighColor(255,255,255);
	StrokeLine(BPoint(1,1),BPoint(frame.right-1,1));
	StrokeLine(BPoint(1,1),BPoint(1,frame.bottom-1));
	SetHighColor(80,80,80);
	StrokeLine(BPoint(frame.right,1),BPoint(frame.right,frame.bottom));
	StrokeLine(BPoint(2,frame.bottom),BPoint(frame.right-1,frame.bottom));
	//
	// Dividing line
	//
	SetHighColor(96,96,96);
	StrokeLine(BPoint(1,frame.bottom+1),BPoint(frame.right,frame.bottom+1));
	SetHighColor(255,255,255);
	StrokeLine(BPoint(1,frame.bottom+2),BPoint(frame.right,frame.bottom+2));
	DrawBitmap(fAnim1,BPoint(10,10));		// Processor picture
#if __INTEL__
	switch(fSysInfo.cpu_type & B_CPU_X86_VENDOR_MASK)
	{
		case B_CPU_INTEL_X86:
			goto do_nothing;
			
		case B_CPU_AMD_X86:
			vendor = "AMD";
			break;
			
		case B_CPU_CYRIX_X86:
			vendor = "CYRIX";
			break;
			
		case B_CPU_IDT_X86:
			vendor = "IDT";
			break;
		
		case B_CPU_RISE_X86:
			vendor = "RISE";
			break;
			
		default: vendor = "Unknown";
	}
	
	SetDrawingMode(B_OP_OVER);
	SetHighColor(240,240,240);

	width = StringWidth(vendor);
	MovePenTo(10 + (32 - width / 2), 30);
	DrawString(vendor);

do_nothing:
	;
#endif

	switch(fSysInfo.cpu_type) {
#if __POWERPC__
		case B_CPU_PPC_603:
			processor = "603";
			break;
		case B_CPU_PPC_603e:
			processor = "603e";
			break;
		case B_CPU_PPC_750:
			processor = "750";
			break;
		case B_CPU_PPC_604:
			processor = "604";
			break;
		case B_CPU_PPC_604e:
			processor = "604e";
			break;
#endif
#if __INTEL__
		case B_CPU_X86:
			processor = "unknown x86";
			break;
		case B_CPU_INTEL_PENTIUM:
		case B_CPU_INTEL_PENTIUM75:
			processor = "Pentium";
			break;
		case B_CPU_INTEL_PENTIUM_486_OVERDRIVE:
		case B_CPU_INTEL_PENTIUM75_486_OVERDRIVE:
			processor = "Pentium 486 Overdrive";
			break;
		case B_CPU_INTEL_PENTIUM_MMX:
		case B_CPU_INTEL_PENTIUM_MMX_MODEL_8:
			processor = "Pentium MMX";
			break;
		case B_CPU_INTEL_PENTIUM_PRO:
			processor = "Pentium Pro";
			break;
		case B_CPU_INTEL_PENTIUM_II_MODEL_3:
		case B_CPU_INTEL_PENTIUM_II_MODEL_5:
			processor = "Pentium II";
			break;
		case B_CPU_INTEL_CELERON:
			processor = "Celeron";
			break;
		case B_CPU_INTEL_PENTIUM_III:
			processor = "Pentium III";
			break;
		case B_CPU_AMD_K5_MODEL0:
		case B_CPU_AMD_K5_MODEL1:
		case B_CPU_AMD_K5_MODEL2:
		case B_CPU_AMD_K5_MODEL3:
			processor = "K5";
			break;
		case B_CPU_AMD_K6_MODEL6:
		case B_CPU_AMD_K6_MODEL7:
			processor = "K6";
			break;
		case B_CPU_AMD_K6_2:
			processor = "K6-2";
			break;
		case B_CPU_AMD_K6_III:
			processor = "K6-III";
			break;
		case B_CPU_CYRIX_GXm:
			processor = "GXm";
			break;
		case B_CPU_CYRIX_6x86MX:
			processor = "6x86MX";
			break;
		case B_CPU_IDT_WINCHIP_C6:
			processor = "WinChip C6";
			break;
		case B_CPU_IDT_WINCHIP_2:
			processor = "WinChip 2";
			break;
		case B_CPU_RISE_mP6:
			processor = "mP6";
			break;
#endif
		default:
			processor = "unknown";
			break;
	}

	sprintf(buf, "%d MHz", CalcCPUSpeed(&fSysInfo));
	SetDrawingMode(B_OP_OVER);
	SetHighColor(240,240,240);

	width = StringWidth(processor);
	MovePenTo(10 + (32 - width / 2), 48);
	DrawString(processor);

	width = StringWidth(buf);
	MovePenTo(10 + (32 - width / 2), 60);
	DrawString(buf);

	_inherited::Draw(updateRect);
}

// --------------------------------------------------------------
// This routine is called by the Interface Kit when a view is
// added to a window. We use it here to spawn our memory checker
// thread. This is a good place in general to put any kind of
// initialization code a view might need.
// --------------------------------------------------------------
void TPulseView::AttachedToWindow()
{
	rgb_color	c = { 168, 168, 168, 0 };

	SetFont(be_bold_font);
#if __INTEL__
	SetFontSize(7.0);
#else
	SetFontSize(9);
#endif
	fThen = system_time();
	SetViewColor(c);
	SetLowColor(c);
	Invalidate();
}

void TPulseView::Pulse()
{
	Update();
}

// --------------------------------------------------------------
// This routine updates the CPU thermometer
// If the "Doit" parameter is TRUE, we update the graph even
// if the activity level hasn't changed.
// --------------------------------------------------------------
void TPulseView::Update( bool DoIt )
{
	int			i;
	bigtime_t	now;
	
	if (Window()->Lock()) {

		//
		// Check to see if a CPU has been turned off or on
		//
		for( i=0; i<fSysInfo.cpu_count; i++ ) {
			if( !_kget_cpu_state_(i) && fButtons[i]->Value() ) {
//+				PRINT(("View::Pulse: turning cpu %d OFF\n", i));
				fButtons[i]->SetValue( 0 );
			}
			if( _kget_cpu_state_(i) && !fButtons[i]->Value() ) {
//+				PRINT(("View::Pulse: turning cpu %d ON\n", i));
				fButtons[i]->SetValue( 1 );
			}
		}

		//
		// Get system info
		//
		get_system_info(&fSysInfo);
		now = system_time();

		//
		// Compute time spend in idle threads
		//
		for( i=0; i<fSysInfo.cpu_count; i++ ) {
			float CPUTime = 100 * (fSysInfo.cpu_infos[i].active_time - fCPUs[i]->PrevTick()) / (now - fThen);
			fCPUs[i]->SetPrev( fSysInfo.cpu_infos[i].active_time );
			fCPUs[i]->Set( max_c(0,CPUTime) );
		}
		
		fThen = now;
		Sync();
		Window()->Unlock();
	}
}

//---------------------------------------------------------------

BPicture *BitmapToPicture(BBitmap *bitmap)
{
	BBitmap *wind = new BBitmap(bitmap->Bounds(), bitmap->ColorSpace(), TRUE);
	BView	*view = new BView(wind->Bounds(), "", B_FOLLOW_NONE, B_WILL_DRAW);

	wind->Lock();
	wind->AddChild(view);

	BPicture *picture = new BPicture();

	view->BeginPicture(picture);
	view->DrawBitmap(bitmap, B_ORIGIN);
	view->EndPicture();

	wind->Unlock();
	
	delete wind;

	return picture;
}

//---------------------------------------------------------------
// C P U B U T T O N
//---------------------------------------------------------------

CPUButton::CPUButton(BRect r, const char* name, const char* label,
				   long cpu, BPicture* normal, BPicture* pushed)
		: BPictureButton(r, name, normal, pushed,
						new BMessage(), B_TWO_STATE_BUTTON,
						B_FOLLOW_LEFT + B_FOLLOW_TOP, B_WILL_DRAW | B_PULSE_NEEDED)
{
	InitObject(cpu);
}

/* ---------------------------------------------------------------- */

void CPUButton::InitObject(long cpu)
{
	fCPU = cpu;

	if (_kget_cpu_state_(fCPU))		{ // Processor is active
		fOn = TRUE;
		SetValue(1);	// Push the button
	} else {
		fOn = FALSE;
		SetValue(0);
	}
}

/* ---------------------------------------------------------------- */

CPUButton::CPUButton(BMessage *data)
	: BPictureButton(data)
{
	InitObject(data->FindInt32("cpu"));
}

/* ---------------------------------------------------------------- */

status_t CPUButton::Invoke(BMessage *msg)
{
	OnOff();
	return B_OK;
}

/* ---------------------------------------------------------------- */

long CPUButton::Archive(BMessage *data, bool deep) const
{
	_inherited::Archive(data, deep);
	data->AddString("add_on", app_signature);
	data->AddInt32("cpu", fCPU);
	return 0;
}

/* ---------------------------------------------------------------- */

CPUButton *CPUButton::Instantiate(BMessage *data)
{
	if (!validate_instantiation(data, "CPUButton"))
		return NULL;
	return new CPUButton(data);
}

/* ---------------------------------------------------------------- */

void CPUButton::AttachedToWindow()
{
	SetTarget(this);
}

/* ---------------------------------------------------------------- */

void CPUButton::Pulse()
{
	// Check to see if a CPU has been turned off or on

	if (_kget_cpu_state_(fCPU) != Value()) {
//+		PRINT(("CPU::Pulse: turning cpu %d %s\n", fCPU, fOn ? "ON" : "OFF"));
		SetValue(!Value());
	}
}

/* ---------------------------------------------------------------- */

bool last_enabled_cpu(long cpu)
{
	system_info	sinfo;

	get_system_info(&sinfo);
	if (sinfo.cpu_count == 1)
		return true;

	for (int i = 0; i < sinfo.cpu_count; i++) {
		if (i == cpu)
			continue;
		if (_kget_cpu_state_(i) == 1)
			// found an enabled cpu so its OK to disable 'cpu'
			return false;
	}

	return true;
}

/* ---------------------------------------------------------------- */

void CPUButton::OnOff()
{
	if (!last_enabled_cpu(fCPU)) {
		fOn = !fOn;
//+		PRINT(("Turning cpu %d %s\n", fCPU, fOn ? "ON" : "OFF"));
		_kset_cpu_state_(fCPU, fOn);
	} else {
		system_info	sinfo;
		BAlert		*alert;
		get_system_info(&sinfo);
		if (sinfo.cpu_count == 1)
			alert = new BAlert("",
				"Sorry, you can't disable the only cpu!", "OK");
		else
			alert = new BAlert("",
				"Sorry, you can't disable the last active cpu!", "OK");

		alert->Lock();

		// move the alert to be center over the cpu button that was
		// clicked
		BRect	cpu_rect = Bounds();
		ConvertToScreen(&cpu_rect);

		BRect	rr = alert->Frame();
		BPoint	pp(rr.Width() / 2, rr.Height() / 2);

		alert->MoveTo(cpu_rect.LeftTop() - pp);

		BScreen	screen;
		BRect	sb = screen.Frame();
		rr = alert->Frame();

		// now make sure that the alert is completely onscreen
		pp = rr.LeftTop();
		if (rr.left < (sb.left + 5))
			pp.x = sb.left + 5;
		else if (rr.right > (sb.right - 5))
			pp.x -= rr.right - (sb.right - 5);

		if (rr.top < (sb.top + 5))
			pp.y = sb.top + 5;
		else if (rr.bottom > (sb.bottom - 5))
			pp.y -= rr.bottom - (sb.bottom - 5);

		if (pp != rr.LeftTop())
			alert->MoveTo(pp);

		alert->Unlock();

		alert->Go(NULL);
	}			
}

/* ---------------------------------------------------------------- */

void CPUButton::MessageReceived(BMessage *msg)
{
	switch(msg->what) {
		case B_ABOUT_REQUESTED:
			(new BAlert("About Pulse", "Pulse (Replicant)\n\n  Brought to you by the folks at Be.\n\n  Copyright Be Inc., 1991-1997","OK"))->Go();
			break;
		default:
			_inherited::MessageReceived(msg);
	}
}

