//******************************************************************************
//
//	File:		progressbar.cpp
//
//	Description:	Generic progress bar code
//
//******************************************************************************

/*
	Copyright 1999, Be Incorporated.   All Rights Reserved.
	This file may be used under the terms of the Be Sample Code License.
*/

#ifndef PBAR_H
#include "progressbar.h"
#endif

#include <Debug.h>

//---------------------------------------------------------------
// T P R O G R E S S B A R
//---------------------------------------------------------------
// Note: Right now, this is rather hardcoded for 15 specific sized
//       segments. Progress bar sizes other than the ones use in Pulse
//       won't look good.
//---------------------------------------------------------------

TProgressBar::TProgressBar(BRect r, char *name, int32 initial,
					bool warn, int32 warnVal)
		   : BView(r, name, B_FOLLOW_NONE, B_WILL_DRAW)
{	
	int32 counter;
	float height;
	
	fPrevious = initial;		// Set initial value
	fCurrent = initial;
	fPrevTick = 0;
	fWarning = false;
	fWidth = r.right - r.left - 1;	// Won't work for vertical bars
	//
	// Create 20 segments
	//
	height = (r.bottom - r.top) - 8;
	for (counter = 0; counter < 20; counter++) {
		segments[counter].rect.Set(r.left + (counter * 7),	//left
									r.top,					// top
									(r.left + (counter * 7) + 5), //right
									r.top + height);		// bottom
		segments[counter].rect.OffsetTo(BPoint((counter * 7) + 4, 4));
		segments[counter].shade = (int32)(80+(8.8 * counter));
		}
	SetLowColor(255,0,0);
}

// --------------------------------------------------------------

void TProgressBar::Set( int32 value )
{
	fCurrent = (int32)(value / 4.9);		// How many segments to light up
	if (fCurrent > 20)
		fCurrent = 20;
	//
	// Check to see if we've gone over or under the warning level
	//
	this->Render( false );
}

//
// Draws the progress bar. If "drawAll" is true the entire bar is redrawn rather
// than just the part that changed.
//
void TProgressBar::Render( bool drawAll )
{
	BRect	updateRect;
	int32	counter;
	
	if (drawAll)	{		// Refresh entire bar
		updateRect = Bounds();
		updateRect.InsetBy(2,2);
		SetHighColor(0,0,0);
		FillRect(updateRect);
		for (counter = 0; counter < fCurrent; counter++) {
			SetHighColor(0,segments[counter].shade,0);
			FillRect(segments[counter].rect);
		}
		SetHighColor(75,75,75);
		if (fCurrent < 20)
			for (counter = fCurrent; counter < 20; counter++)
				FillRect(segments[counter].rect);
		}
	else if (fCurrent > fPrevious) {
		for (counter = fPrevious; counter < fCurrent; counter++ ) {
			updateRect = segments[counter].rect;
			SetHighColor(0,segments[counter].shade,0);
			FillRect(updateRect);
			}
		}
	else if (fCurrent < fPrevious) {
		SetHighColor(75,75,75);
		for (counter = fCurrent; counter < fPrevious; counter++ ) {
			updateRect = segments[counter].rect;
			FillRect(updateRect);
			}
		if (fCurrent == 0) {			// Special case
			updateRect = segments[0].rect;
			FillRect(updateRect);
			}
		}
		Sync();
		fPrevious = fCurrent;
}

//---------------------------------------------------------------
void	TProgressBar::Draw( BRect updateRect )
{
	BRect	frame;
	
	SetHighColor(0,0,0);
	frame = Bounds();
	frame.InsetBy(2,2);
	StrokeRect(frame);
	frame.InsetBy(3,3);
	StrokeRect(frame);
	//
	// Add bevels
	//
	SetHighColor(dkgray, dkgray, dkgray);
	frame = Bounds();
	StrokeLine(BPoint(frame.left,frame.top),BPoint(frame.right,frame.top));
	StrokeLine(BPoint(frame.left,frame.top+1),BPoint(frame.right,frame.top+1));
	StrokeLine(BPoint(frame.left,frame.top),BPoint(frame.left,frame.bottom));
	StrokeLine(BPoint(frame.left+1,frame.top),BPoint(frame.left+1,frame.bottom));
	SetHighColor(ltgray, ltgray, ltgray);
	StrokeLine(BPoint(frame.right-1,frame.top+2),BPoint(frame.right-1,frame.bottom));
	StrokeLine(BPoint(frame.right,frame.top+1),BPoint(frame.right,frame.bottom));
	StrokeLine(BPoint(frame.left+1,frame.bottom-1),BPoint(frame.right-1,frame.bottom-1));
	StrokeLine(BPoint(frame.left,frame.bottom),BPoint(frame.right,frame.bottom));
	this->Render( true );
}

