/*
	Copyright 1999, Be Incorporated.   All Rights Reserved.
	This file may be used under the terms of the Be Sample Code License.
*/

#include <Screen.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>

#include "MiniPulse.h"
#include "pulse.h"

const char *app_signature = "application/x-vnd.Be-PULS";

PulseApplication::PulseApplication(
		bool mini,
		int width,
		bool install,
		rgb_color framecolor,
		rgb_color activecolor,
		rgb_color idlecolor
	)
	:	BApplication(app_signature)
{
	if(mini) {
		if (install) {
			// install into the deskbar
			BRect buttonRect(0, 0, width, 16);
			buttonRect.OffsetBy(10, 0);
			MiniPulse *replicant =
				new MiniPulse( buttonRect, "MiniPulse",
							   framecolor, activecolor, idlecolor );
			BMessage archiveMsg(B_ARCHIVED_OBJECT);
			replicant->Archive(&archiveMsg);
			BMessenger messenger("application/x-vnd.Be-TSKB", -1, NULL);
			messenger.SendMessage(&archiveMsg);
			PostMessage(B_QUIT_REQUESTED);
			return;
		}
		BRect windowRect(100, 100, 100+width, 100+16);
		BWindow *window = new
			BWindow(windowRect, "", B_MODAL_WINDOW_LOOK, B_NORMAL_WINDOW_FEEL,
			        B_NOT_CLOSABLE | B_NOT_ZOOMABLE);
		window->SetSizeLimits(2,10000,2,10000);
		window->ResizeTo(width, 16);
		BRect buttonRect(window->Bounds());
		
		BView *button = new MiniPulse( buttonRect, "MiniPulse", framecolor,
									   activecolor, idlecolor, B_FOLLOW_ALL );
		window->AddChild(button);
		window->Show();
	}
	else {
		const short		kBorderWidth = 5;		// Width of window border

		BRect			r;
		TPulseWindow	*theWindow;
		BRect sframe;
		
		// TEMPORARY BScreen object... it will be destroyed
		// at the ending semicolon and therefore should not
		// hog the screen while the Pulse app is open.
		sframe = BScreen(B_MAIN_SCREEN_ID).Frame();
		
		r = TPulseView::CalcViewSize();
		r.OffsetTo( sframe.right - r.Width() - kBorderWidth,
					sframe.bottom - r.Height() - kBorderWidth );
		theWindow = new TPulseWindow(r);
		r.OffsetTo(BPoint(0, 0));
		theWindow->fPulseView = new TPulseView(r);
		theWindow->Lock();
		theWindow->AddChild(theWindow->fPulseView);
		theWindow->Unlock();
		theWindow->Show();
	}
}

void Usage(int /*argc*/, char **argv);
void Usage(int /*argc*/, char **argv)
{
	printf("Usage: %s [--mini] [-w width] [--width=width]\n"
	       "\t[--deskbar] [--framecolor 0xrrggbb]\n"
	       "\t[--activecolor 0xrrggbb] [--idlecolor 0xrrggbb]\n", argv[0]);
	exit(0);
}

extern int pulse_main(int argc, char **argv);

int main(int argc, char **argv)
{
	int c;
	int install_in_deskbar = false;
	int mini_form = false;
	int width = 7;
	rgb_color framecolor = default_framecolor;
	rgb_color idlecolor = default_idlecolor;
	rgb_color activecolor = default_activecolor;
	
	while(1) {
		int option_index = 0;
		static struct option long_options[] = {
			{"deskbar", 0, &install_in_deskbar, true},
			{"width", 1, 0, 'w'},
			{"framecolor", 1, 0, 0},
			{"activecolor", 1, 0, 0},
			{"idlecolor", 1, 0, 0},
			{"mini", 0, &mini_form, true},
			{"help", 0, 0, 'h'},
			{0,0,0,0}
		};
		c = getopt_long (argc, argv, "hw:", long_options, &option_index);
		if (c == -1)
			break;
		switch(c) {
			case 0:
				switch(option_index) {
					case 2: /* framecolor */
					case 3: /* activecolor */
					case 4: /* idlecolor */
						uint32 rgb = strtoul(optarg, NULL, 0);
						rgb_color newcolor;
						newcolor.blue = rgb & 0xff; rgb>>=8;
						newcolor.green = rgb & 0xff; rgb>>=8;
						newcolor.red = rgb & 0xff; rgb>>=8;
						newcolor.alpha = rgb;
						switch(option_index) {
							case 2: /* framecolor */
								framecolor = newcolor;
								break;
							case 3: /* activecolor */
								activecolor = newcolor;
								break;
							case 4: /* idlecolor */
								idlecolor = newcolor;
								break;
						}
						break;
				}
				break;
			case 'w':
				width = atoi(optarg);
				break;
			case 'h':
			case '?':
				Usage(argc,argv);
			default:
				printf ("?? getopt returned character code 0%o ??\n", c);
		}
	}
	if (optind < argc) {
		Usage(argc,argv);
	}

	if(width < 3) {
		printf("minimum width is 3\n");
	}
	else {
		PulseApplication app(mini_form, width-1, install_in_deskbar,
		                     framecolor, activecolor, idlecolor);
		app.Run();
	}

	return 0;
}
