////////////////////////////////////////////////////////////
// TestApp.cpp
// -----------
// A simple example of MultiInvoker in action.
//
// Copyright 1999, Be Incorporated.   All Rights Reserved.
// This file may be used under the terms of the Be Sample
// Code License.

#include <stdio.h>
#include <Application.h>
#include <Handler.h>
#include <Looper.h>
#include <Message.h>
#include <MessageFilter.h>
#include "MultiInvoker.h"



////////////////////////////////////////////////////////////
// Class: Countdown
// ----------------
// This is a simple example of an observable object.
// It sends messages to its observers when its state
// changes.

class Countdown : public MultiInvoker
{
public:
	void Run(int32 startValue, bigtime_t period=1000000);
};

void Countdown::Run(int32 startValue, bigtime_t period)
{
	BMessage msg('cDWN');
	msg.AddInt32("value", 0);
	while (startValue >= 0) {
		msg.ReplaceInt32("value", startValue--);
		Invoke(&msg);
		if (startValue >= 0)
			snooze(period);
	}
}



////////////////////////////////////////////////////////////
// Class: Observer
// ---------------
// A simple observer class. It simply prints any message
// that it receives.

class Observer : public BHandler
{
public:
	Observer(const char* name = NULL);
	void MessageReceived(BMessage* msg);
};

Observer::Observer(const char* name)
	: BHandler(name)
{}

void Observer::MessageReceived(BMessage* msg)
{
	printf("%s received: ", Name());
	msg->PrintToStream();
}



////////////////////////////////////////////////////////////
// Class: TestApp
// ---------------
// An application that tests the Countdown and Observer
// classes.

class TestApp : public BApplication
{
public:
	TestApp();
	void ReadyToRun();
};

TestApp::TestApp()
	: BApplication("application/x-vnd.Be-DTS.TestApp")
{}

void TestApp::ReadyToRun()
{
	Observer *h1 = new Observer("Observer 1");
	Observer *h2 = new Observer("Observer 2");
	Observer *h3 = new Observer("Observer 3");
	
	BLooper* loop = new BLooper("observer looper");
	loop->AddHandler(h1); loop->AddHandler(h2); loop->AddHandler(h3);
	loop->Run();
	
	Countdown c;
	c.AddTarget(h1);
	c.AddTarget(h2);
	c.AddTarget(h3);

	c.Run(5);

	if (loop->Lock())
		loop->Quit();
		
	PostMessage(B_QUIT_REQUESTED);
}



////////////////////////////////////////////////////////////
// main entry point

int main()
{
	TestApp app;
	app.Run();
	return 0;
}

