#include <posix/sys/stat.h>

#include "FtpIt.h"

#define DELETE_SERVER 'dlsv'
#define GO_SERVER 'gOsv'
#define GO_SERVER_BUTTON 'gosb'
#define	HOST_CHANGED 'svch'
#define PATH_CHANGED 'pach'
#define USER_CHANGED 'usch'
#define PASS_CHANGED 'psch'
#define FTP_MESSAGE 'ftpm'
#define PASSIVE_CHANGED 'psvc'

#define kSViewHeight 145
#define kSViewWidth 250
#define kSViewIconWidth	60
#define kSViewBarPixels 240
#define kSettingsFile "/settings"
#define kSettingsDir "/FtpIt"
#define kHostAttribute "FtpIt:HOST"
#define kPathAttribute "FtpIt:PATH"
#define kUserAttribute "FtpIt:USER"
#define kPassAttribute "FtpIt:PASS"
#define kPassiveAttribute "FtpIt:PASSIVE"

void process_refs(entry_ref dir_ref, BMessage *msg, void *) 
{
	//create the FtpIt window
	FIWindow *window = new FIWindow(dir_ref, msg);
	window->Show();

	//now wait for the window to quit so that this Tracker thread isn't
	//deleted out from under us
	thread_id thread = window->Thread();
	status_t win_status = B_OK;
	wait_for_thread(thread, &win_status);
} 


ServerView::ServerView(BRect bounds, const char *host, const char *path, const char *user, const char *pass, bool passive) :
	BView(bounds, host, B_FOLLOW_ALL, B_WILL_DRAW), mIcon(NULL), mClientBusy(false),
	mAmountTransfered(0), mTotalAmount(0)
{
	//add the UI
	BRect workingRect(bounds);

	//Host text control
	workingRect.left += kSViewIconWidth + 5;
	workingRect.right = bounds.right - 5;
	workingRect.top += 5;
	workingRect.bottom = workingRect.top + 20;
	mHostView = new BTextControl(workingRect, "HostTextControl", "Host:", host, new BMessage(HOST_CHANGED));
	mHostView->SetDivider(35);
	AddChild(mHostView);
	
	//Path text control
	workingRect.top = workingRect.bottom + 5;
	workingRect.bottom = workingRect.top + 20;
	mPathView = new BTextControl(workingRect, "PathTextControl", "Path:", path, new 	BMessage(PATH_CHANGED));
	mPathView->SetDivider(35);
	AddChild(mPathView);

	//User text control
	workingRect.top = workingRect.bottom + 5;
	workingRect.bottom = workingRect.top + 20;
	workingRect.right -= 55;
	mUserView = new BTextControl(workingRect, "UserTextControl", "User:", user, new BMessage(USER_CHANGED));
	mUserView->SetDivider(35);
	AddChild(mUserView);
	
	float buttonTop = workingRect.top;

	//Password text control
	workingRect.top = workingRect.bottom + 5;
	workingRect.bottom = workingRect.top + 20;
	mPassView = new BTextControl(workingRect, "PassTextControl", "Pass:", pass, new BMessage(PASS_CHANGED));
	mPassView->SetDivider(35);
	AddChild(mPassView);

	//passive checkbox
	workingRect.left += 35;
	workingRect.top = workingRect.bottom;
	workingRect.bottom = workingRect.top + 15;
	mPassiveBox = new BCheckBox(workingRect, "passive check", "Passive", new BMessage(PASSIVE_CHANGED));
	AddChild(mPassiveBox);
	mPassiveBox->SetValue(passive ? B_CONTROL_ON : B_CONTROL_OFF);

	//Button
	workingRect.top = buttonTop;
	workingRect.left = workingRect.right + 15;
	workingRect.right = workingRect.left + 35;
	mGoButton = new BButton(workingRect, "GoButton", "Go", new BMessage(GO_SERVER));
	AddChild(mGoButton);
	mGoButton->MakeDefault(true);
	
	//message view
	BRect messRect(Bounds());
	messRect.left += 5;
	messRect.top += 40;
	messRect.bottom -= 22;
	messRect.right = kSViewIconWidth + 2;
	mMessageView = new BTextView(messRect, "message view", BRect(0,0,messRect.Width(), messRect.Height()), B_FOLLOW_NONE, B_WILL_DRAW);	
	AddChild(mMessageView);
	mMessageView->MakeEditable(false);
	mMessageView->SetText("Ready");
	
	//status bar
	BRect statusRect(Bounds());
	statusRect.top = statusRect.bottom - 35;
	statusRect.bottom -= 5;
	statusRect.left += 5;
	statusRect.right -= 5;
	mStatusBar = new BStatusBar(statusRect, "status bar");
	AddChild(mStatusBar);

}


ServerView::~ServerView()
{
	if(mClientBusy){
		kill_thread(mClientThread);
	}
	delete mIcon;
}

const char *
ServerView::GetHost()
{
	return mHostView->Text();
}

const char *
ServerView::GetUser()
{
	return mUserView->Text();
}

const char *
ServerView::GetPath()
{
	return mPathView->Text();
}

const char *
ServerView::GetPass()
{
	return mPassView->Text();
}

bool //should be passed in the form of dir/dir/file or dir/dir/
ServerView::CreateDirectoryFor(const char *newDirectory, FtpClient *client)
{
	BString lDir(newDirectory);
	int32 lIndex = lDir.FindLast("/");
	if(lIndex == lDir.Length() - 1 || lIndex > 0) //get rid of a final slash or leafName
		lDir.Truncate(lIndex);
	else
		return false;
	
	int32 deepness = 0; //keep the number of directories in we are from the time of the call
	
	//lDir should now be dir/dir/dir	
	lIndex = lDir.FindFirst("/");
	//recursively try to create directories for the file
	while(true){ 
		lIndex = lDir.FindFirst("/");
		BString tDir;
		if(lIndex > 0)
			tDir.Append(lDir, lIndex);
		else
			tDir.SetTo(lDir);
		if(!client->cd(tDir.String())){
			if(!client->makeDir(tDir.String())){
				for(int i=0; i<deepness; ++i)
					client->cd("..");
				return false;
			}
			if(!client->cd(tDir.String())){
				for(int i=0; i<deepness; ++i)
					client->cd("..");
				return false;
			}
		}
		++deepness;
		if(tDir == lDir)
			break;
		lDir.Remove(0, lIndex + 1);
	}

	//now bring the pwd back to where it was at the time of the call
	for(int i=0; i<deepness; ++i)
		client->cd("..");
	return true;
}

//this keeps some state on whether the client thread is active and
//allows the client thread to thread safely change the state and button label
bool 
ServerView::SetBusy(bool busy)
{
	bool worked = false;
	if(!Window()->Lock())
		return worked;

	if(mClientLock.Lock()){
		mClientBusy = busy;
		mClientLock.Unlock();
		worked = true;
	}
	//change the control state accordingly
	if(busy){
		mHostView->SetEnabled(false);
		mPathView->SetEnabled(false);
		mUserView->SetEnabled(false);
		mPassView->SetEnabled(false);
		mGoButton->SetLabel("Stop");
	}
	else{
		mHostView->SetEnabled(true);
		mPathView->SetEnabled(true);
		mUserView->SetEnabled(true);
		mPassView->SetEnabled(true);
		mGoButton->SetLabel("Go");
	}
	Window()->Unlock();
	return worked;
}

bool 
ServerView::AmountTransfered(float amount)
{
	if(!Window()->Lock())
		return false;
	if(!mClientLock.Lock())
		return false;
	if(amount < 0){
		mStatusBar->Update(-mAmountTransfered);
		mAmountTransfered = 0;
	}
	else{
		mStatusBar->Update(amount);
		mAmountTransfered += amount;
	}
	mClientLock.Unlock();
	Window()->Unlock();
	return true;
}

bool 
ServerView::SetTotalAmount(float amount)
{
	if(!mClientLock.Lock())
		return false;
	mTotalAmount = amount;
	mStatusBar->SetMaxValue(mTotalAmount);
	mClientLock.Unlock();
	return true;
}

bool 
ServerView::SetMessage(const char *message)
{
	if(!mClientLock.Lock())
		return false;
	if(mMessageView && message)
		mMessageView->SetText(message);
	mClientLock.Unlock();
	return true;
}

int32 
ClientThread(void *data)
{
	ServerView *view = static_cast<ServerView *>(data);
	FIWindow *window = dynamic_cast<FIWindow *>(view->Window());
	BList *pathList = window->GetPathList();
	BPath *workingDirectory = window->GetWorkingDirectory();


	FtpClient client(view);
	//open the connection
	if(client.connect(view->GetHost(), view->GetUser(), view->GetPass())){
		client.setPassive(view->GetPassive());
		//change the pwd to the local pwd
		if(!client.cd(view->GetPath())){
			BMessage badDir(FTP_MESSAGE);
			badDir.AddString("message", "Invalid directory");
			window->PostMessage(&badDir, view);
			view->AmountTransfered(-1);
			view->SetBusy(false);
			return -1;
		}
		//iterate through the list of paths and try to send, creating dirs if necessary
		for(int i=0; i < pathList->CountItems(); ++i){
			BPath filePath(static_cast<BPath *>(pathList->ItemAt(i))->Path());
			BString leafName(filePath.Path());
			leafName.RemoveFirst(workingDirectory->Path());
			leafName.RemoveFirst("/");
			BMessage ftpMessage(FTP_MESSAGE);
			ftpMessage.AddString("message", leafName.String());
			window->PostMessage(&ftpMessage, view);
			//try to put the file
			if(!client.putFile(filePath.Path(), leafName.String())){
				//try to create the directory for a failed put
				if(view->CreateDirectoryFor(leafName.String(), &client) == false){
					BMessage badDir(FTP_MESSAGE);
					badDir.AddString("message", "Couldn't create directory");
					window->PostMessage(&badDir, view);
					view->SetBusy(false);
					view->AmountTransfered(-1);
					return -2;
				}
				else if(!client.putFile(filePath.Path(), leafName.String())){
					BMessage invDir(FTP_MESSAGE);
					BString errString("Couldn\'t put ");
					errString.Append(leafName.String());
					invDir.AddString("message", errString.String());
					window->PostMessage(&invDir, view);
					view->SetBusy(false);
					view->AmountTransfered(-1);
					return -3;
				}
			}
		}
		BMessage completeMessage(FTP_MESSAGE);
		completeMessage.AddString("message", "Complete");
		window->PostMessage(&completeMessage, view);
		view->SetBusy(false);
		return 0;					 
	}
	else{
		BMessage userPassMessage(FTP_MESSAGE);
		userPassMessage.AddString("message", "Couldn't connect");
		window->PostMessage(&userPassMessage, view);
		view->AmountTransfered(-1);
		view->SetBusy(false);
		return -4;
	}
}

void 
ServerView::MessageReceived(BMessage *msg)
{
	switch (msg->what){
		case GO_SERVER:{
			if(mClientLock.LockWithTimeout(1000) == B_OK){
				if(mClientBusy){ //stop everything
					kill_thread(mClientThread);
					mClientBusy = false;
					SetMessage("Stopped");
					mHostView->SetEnabled(true);
					mPathView->SetEnabled(true);
					mUserView->SetEnabled(true);
					mPassView->SetEnabled(true);
					mGoButton->SetLabel("Go");
					AmountTransfered(-1);
				} else { //go, baby, go
					mClientBusy = true;
					mClientThread = spawn_thread(ClientThread, "FtpIt Client Thread", B_NORMAL_PRIORITY, this);
					resume_thread(mClientThread);
					SetMessage("Connecting");
					mHostView->SetEnabled(false);
					mPathView->SetEnabled(false);
					mUserView->SetEnabled(false);
					mPassView->SetEnabled(false);
					mGoButton->SetLabel("Stop");
					AmountTransfered(-1);
				}
				mClientLock.Unlock();
				
			}
			break;
		}
		case AMOUNT_SENT:{ //update the amount sent (and thusly the status bar)
			int32 amount = 0;
			float oldAmount =  kSViewBarPixels * (mAmountTransfered / mTotalAmount);
			if(msg->FindInt32("amount", &amount) == B_OK)
				AmountTransfered(amount);
			break;
		}
		case FTP_MESSAGE: {
			if(!msg->HasString("message"))
				break;
			SetMessage(msg->FindString("message"));
		}
		default:
			BView::MessageReceived(msg);
			break;
	}
}

void 
ServerView::AttachedToWindow()
{
	BRect barRect(Bounds());
	mGoButton->SetTarget(this);
	
	//add icon
	mIcon = new BBitmap(BRect(0,0,B_LARGE_ICON-1,B_LARGE_ICON-1), B_CMAP8);
	BList *pathList = dynamic_cast<FIWindow *>(Window())->GetPathList();
	status_t error;
	if(pathList->CountItems() >= 1){
		BPath *filePath = static_cast<BPath *>(pathList->ItemAt(0));
		BEntry lEntry(filePath->Path());
		if(lEntry.InitCheck() != B_OK){
			delete mIcon;
			mIcon = NULL;
			BView::AttachedToWindow();
			return;
		}
		entry_ref eRef;
		if(lEntry.GetRef(&eRef) != B_OK){
			delete mIcon;
			mIcon = NULL;
			BView::AttachedToWindow();
			return;
		}
		error = BNodeInfo::GetTrackerIcon(&eRef, mIcon);
		if(error != B_OK){
			delete mIcon;
			mIcon = NULL;
			BView::AttachedToWindow();
			return;
		}		
	} else {
		delete mIcon;
		mIcon = NULL;
		BView::AttachedToWindow();
		return;
	}
	
	BView::AttachedToWindow();
	return;
}


void 
ServerView::Draw(BRect updateRect)
{
	//draw the icon if it exists
	if(mIcon){
		SetDrawingMode(B_OP_OVER);
		DrawBitmap(mIcon, BPoint(5,5));
	}
}

FIWindow::FIWindow(entry_ref dir_ref, BMessage *msg) :
	BWindow(BRect(100,100,100 + kSViewWidth,100 + kSViewHeight), "FtpIt", B_TITLED_WINDOW, B_NOT_ZOOMABLE | B_NOT_RESIZABLE)
{
	//determine the path
	BEntry entry(&dir_ref); 
	entry.GetPath(&mWorkingDirectory); 

	//add all of the entry refs to mPathList
	int refs = 0; 
	entry_ref file_ref;
	off_t transferSize = 0;
	while(msg->FindRef("refs", refs++, &file_ref) == B_NO_ERROR)
		transferSize += AddEntry(&file_ref);
	ResizeTo(kSViewWidth,kSViewHeight);
	SetSizeLimits(kSViewWidth, kSViewWidth, kSViewHeight, kSViewHeight);

	//create the server view	
	BString host;
	BString path;
	BString user;
	BString password;
	bool passive = false;
	ReadSettings(host, path, user, password, passive);
	mServerView = new ServerView(BRect(0,0,kSViewWidth,kSViewHeight), host.String(), path.String(), user.String(), password.String(), passive);
	AddChild(mServerView);
	mServerView->SetTotalAmount(transferSize);
}


FIWindow::~FIWindow()
{
	for(int i=0; i < mPathList.CountItems(); ++i)
		delete static_cast<BPath *>(mPathList.ItemAt(i));
}

bool 
FIWindow::QuitRequested()
{
	if(mServerView)
		WriteSettings(mServerView->GetHost(), mServerView->GetPath(), mServerView->GetUser(), mServerView->GetPass(), mServerView->GetPassive());
		
	return BWindow::QuitRequested();
}

int32
FIWindow::AddEntry(const entry_ref *entryRef)
{
	int32 size = 0;
	BEntry entry;
	entry.SetTo(entryRef); 
	if(!entry.Exists() || entry.IsSymLink())
		return size;
	
	if(entry.IsDirectory()){
		BDirectory workingDir(&entry);
		status_t lErr = workingDir.InitCheck();
		if(lErr != B_OK)
			return size;

		workingDir.Rewind();
		BEntry workingEntry;
		while(workingDir.GetNextEntry(&workingEntry) == B_OK){
			entry_ref tRef;
			workingEntry.GetRef(&tRef);
			size += AddEntry(&tRef);
		}
		return size;
	}
	BPath *tempPath = new BPath;
	entry.GetPath(tempPath);
	mPathList.AddItem(tempPath);
	off_t entrySize = 0;
	if(entry.GetSize(&entrySize) == B_OK)
		size += entrySize;
	return size;
}

bool 
FIWindow::ReadSettings(BString &host, BString &path, BString &user, BString &pass, bool &passive)
{
	//find the preference file
	BPath prefDirPath;
	if(find_directory(B_USER_SETTINGS_DIRECTORY, &prefDirPath, true) != B_OK)
		return false;
	BString lSettings(prefDirPath.Path());
	lSettings.Append(kSettingsDir);
	lSettings.Append(kSettingsFile);
	BNode settingsNode(lSettings.String());
	if(settingsNode.InitCheck() != B_OK || !settingsNode.IsFile())
		return false; //it could be that it was never saved

	char lHost[256];
	char lPath[256];
	char lUser[256];
	char lPass[256];
	ssize_t stringLen = settingsNode.ReadAttr(kHostAttribute, B_STRING_TYPE, 0, lHost, 255);
	if(stringLen > 0)
		host.SetTo(lHost, stringLen);

	stringLen = settingsNode.ReadAttr(kPathAttribute, B_STRING_TYPE, 0, lPath, 255);
	if(stringLen > 0)
		path.SetTo(lPath, stringLen);

	stringLen = settingsNode.ReadAttr(kUserAttribute, B_STRING_TYPE, 0, lUser, 255);
	if(stringLen > 0)
		user.SetTo(lUser, stringLen);
	
	stringLen = settingsNode.ReadAttr(kPassAttribute, B_STRING_TYPE, 0, lPass, 255);
	if(stringLen > 0)
		pass.SetTo(lPass, stringLen);

	int32 lPassive = 0;
	stringLen = settingsNode.ReadAttr(kPassiveAttribute, B_INT32_TYPE, 0, (void *)(&lPassive), sizeof(lPassive));
	if(stringLen == sizeof(lPassive))
		passive = (lPassive == 0) ? false : true;
	else
		passive = false;
	
	return true;
}

bool 
FIWindow::WriteSettings(const char *host, const char *path, const char *user, const char *pass, bool passive)
{
	//find or create the settings directory
	BPath prefDirPath;
	if(find_directory(B_USER_SETTINGS_DIRECTORY, &prefDirPath, true) != B_OK)
		return false;
	BString lDirString(prefDirPath.Path());
	lDirString.Append(kSettingsDir);
	BDirectory prefDir;
	if(prefDir.SetTo(lDirString.String()) != B_OK){
		if(create_directory(lDirString.String(), 0700) != B_OK){
			return false;
		}
		if(prefDir.SetTo(lDirString.String()) != B_OK){
			return false;
		}
	}

	//find or create the settings file
	BNode settingsNode;
	BString lSettingsString(lDirString);
	lSettingsString.Append(kSettingsFile);
	if(settingsNode.SetTo(lSettingsString.String()) != B_OK ){
		BFile settingsFile;
		if(prefDir.CreateFile(lSettingsString.String(), &settingsFile) != B_OK){
			return false;
		}			
		if(settingsNode.SetTo(lSettingsString.String()) != B_OK){
			return false; //couldn't read the settings file (maybe first time FtpIt was run)
		}
	}

	if(settingsNode.Lock() != B_OK){
		return false;
	}
	settingsNode.WriteAttr(kHostAttribute, B_STRING_TYPE, 0, host, strlen(host));
	settingsNode.WriteAttr(kPathAttribute, B_STRING_TYPE, 0, path, strlen(path));
	settingsNode.WriteAttr(kUserAttribute, B_STRING_TYPE, 0, user, strlen(user));
	settingsNode.WriteAttr(kPassAttribute, B_STRING_TYPE, 0, pass, strlen(pass));
	int32 pInt = passive ? 1 : 0;
	settingsNode.WriteAttr(kPassiveAttribute, B_INT32_TYPE, 0, (void *)(&pInt), sizeof(pInt));
	settingsNode.Unlock();	
	return true;
}

int main() 
{
	return 0;
}
