/*
 * PCIProbe
 *
 * Copyright(C) 1997 Kazuki Sakamoto (sakamoto@cec.co.jp)
 *	     *** ABSOLUTELY NO WARRANTY ***
 */

#include <Application.h>
#include <Locker.h>
#include <Messenger.h>
#include <Window.h>
#include <View.h>
#include <MenuItem.h>
#include <drivers/PCI.h>
#include <stdio.h>
#include <stdlib.h>

#include "pci_code.h"

#define	PCI_DEV_MAX	256

#define	ID_BYTES	3

#define	GEO_X		120
#define	GEO_Y		120
#define	X_SIZE		200
#define	WY_SIZE		14
#define	FONT_SIZE	11
#define	TOP		14
#define	BOT_MARGIN	6
#define	DEV_ID_X	6
#define	VEN_S_X		24
#define	CHIP_X		X_SIZE / 2 + 4
#define	LINE_TOP	16
#define	LINE_LEFT	2
#define	LINE_RIGHT	X_SIZE - 2
#define	LINE1_X		VEN_S_X - 4
#define	LINE2_X		X_SIZE / 2

/* Declaration -------------------------------------*/
class PciApp : public BApplication 
{
public:
			PciApp();
	virtual	~PciApp();
};

class PciWin : public BWindow 
{
public:
			PciWin(BRect frame, const char *title);
	virtual	~PciWin();
	
	virtual bool QuitRequested();
	virtual void FrameMoved(BPoint point);
};

class PciView : public BView 
{
public:
			PciView(BRect frame, char *name);
	virtual	~PciView();

	virtual void AttachedToWindow();
	virtual void Draw(BRect updateRect);
	virtual void MouseDown(BPoint point);
};

typedef struct pci_info_tab 
{
	pci_info_tab *prev;
	pci_info info;
	char *VenShort;
	char *VenFull;
	char *Chip;
	char *ChipDesc;
} pci_info_tab;

void pci_read();
void pci_alloc_free();

/* Grobal Valiable-----------------------------------*/
PciApp *my_app;
PciWin *my_win;
BView *my_view;
BPopUpMenu *my_menu;
BPoint my_point;

pci_info_tab *pci_table;
int pci_table_size;

const ulong MY_SIG = 'pcip';

/*---------------------------------------------------*/
void
pci_read() 
{
	int i;
	pci_info pciinfo;
	pci_info_tab *tmp;
	PPCI_VENTABLE ven_p;
	PPCI_DEVTABLE dev_p;

	pci_table_size = 0;
	pci_table = (pci_info_tab *)0;

	for (i = PCI_DEV_MAX; i >= 0; i--) 
	{
		if (get_nth_pci_info(i, &pciinfo) == B_NO_ERROR) 
		{
			if ((tmp = (pci_info_tab *)malloc(sizeof(pci_info_tab)))
				< (pci_info_tab *)0) 
			{
				fprintf(stderr, "Can't malloc\n");
				exit(1);
			}

			printf("Vendor - %d\n", pciinfo.vendor_id);
			printf("Device - %d\n", pciinfo.device_id);
			
			tmp->info = pciinfo;
			tmp->prev = pci_table;

			for (ven_p = PciVenTable; ven_p->VenId != 0xFFFF;
			    ven_p++) 
			{
				if (ven_p->VenId == pciinfo.vendor_id) 
				{
					if (ven_p->VenShort)
						tmp->VenShort = ven_p->VenShort;
					if (ven_p->VenFull)
						tmp->VenFull = ven_p->VenFull;
				}
			}

			for (dev_p = PciDevTable; dev_p->VenId != 0xFFFF;
			    dev_p++) {
				if (dev_p->VenId == pciinfo.vendor_id &&
				    dev_p->DevId == pciinfo.device_id) {
				    if (dev_p->Chip)
						tmp->Chip = dev_p->Chip;
					if (dev_p->ChipDesc)
						tmp->ChipDesc = dev_p->ChipDesc;
				}
			}

			pci_table = tmp;
			pci_table_size++;
		}
	}
}

void
pci_alloc_free()
{
	pci_info_tab *tmp;

	do {
		tmp = pci_table->prev;
		free(pci_table);
	} while (pci_table = tmp->prev);
	
	free(tmp);
}

/*---------------------------------------------------*/
main(int argc, char *argv[])
{
	pci_read();

	// make the new application object and start
	my_app = new PciApp();
	my_app->Run();

	// application is finished
	delete my_app;
	pci_alloc_free();

	return 0;
}

/*---------------------------------------------------*/
PciApp::PciApp()
	:BApplication(MY_SIG)
{

	/*
	 * PCIProbe only supports 1 instance
	 * of the application running at the same time.
	 */
	BList list;
	be_roster->GetAppList(MY_SIG, &list);
	long app_count = list.CountItems();
	if (app_count > 1) {
		PostMessage(B_QUIT_REQUESTED);
		return;
	}

	/*
	 * Window Open
	 */
	BRect my_rect;
	my_rect.Set(GEO_X, GEO_Y, GEO_X + X_SIZE,
	    GEO_Y + pci_table_size * WY_SIZE + TOP + BOT_MARGIN);
	my_win = new PciWin(my_rect, "PCIProbe");
	my_point = BPoint(GEO_X, GEO_Y);

	/*
	 * View Add
	 */
	my_rect.OffsetTo(B_ORIGIN);
	my_view = new PciView(my_rect, "PciView");
	my_win->AddChild(my_view);

	/*
	 * Window Show
	 */
	my_win->Show();
}

PciApp::~PciApp()
{
	my_win = NULL;
}

/*---------------------------------------------------*/
PciWin::PciWin(BRect frame, const char *title)
	: BWindow(frame, title, B_TITLED_WINDOW, B_NOT_RESIZABLE | B_NOT_ZOOMABLE)
{
}

PciWin::~PciWin()
{
}

bool
PciWin::QuitRequested()
{
	my_app->PostMessage(B_QUIT_REQUESTED);
	return true;
}

void
PciWin::FrameMoved(BPoint point)
{

	my_point = point;
}

/*---------------------------------------------------*/
PciView::PciView(BRect frame, char *name)
	: BView(frame, name, B_FOLLOW_ALL, B_WILL_DRAW)
{
}

PciView::~PciView()
{
}

void
PciView::AttachedToWindow()
{

	SetFont(be_plain_font);
	SetFontSize(FONT_SIZE);
}

void
PciView::Draw(BRect)
{
	char p[ID_BYTES];

	pci_info_tab *tmp = pci_table;
	int y = TOP;
	StrokeLine(BPoint(LINE_LEFT, LINE_TOP),
		BPoint(LINE_RIGHT, LINE_TOP),
		B_SOLID_HIGH);
	MovePenTo(BPoint(DEV_ID_X, TOP));
	DrawString("ID");
	MovePenTo(BPoint(VEN_S_X, TOP));
	DrawString("Vendor");
	MovePenTo(BPoint(CHIP_X, TOP));
	DrawString("Chip");

	do {
		y += WY_SIZE;

		MovePenTo(BPoint(DEV_ID_X, y));
		sprintf(p, "%02x", tmp->info.device);
		DrawString(p);

		MovePenTo(BPoint(VEN_S_X, y));
		if (tmp->VenShort)
			DrawString(tmp->VenShort);

		MovePenTo(BPoint(CHIP_X, y));
		if (tmp->Chip)
			DrawString(tmp->Chip);

		StrokeLine(BPoint(LINE_LEFT, y + 2), BPoint(LINE_RIGHT, y + 2),
			B_SOLID_HIGH);
	} while (tmp = tmp->prev);

	StrokeLine(BPoint(LINE_LEFT, LINE_TOP),
		BPoint(LINE_LEFT, y + 2),
		B_SOLID_HIGH);
	StrokeLine(BPoint(LINE1_X, LINE_TOP),
		BPoint(LINE1_X, y + 2),
		B_SOLID_HIGH);
	StrokeLine(BPoint(LINE2_X, LINE_TOP),
		BPoint(LINE2_X, y + 2),
		B_SOLID_HIGH);
	StrokeLine(BPoint(LINE_RIGHT, LINE_TOP),
		BPoint(LINE_RIGHT, y + 2),
		B_SOLID_HIGH);
}

void
PciView::MouseDown(BPoint point)
{
	char *p;
	int i, no;
	pci_info_tab *tmp;

	if (point.y < LINE_TOP)
		return;
	if ((no = (point.y - LINE_TOP) / WY_SIZE) >= pci_table_size)
		return;
	tmp = pci_table;
	for (i = 0; i < no; i++)
		tmp = tmp->prev;

	if (LINE1_X < point.x && point.x < LINE2_X) {
		p = tmp->VenFull;
	} else if (LINE2_X < point.x && point.x < LINE_RIGHT) {
		p = tmp->ChipDesc;
	} else {
		return;
	}

	if (p)
	{
		BPopUpMenu *my_menu
			= new BPopUpMenu("PCIMenu", TRUE, TRUE, B_ITEMS_IN_COLUMN);
		BMenuItem *item
			= new BMenuItem(p, NULL);
		my_menu->AddItem(item ,0);
		my_menu->Go(point + my_point);
		my_menu->RemoveItem(item);
		delete item;
		delete my_menu;
	}
}

