#include "TreeList.h"

static const float BOX_WIDTH = 6.0;
static const float HALF = BOX_WIDTH/2.0;
static rgb_color LINECOLOR = { 153,153,153,0 };
static rgb_color CURRENCYCOLOR = { 0, 0, 0, 0};

class TreeRoot : public TreeItem
{
	public:
		TreeRoot() : owner(0) {}
		
		void SetOwner( TreeList* l) { owner = l; }
		void ChildAdded( TreeItem*);
		void ChildRemoved( TreeItem*);
		TreeList* Owner() const { return owner; }
		
	private:
		TreeList* owner;
};

void TreeRoot::ChildAdded( TreeItem* who)
{
	who;
	if (owner)
		owner->Refresh( true);
}

void TreeRoot::ChildRemoved( TreeItem* who)
{
	if (owner)
	{
		if ( owner->Current() == who)
			owner->SetCurrent(0);
			
		owner->Refresh( true);
	}
}


TreeView::TreeView( TreeList* _owner, BRect frame)
	:	BView( frame, "treeview", B_FOLLOW_ALL, B_WILL_DRAW | B_NAVIGABLE | B_FRAME_EVENTS),
	    owner( _owner)
{
	SetFontName("Kate");
	SetFontSize(9);
}

TreeView::~TreeView()
{
}

void TreeView::KeyDown( ulong k)
{
	owner->KeyDown(k);
}

void TreeView::Draw( BRect updaterect)
{
	updaterect;
	owner->Redraw();
}

void TreeView::MouseDown(BPoint p)
{
	BMessage* msg = Window()->CurrentMessage();
	if (msg)
	{
		long mods = msg->FindLong( "modifiers");
		long buttons = msg->FindLong( "buttons");
		long clicks = msg->FindLong( "clicks");
		owner->ClickAt(p, mods, buttons, clicks);
	}
}
		
void TreeView::FrameResized( float width, float height)
{
	inherited::FrameResized( width, height);
	owner->RecalculateScrollbars( width, height);
}
 
TreeList::TreeList( BRect frame, 
				    const char* name,
				    const char* label,
				    BMessage* message,
				    ulong resizing_mode,
				    ulong flags
                  ) 
  : BControl( frame, name, label, message, resizing_mode, flags),
    root( new TreeRoot),
    current(0),
    dataheight(0),
    datawidth(0),
    selection_mode(MULTIPLE_SELECT),
    boring(false)
{
	SetFontName("Kate");
	SetFontSize(9);
	
	BRect clientrect( 0, 0, 
	                  frame.Width() - B_V_SCROLL_BAR_WIDTH, 
	                  frame.Height() - B_H_SCROLL_BAR_HEIGHT
	                );
	
	BRect hscrollrect( 0, clientrect.bottom + 1,
	                   clientrect.right, frame.Height()
	                 );
	
	BRect vscrollrect( clientrect.right + 1, 0,
	                   frame.Width(), clientrect.bottom
	                 );
	
	treeview = CreateTreeView( clientrect);
	AddChild( treeview);
	
	AddChild(
		hscroll = new BScrollBar( hscrollrect, "hscroll", treeview, 
		                0, 1000, B_HORIZONTAL)
	);
	AddChild(
		vscroll = new BScrollBar( vscrollrect, "vscroll", treeview, 
		                0, 1000, B_VERTICAL)
	);
	
	root->SetOwner(this);
	
	RecalculateScrollbars( clientrect.Width(), clientrect.Height());
}

TreeView* TreeList::CreateTreeView( BRect frame)
{
	return new TreeView( this, frame);	
}

TreeList::~TreeList()
{
	root->DeleteChildren();
	delete root;
}

void TreeList::Purge()
{
	// Do a deep delete of all children
	root->DeleteChildren();
	Refresh(true);	
}
		
TreeItem* TreeList::Current()
{
	return current;
}

void TreeList::SetCurrent( TreeItem* item)
{
	if ( current != item)
	{
		TreeItem* oldcurrent = current;
		
		current = item;

		if(oldcurrent)
		{
			DrawCurrency( oldcurrent, false);
			oldcurrent->CurrencyChanged(false);
		}
					
		if(current)
		{
			DrawCurrency( current, true);
			current->CurrencyChanged(true);
		}
	}
}

void TreeList::GetSelected( BList& list)
{
	list.MakeEmpty();
	root->FindSelectedChildren( list);
}

TreeItem* TreeList::GetSelected()
{
	return root->FindChild( &TreeItem::isSelected);	
}

bool TreeList::AnySelected()
{
	return root->FindChild( &TreeItem::isSelected) != 0;
}

float TreeList::ItemIndent() const
{
	return boring ? TL_DEFAULT_BORING_INDENT : TL_DEFAULT_INDENT; 
}

float TreeList::BoxIndent() const
{
	return boring ? TL_DEFAULT_BORING_BOX_INDENT : TL_DEFAULT_BOX_INDENT; 
}

bool TreeList::DisplayLines() const
{
	return !boring;
}

void TreeList::SetBoring( bool bored)
{
	if ( boring != bored)
	{
		boring = bored;
		Refresh(true);
	}
}

void TreeList::Refresh( bool full)
{
	RecalculatePositions();

	if( full)
		treeview->Invalidate();
	else
		root->DoForAllChildren( &TreeItem::Invalidate, this);
}

void TreeList::Redraw()
{	
	TreeItem* here = root->Children();
	
	while( here)
	{
		DrawItem( here);
		here = here->Sibling();
	}	
}

void TreeList::AddItem( TreeItem* item)
{
	root->AddChild( item);
}

TreeItem* TreeList::RemoveItem( TreeItem* item)
{
	return root->RemoveChild(item);
}
		
// Open up the whole tree
void TreeList::ExplodeAll()
{
	root->DoForAllChildren(&TreeItem::Explode);

	Refresh(true);
}

void TreeList::SelectAll( bool select)
{
	root->DoForAllChildren( &TreeItem::Select, select);		
	Refresh(false);
}


void TreeList::AllAttached()
{
	BRect frect = treeview->Frame();
	RecalculateScrollbars( frect.Width(), frect.Height());
}

void TreeList::RecalculatePositions()
{
	BPoint where(4,0);
	where = RePosChildren( root, where);
	
	dataheight = where.y;
	
	// Now we know where the last item would go we can recalculate
	// the vertical scrollbar
	BRect frect = treeview->Frame();
	RecalculateScrollbars( frect.Width(), frect.Height());
}

void TreeList::RecalculateScrollbars( float w, float h)
{
	float prop = h / dataheight;
	
	w;
	
	if ( prop >= 1.0)
	{
		vscroll->SetRange(0,0);
		vscroll->SetProportion(1.0);
	}
	else
	{
		vscroll->SetProportion( prop);
		vscroll->SetRange( 0, dataheight - h);
	} 
}

BPoint TreeList::RePosChildren( TreeItem* item, BPoint where)
{
	TreeItem* here = item->Children();
	
	while( here)
	{
		here->SetPosition(where);
		where.y += here->Height();
		if ( !(here->isCollapsed()) && here->HasChildren())
		{
			where.x += ItemIndent();	
			where = RePosChildren( here, where);
			where.x -= ItemIndent(); 
		}
			
		here = here->Sibling();
	}
	return where;
}

void TreeList::ClickAt( BPoint p, long modifiers, long buttons, long clicks)
{
	MakeFocus();

	// Right now I am only interested in primary clicks
	if ( buttons & B_PRIMARY_MOUSE_BUTTON )
	{
		// Check direct children to find a possible hit
		TreeItem* candidate = 0;
		TreeItem* here = root->Children();
	
		while( here)
		{
			candidate = here->ItemAt( p);
			if ( candidate)
				break;
			here = here->Sibling();
		}
	
		if (!candidate)
			return;
		
		// Found the candidate, do a more refined hittest
		int hr = candidate->HitTest( this, p); 
	
		switch( hr)
		{
			case TreeItem::PLUSMINUS_BOX:
				if ( candidate->isCollapsed())
					candidate->Explode();
				else	
					candidate->Collapse();
				
				Refresh(true);
				break;
			
			case TreeItem::LABEL_BOX:
				if ( (clicks == 2) && (Current() == candidate))
				{
					Notify( LIST_INVOKED, candidate);
				}
				else
				{
					if ( (selection_mode == SINGLE_SELECT) || !(modifiers & B_SHIFT_KEY))
					{
						SelectAll(false);
						candidate->Select(true);
					}
					else
					{
						candidate->Select( !candidate->isSelected());
					}
					SetCurrent( candidate);			
					Refresh(false);
				}
				break;
		
			default:
				break;		
		}
	}
}

void TreeList::SetSelectionMode( int mode)
{
	if (mode != selection_mode)
	{
		SelectAll(false);
		selection_mode = mode;
	}
}

void TreeList::DrawCurrency( TreeItem* item, bool on)
{
	if ( item)
	{
		BPoint position = item->Position();
		float h1 = (item->Height() - 1.0);
		BRect  irect( 0, position.y, 3, position.y+h1);
		
		if (on)
		{	
			rgb_color prev = HighColor();
			treeview->SetHighColor(CURRENCYCOLOR);
			treeview->FillRect( irect);
			treeview->SetHighColor(prev);
		}
		else
		{
			treeview->FillRect( irect, B_SOLID_LOW);
		}
	} 
}

void TreeList::DrawItem( TreeItem* item)
{
	if ( item && !item->isHidden())
	{
		// If this is "current" then draw the currency indicator
		if ( item == Current())
			DrawCurrency( item, true);
	
		bool lines_on = DisplayLines();
		
		if ( lines_on)
			DrawParentLine( item);
		if ( item->HasChildren())
		{
			DrawExploder(item);
			if ( !item->isCollapsed())
			{
				if ( lines_on)
					DrawChildLine(item);
					
				TreeItem* child = item->Children();
	
				while( child)
				{
					DrawItem( child);
					child = child->Sibling();
				} 
			}
		}
		item->Draw(this);
	}
}

void TreeList::DrawExploder( TreeItem* item)
{
	BPoint position( item->Position());
	BPoint center( position.x + BoxIndent(), position.y + (item->Height() - 1.0)/2.0);
	
	DrawCenteredImage( treeview, center, ExploderImage( item->isCollapsed()));
}

void TreeList::DrawParentLine( TreeItem* item)
{
	float box_indent = BoxIndent();
	rgb_color prev = HighColor();
	
	treeview->SetHighColor(LINECOLOR);
	BPoint position( item->Position());
	BPoint center( position.x + box_indent, position.y + (item->Height() - 1)/2.0);
	BPoint to( position.x + ItemIndent() + box_indent, center.y);	
	treeview->StrokeLine(center, to);
	treeview->SetHighColor(prev);
}

void TreeList::DrawChildLine( TreeItem* item)
{
	float box_indent = BoxIndent();
	rgb_color prev = HighColor();
	treeview->SetHighColor(LINECOLOR);
	
	BPoint position( item->Position());
	// Child line links the children together, I need to find my
	// last child to draw it...
	TreeItem* lastchild	= item->Children();
	
	if ( lastchild)
	{
		// Scan sibling list until last child is found
		while( lastchild->Sibling())
			lastchild = lastchild->Sibling();
			
		// Draw from the halfway down me, to halfway down him
		BPoint lastchildposition( lastchild->Position());
		float xpos = lastchildposition.x + box_indent;
		BPoint from( xpos,  position.y + (item->Height() - 1.0)/2.0);
		BPoint to( xpos, lastchildposition.y + ((lastchild->Height() - 1.0) / 2.0));
		treeview->StrokeLine( from, to);
	}
	treeview->SetHighColor(prev);
}

void TreeList::Notify( int event, const TreeItem* item)
{
	BMessage* msg = Message();
	
	if (msg)
	{
		// Set extra data members
		if (msg->HasLong( "t_event"))
			msg->ReplaceLong( "t_event", event);
		else
			msg->AddLong( "t_event", event);
			
		if (msg->HasObject( "t_item"))
			msg->ReplaceObject( "t_item", (BObject*)item);
		else
			msg->AddObject( "t_item", (BObject*)item);
			
		// Send the sucker
		Invoke();
	}			
}

void TreeList::KeyDown( ulong k)
{
	long modifiers = 0;

	
	BMessage* msg = Window()->CurrentMessage();

	if (msg)
		modifiers = msg->FindLong( "modifiers");

	TreeItem* curr = Current();
	switch(k)
	{
		case B_UP_ARROW:
			if ( curr)
			{
				TreeItem* next = curr->Prev();
				if (next && (next != root))
				{
//					if ( (selection_mode == SINGLE_SELECT) || !(modifiers & B_SHIFT_KEY))
//						SelectAll(false);
//					next->Select( TRUE);
					SetCurrent( next);
				}
			}
			break;
			
		case B_DOWN_ARROW:
			if ( curr)
			{
				TreeItem* next = curr->Next();
				if (next)
				{
//					if ( (selection_mode == SINGLE_SELECT) || !(modifiers & B_SHIFT_KEY))
//						SelectAll(false);
//					next->Select( TRUE);
					SetCurrent( next);
				}
			}
			break;

		case B_ENTER:
			Notify( LIST_INVOKED, Current());
			break;
		
		case '.':	
		case B_SPACE:
			if ( curr)
			{
				if ( (selection_mode == SINGLE_SELECT) || !(modifiers & B_SHIFT_KEY))
				{
					SelectAll(false);
					curr->Select(true);
				}
				else
				{
					curr->Select( !curr->isSelected());
				}
				Refresh(false);
			}
			break;
			
		case B_RIGHT_ARROW:
			if ( curr)
				curr->Explode();
			Refresh(true);
			break;
			
		case B_LEFT_ARROW:
			if ( curr)
				curr->Collapse();
			Refresh(true);
			break;
			
		default:
			inherited::KeyDown(k);
	}
}


BBitmap* TreeList::ExploderImage( bool collapsed)
{ 
	ImageIndex ino;
	if ( boring)
		ino = collapsed ? TL_RIGHTTRIANGLE : TL_DOWNTRIANGLE;
	else
		ino = collapsed ? TL_PLUSBOX : TL_MINUSBOX;
		
	return GetStandardImage( ino);
}
