#pragma once

#if defined(DEBUG_TREELIST)
#include "Logger.h"
#endif

class TreeList;
class TreeView;
class TreeRoot;

/// TreeItem, Base item that can be added to a TreeList
 /** 
 A TreeItem is an item that can be displayed in a TreeList. A TreeItem
 is heavily dependent on the TreeList class.
 TreeItems control the behaviours that vary from item to item.
 List wide behaviours are the perogative of the TreeList class.
 
 TreeList assumes all it's items are TreeItems, but
 you can derive from this to get notifications, add your own data
 and overide methods.  It is done this way so that trees that
 contain different types of items do not have to spend time
 switching on the item type.
 */
//@See: TreeList TreeView
class TreeItem : public BObject
{
	public:
		//@Man: Constructors
		/**
		Create a TreeItem, optionally providing a name.  Items will not
        be displayed until they are inserted into a list.
		*/
		//@{
		/// Construct a nameless TreeItem
		TreeItem();
		/// Constructor
		TreeItem( const char* name);  // Name is copied in
		//@}
		
		
		/// Destructor
		/**
		  Frees up any dynamically allocated objects, also calls DeleteChildren to
          free items that are children of this one.  A consequence of this is that
          ListItems should always be dynamically allocated.
		*/
		virtual ~TreeItem();

		/// Delete all my descendents.
		void DeleteChildren();
		
		//@Man: Tree builders
		/** The next group of functions manipulate the tree
		None of them copy or delete the items concerned
	    Items are held by reference.
		Various ways to Add, added items MUST NOT have siblings
		before the addition!
		*/
		//@{
		/// Adds a child at a specified position, by default at the end
		/**
		Add a child to this one.  The second parameter determines where the child is
		added, 0 indicates that the added child should be first, 1 after the first,
		2 after the second etc.  The default is to tag the child on at the end.
		*/
		virtual void AddChild( TreeItem* item, int at = INT_MAX);
		
		
		/// Adds an item as a sibling
		/**
		Add an item as the immediate sibling to this one.  An example may help...
		assume that we have a parent P and three children A, B and C; if I now		
		apply AddSibling to B with a new child X then I end up with the parent P and
		four children A, B, X and C.

		I could achieve the same with Addchild, but AddSibling is often useful when
		the only context is that I want to add an item after the current one (as
		happens often when adding items to a list).
		*/
		virtual void AddSibling( TreeItem* item);
		/// Adds an item as the first child
		/**
		Adds a new first child to the parent, all the existing children move on.
		This is just an inline for AddChild( item, 0).
		*/
		void AddFirstChild( TreeItem* item) { AddChild( item, 0); }
		//@}
		
		/// Remove may return 0 if item not found
		/**
		Looks for the provided item as a child of the item this method is applied to
		then removes it from the tree.  This function will search as deep as it needs
		to to find the child.  When the child is removed all its children come with it.
		If the child is not found this function returns 0.
		*/
		TreeItem* RemoveChild( TreeItem*);
		
		//@Man: Simple iterators
		/**
		These are the simplest iterators, and are very lightweight, all inlines for
		minimal overhead as we spend a lot of time chasing all over the tree.
		They all return 0 if the required item is not found.
		*/
		//@{
		/// Finds immediate sibling of current item
		TreeItem* Sibling() const { return sibling; }
		/// Finds first child of current item
		TreeItem* Children() const { return children; }
		/// Finds parent of current item
		TreeItem* Parent() const { return parent; }
		//@}
		
		//@Man: Visual iterators
		/**
		These are the iterators move over the list in visible order rather than
		simply tree order as is the case with Children, Parent and 
		Sibling.
		Since these functions have to "flatten" the list to traverse it and pay
		attention to whether something is hidden or not, they are rather more expensive
		than the simple iterators.  These are used (for example) when the list cursor
		is moved up and down.
		*/
		//@{
		/// Returns the item immediately before this one
		TreeItem* Prev() const;
		/// Returns the item immediately after this one
		TreeItem* Next() const;
		/// Searches for the last visible child of this item
		TreeItem* Last() const;
		//@} 
		
		//@Man: Method applicators
		/** These are used to apply a TreeItem method to all
		    connected children (preorder).
		*/
		//@{
		/// Apply function with no parameters
		void DoForAllChildren( void (TreeItem::*mf)()); 
		/// Apply function with no parameters
		void DoForAllChildren( void (TreeItem::*mf)(bool), bool p); 
		/// Apply function with no parameters
		void DoForAllChildren( void (TreeItem::*mf)(TreeList*), TreeList* p);
        //@}
        		
		/// Search for descendant that satisfies the supplied predicate
		/**
		In a similar fashion to the DoForAllChildren method, this method applies
		a simple predicate method to all children, in preorder.
		Unlike the DoForAllChildren methods this one stops as soon as the predicate
		returns true, in this case FindChild returns a pointer to the child that
		returned true.  If no children satisfy the predicate then FindChild returns 0.
 
		As an example, by calling "item->FindChild(&TreeItem::isCollapsed)"	one can
		get a pointer to the first collapsed child.

		*/
		TreeItem* FindChild( bool (TreeItem::*mf)()); 
		
		//@Man: Predicates
		/** These are used to test for various binary
		    conditions.  The apply only to the item in question.
		    These may usefully be used in conjunction with FindChild
		    to look for a descendant that satisfies the given predicate.
		*/
		//@{
		/// Returns true if item is in the selected state
		bool isSelected() const { return flags & SELECTED; }
		
		/// Returns true if the item is hiding its descendents
		bool isCollapsed() const { return flags & COLLAPSED;};		
		
		/// Returns true if one of my ancestors is collapsed 
		bool isHidden() const;
		
		/// is item in a state where it needs a redraw	
		bool isInvalid() const { return flags & INVALID; }
		//@}
		
		/// Get iconic representation of item
		/**
		This is a virtual function which is used to obtain the icon (BBitmap) which
		is displayed to indicate the type and state of the item.  It is very likely
		that you will need to override this.  The BBitmap returned is not a copy and
		the caller must not delete it.  Most items will need to return one of two
		bitmaps, one for the selected state and one for the non-selected state.

		A good size of icon to return is a MiniIcon, although other sizes are fine and may be
		used to good effect if the item height is changed.

		If this function returns 0 no icon will be displayed and the item text will be
		moved over to fill the space that would have been occupied otherwise.

		Normally selection is displayed using a dimmed form of the icon, a 
		static helper function, CreateDimBitmap is available to create
		such a bitmap.
		*/ 		
		virtual const BBitmap* Icon() const;
		
		/// Sets the stored text for the item
		/**
		Unlike the Icon, the text label is stored in the base class.  Label
		is used to access it and SetLabel is used to set the text.
		SetLabel copies the provided text into an internal buffer.  It is OK
		to set the text using a NULL pointer, this clears out the internal stored text.
		*/
		void SetLabel( const char* newtext);
		
		/// Accesses the stored text for the item
		const char* Label() const { return label; }
		
		/// Returns true if item has any children
		/**
		The result of this function defines whether or not the exploder box is drawn
		for this item.  The base class simply returns true if the current item has children
		attached.

		The important subtlety here is that by overiding this you can display a sparse
		tree.  This is a tree
		where not all the children are actually held in the tree, only the visbile childre
		need be around.  This may be very useful when
		displaying a very large list, for example a file tree.  As long as this function
		tells whether the item <I>may</I> have children or not, the actual
		TreeItem children need only be created when the item is exploded, and the children
		can be freed again when the item is collapsed.
		*/
		virtual bool HasChildren() const;
		
		/// Accesses the current position of the item
		/**
		This lets you find out where the top-left of the items rectangle is positioned.
		The position is set dynamically by the TreeList control
		the layout of the items.
		*/
		const BPoint& Position() const { return position; }
		
		/// Gets the recommended height for the item
		/**
		The Height is an attribute of the item that normally stays the same for
		the lifetime of the item.  Height defaults to a good size for MiniIcons, but can
		be modified (though derivation) to provide a value that is more appropriate for
		larger (or smaller) images.
		*/ 
		virtual float Height() const;
		
		/// Find out which list (if any) this item is in.
		/**
		Access a pointer to the TreeList control that this item belongs to.  This may
		return NULL if the item is not part of a tree rooted on a TreeList.

		This is virtual for a mainly internal reason.  A TreeList control accesses all
		its items through a special internal derivation of a TreeItem (this is a little
		like the root view that is part of a BWindow I guess).  Base class TreeItems
		return the Parents Owner, of NIL if they do not have a Parent.  The special
		root Item in the TreeList "knows" to return a pointer to the TreeList.

		I cannot think of a good reason for user derived classes to overide this
		function.
		*/		
		virtual TreeList* Owner() const;

		/// Augments the provided list with all selected descendents
		/**
		Looks for all descendants that are selected.  Adds any it finds to the provided
		list and then returns the number of children it added.

		This does NOT clear the list before the addition.  This function only considers
		descendants, not the item itself.
		*/
		int FindSelectedChildren( BList& results);
		
		//@Man: Hit tests
		/**
		Hit testing is the mechanism the list uses to determine what is under
		the mouse pointer.
		*/
		//@{
		/// Enumeration for result of HitTest method
		/** 
		The base class
		only uses the provided values.  A more sophisticated derived
		class could add to this since "HitTest" returns an "int"
		*/
		enum HitResult {
			/// Not on this item at all 
			NOHIT,
			/// On the exploder box
			PLUSMINUS_BOX, 
			/// On the icon or label area
			LABEL_BOX 
		};
		
		/// Tells which area in this object is underneath the point
		/**
		Checks which area of the subject item is under the given point.  Returns
		an integer to categorize the area.  The base class will only return one of
		the values defined in the HitResult enumeration.

		In derived items that have custom drawing routines, this method will
		probably need to be overridden to ensure that mouse-clicks work correctly.
		*/ 
		virtual int HitTest( TreeList* owner, BPoint where);
		 
		/// Look for the item under the point provided
		/**
		Looks for a hit anywhere over this item or any of it's descendants (this does not
		recurse into hidden children).  Returns a pointer to the hit child, or NIL
		for no child found.
		*/
		TreeItem* ItemAt( BPoint where);
		//@}
		
		/// Notify
		/**		
		This is a key function to override in your items if you want to know
		what is going on at an item level.  Alternatives are to overide the similar
		function on the list itself, or to wait for messsages to arrive.

		The base class implementation simply invokes TreeList::Notify to pass on
		the event, so be careful when overiding to decide whether or not you want to
		send these notification.

		The integer parameter defines what happened.  Values are defined in the
		enumeration TreeList::EventTypes.  You will only see selection events right
		now.  Derived classes can augment this with other values.
		*/		
		virtual void Notify( int event);
		
		/// Collapse this item, hiding all children
		virtual void Collapse();
		
		/// Explode this item, showing its first level children
		virtual void Explode();
		
		/// Controls selection status for this child
		virtual void Select( bool select);
		
		/// Modifies position stored for child
		virtual void SetPosition( BPoint newpos);
		
		//@Man: Births and deaths
		/**
		Notifications are passed up the tree when items are added or removed.
		These methods pass this up the parentage chain to the root, where they
		cause a refresh of the list.
		*/
		//@{
		/// Notifies when a new child is added, invoked on parent		
		virtual void ChildAdded( TreeItem* new_child);
		
		/// Notifies when an existing child has been removed, invoked on ex-parent.
		virtual void ChildRemoved( TreeItem* ex_child);
		//@}

		/// Called whenever an item gains or loses currency		
		virtual void CurrencyChanged( bool current);
		
		/// Draw item in view provided, uses DrawIcon and DrawLabel
		virtual void Draw( TreeList* owner);
		
		/// Conditionally Invalidate this items rectangle
		/** 
		This function determines whether the item could be visible
		an if so calculates its rectangle and invalidates that area of
		the list provided.
		*/
		virtual void Invalidate( TreeList* owner);
		
		/// Creates a new bitmap based on an old one, but with all the colors subdued
		/**
		 A static function to perform a common BBitmap function
		 This returns a new copy of the provded bitmap with all
		 the colors somewhat subdued, this imitates what the browser
		 does.
		*/
		static BBitmap* CreateDimBitmap( const BBitmap* bitmap);
		
	protected:
		//@Man: Building blocks for Draw of an item
		/**
		When items need to draw themselves they call on these functions to draw to
		icon and text.  You can override these to change the appearence, either to
		get a different look, or to take into account other information.
		*/
		//@{
		/// Draws the icon for the list item
		virtual void DrawIcon( TreeList* owner);
		/// Draws the label for the list item
		virtual void DrawLabel( TreeList* owner);
		//@}

	private:
		// Items cannnot be assigned or copied! Why can't Be do this?
		TreeItem( const TreeItem&);
		TreeItem& operator=(const TreeItem&);
		
		// Items are linked via these next three
		TreeItem* sibling;
		TreeItem* children;
		TreeItem* parent;
				
		// A few binary flags stuffed into an int
		enum FLAG_BITS { SELECTED = 1, COLLAPSED = 2, INVALID = 4 };
		int flags;
		
		// Other variables
		BPoint position;   // top-right of my rectangle
		char* label;       // text label for this item
};



/// TreeView, a type of BView
/**
 This view is used to render the graphics
 This is used as a child of the main view and is the target of the
 scrollbars.

 Normally you will not need to worry about this class, either from
 a point of view of deriving from it or understanding it.
 It is really just a by-product of the way scrolling views are coordinated
 in the Be Interface Kit
*/
class TreeView : public BView
{
	public:
		TreeView( TreeList* owner, BRect rect);
		virtual ~TreeView();
	
		// Process a few events, these are (in one way or enother)
		// passed off to the enclosing view
		virtual void MouseDown( BPoint);
	    virtual void FrameResized( float width, float height);
		virtual void Draw( BRect updaterect);
		virtual void KeyDown( ulong);

	private:
		TreeList* owner;		
};


/// A list control that displays a tree structure
/**
A TreeList is a view that displays a list of items the user can select and invoke.
TreeList is heavily dependent on the TreeItem class.  The TreeList is really
a composite object consisting of two BScrollbars, a child view of class TreeView to render the graphics
and a collection of TreeItems.

The list keeps track of pointers to TreeItems.  Adding an item to the list adds only the pointer; the data itself isn't copied.
The TreeList provides methods to render list-wide aspects of the list such as the connecting lines, and the exploder boxes.  It
also keeps a track of the current item.  However, much of what you may expect the TreeList to do is offloaded to the TreeItems.
As a general policy, things that may change from item to item are atrributes of and methods on the items rather than the list.

When the contents of the list change, the TreeList makes sure the visible list on-screen is updated.  TreeItems know that they
are attached to a list, but in some circumstances you will have to manually ask to redraw the list by calling
the TreeList::Redraw function or the 
TreeView's Invalidate.

The user can click an item in the list to select it and double-click an item to both select and invoke it.
The user can also select and invoke items from the keyboard.
The navigation keys (such as Down Arrow and Up Arrow) select items; Enter invokes the item that's currently selected.

The TreeList highlights the selected item, but otherwise it doesn't define what,
if anything, should take place when an item is selected.
You can determine that yourself by registering a "selection message" (a BMessage object)
that should be delivered to a target destination whenever the user selects an item. 

Similarly, the TreeList doesn't define what it means to "invoke" an item.  You can register a separate "invocation message" that's posted whenever the user double-clicks an item or presses Enter while an item is selected.  For example, if the user double-clicks an item in a list of file names, a message might be posted telling the BApplication object to open that file.

A TreeList doesn't have a default selection message or invocation message, invocation message functions are all inherited from BControl.
*/
class TreeList : public BControl
{
	friend class TreeView;
	public:
	
		/// Constructor like any other BControl
		TreeList( BRect frame, 
				  const char* name,
				  const char* label,
				  BMessage* message,
				  ulong resizing_mode = B_FOLLOW_LEFT | B_FOLLOW_TOP,
				  ulong flags = 0
		);
		
		/// Destructor which destroys any items this list refers to.
		/**
		Frees up any dynamically allocated objects, also calls Purge to free all memory
		allocated to hold the list of items.  A consequence of this is that ListItems
		should always be dynamically allocated.
		*/
		virtual ~TreeList();
		
		/// Types of events that the base class will raise
		enum EventTypes {
			/// User double-clicked on an item
			LIST_INVOKED,
			/// An item became selected 
			ITEM_SELECTED,
			/// An item lost selection 
			ITEM_DESELECTED
		};
		
		/// Uses BMessages to notify the user of the list that something has happened
		/**
		This method is called for anything that needs to be able to send a message (via Invoke)
		to the outside world, this function builds on the functionality of Invoke.

		Invoke is not used directly as special values need to be set in the
		model message.  You can overide this to avoid using BMessages, or to add data of your own.
		The parameter eventtype is an integer identification of the particular type of event
		(It would be nice if BControl handled this, but it doesn't). TreeItems and TreeLists only use
		values from the EventTypes enumeration, but any integer is OK, this may be useful in derived classes
		so that events the bases classes are unaware of can still be sent using the default implementation.

		Internally this function adds two new pieces of information to the model BMessage, <I>eventtype</I>
		as a B_LONG_TYPE called "t_event" and "item" as a B_OBJECT_TYPE called "t_item".  Because of the
		built in action of BControls Invoke you can also rely on "when" and more usefully "source" being added
		("source" being a pointer to the TreeList). 
		*/
		virtual void Notify( int eventtype, const TreeItem* item);

		//@Man: Adding and removing root items
		/**		
		Add or remove items to/from top level of tree (yes, a tree
		can have multiple roots).  To add/remove to lower levels
		use methods on ListItems.  These methods are really shorthand for
		getting the root item then applying TreeItem methods to that
	    These methods do not copy or delete any items,
		all items are held by reference.
		*/
		//@{
		/// Add an item
		virtual void AddItem( TreeItem*);
		/// Remove an item
		virtual TreeItem* RemoveItem( TreeItem* item);
		//@}

		/// Get rid of all items (recursively) and delete them
		virtual void Purge();

		/// Get at the root of the list
		/**
		Get at the real root of the list.  Although the tree may look as if it has multiple roots
		in fact there is a "magic" single root that all TreeLists own from the time they are constructed
		until they die.  The root is an invisible item and should never be
		deleted unless you really want to test the robustness of your BeBox.
		Iterate over the roots children to get at the items you added.
		The root item never has a sibling (and please don't add one, I wouldn't like it).
		*/
		TreeItem* Root() { return (TreeItem*)root; }
		
		/// Open up the whole tree, I wanna see it all
		virtual void ExplodeAll();
		
		/// Select or deselect everything
		virtual void SelectAll( bool select);
		
		//@Man: List currency.
		/**
		This is mainly to help with keyboard navigation of the list, the current
		is the one to which keystrokes apply.
		
		This may have nothing to
		do with selection, although this is often updated as selection
		precedes.
		*/
		//@{
		/// Find out the current item (may be 0)  
		TreeItem* Current();
		
		/// Set the current item (you can pass 0 so that nothing is current)
		virtual void SetCurrent( TreeItem* item);
		//@}
				
		/// Get a list of all items in list that are selected
		/**
		When getting a list of selected items, the list is cleared on entry.  Be careful
		when working on the returned list since the list may become invalidated by other
		operations.  This is particularly evil when deleting items.  The safest way to remove
		selected items from the list is to loop getting the first selected and remove them one by one.
		*/
		void GetSelected( BList&);
		
		/// Get a pointer to the first item in the list that is selected
		TreeItem* GetSelected();
		
		/// Return true if any items in list are selected
		bool AnySelected();

		/// Constants used to determine the style of selection
		enum SelectionStyle {
			/// Allow selection of a maximum of one item 
			SINGLE_SELECT,
			/// No maximum to the number of items that may be selected
			MULTIPLE_SELECT
		};
		
		/// Set the selection style		
		virtual void SetSelectionMode( int mode);
		
		/// Returns the current selection style
		int SelectionMode() { return selection_mode; }
		
		/// Refreshes the visuals of the list
		/**
		When you change the list via purely item level operations
		so the list is not aware of this, you better call this
		to make sure the display gets updated
		Set the flag to TRUE to get everything updated, FALSE
		if the list layout was unaltered (selection,labels,etc.)
		*/
		virtual void Refresh( bool refresh_all);
		
		//@Man: Control general look of tree
		/**
		The values these virtuals return control some of the overall look
		of the tree.  The base class does quite a lot and implements two
		overall styles, interesting and boring.
		*/
		//@{
		/// How much to indent each child from its parent
		virtual float ItemIndent() const;
		/// How far is the exploder box from the left of the item
		virtual float BoxIndent() const;
		/// Should I bother to draw those lines at all
		virtual bool  DisplayLines() const;
		//@}
		
		/// Wanna be tedious - make it so
		/**
		OK, so this betrays my bias, pass true in here to get no lines
		and dull little rotating triangles, I guess it does save some space though!
		*/
		void SetBoring( bool boring);
		
		/// How boring am I anyway.
		bool Boring() const { return boring; }
		
		/// Access the image used for the exploder box
		/**
		The exploder box is the little gizmo the user clicks on to collapse or
		explode items that have children.  The list controls these images rather than
		the item because we've gotta have some consistency, right?
		
		Pass in true to get the image for a collapsed parent, false to get the image
		for an exploded one (I always wanted to explode my parents).
		
		The base class will provide either the [+] [-] thingies or the rotating triangle
		thing.
		*/
		virtual BBitmap* ExploderImage( bool collapsed);
		
		/// Returns the child view I render my graphics into
		/**
		Provide access to the view used to render that graphics.  Logically the TreeList is the whole thing
		so pretty much all the operations are defined on it.  In fact the TreeList is an enclosing view for
		three subviews, two scrollbars and the view used to render the graphical depiction of the list.  Since you
		may want to get access to this child view I give you a direct way to get to it.  This is mainly used by ListItems
		to decide which view they should render into.
		*/
		TreeView* TView() { return treeview; }
		
		// Overrides from base class
		virtual void AllAttached();
		virtual void KeyDown( ulong);
		
		/// I have a few built-in images, this enumeration lists them
		enum ImageIndex {
			/// Little blue triangle, pointing downward 
			TL_DOWNTRIANGLE, 
			/// Equally blue triangle pointing right
			TL_RIGHTTRIANGLE,
			/// Small box with a minus sign in it [-]
		    TL_MINUSBOX,
		    /// Samll box with a plus sign in it [+]
		    TL_PLUSBOX,
		    /// This marks the end of the enumeration
		    TL_IMAGECOUNT
		};
		
		/// Roll-up, get your fresh images here
		static BBitmap* GetStandardImage( ImageIndex image);

		/// A helper function for drawing images in a view centered on a point 
		static void DrawCenteredImage( BView* v, BPoint where, const BBitmap* bitmap); 		

	protected:
		// Used to pass off mouse clicks from child view
		/**
		Since the actual mouse-clicks come in on a sub-view, it passes them off to this virtual
		after decoding the message.  All the real handling of the click goes on in here.
		The parameters are simply the like-named entries in the message.
		*/
		void ClickAt( BPoint, long modifiers, long buttons, long clicks);
		
		/// Recalculate the scrollbars based on new sizes
		void RecalculateScrollbars( float width, float height);
		
		/// Force a recalculation of all child positions.
		virtual void RecalculatePositions();
	
		/// Used to create my child view
		/**
		This is virtual in the unlikely event that you want to replace
		the view I use to render into!
		*/
		virtual TreeView* CreateTreeView( BRect rect);
		
		/// Draw everything
		/**
		Entirely redraws the list view content.  Useful if you are not sure
		everything is up-to-date.  In general the list will keep itself up to
		date bu who knows
		*/
		virtual void Redraw();
		
		/// Draw an item (and all it's visible children)
		/**
  		 This is the key drawing
		 function, and it delegates all the actual drawing to helper functions.  This function
		 is concerned with the logic of drawing, determining the order of drawing
		 and recursing through all the children.
		 */
		 virtual void DrawItem( TreeItem*);

		//@Man: Draw Methods
		//@Memo: Building blocks for DrawItem
		/** The various drawing operations are broken down
		    into a number of "building-block" functions.
		    This is intended to make it easy to modify parts of the
		    redraw mechanism without having to reimplement
		    all of it.
		*/
		//@{
		/// Draw exploder box
		virtual void DrawExploder( TreeItem*);
		/// Draw line back to parent item
		virtual void DrawParentLine( TreeItem*);
		/// Draw line connecting children of this item
		virtual void DrawChildLine( TreeItem*);
		/// Draw indication of which is the "current" item
		virtual void DrawCurrency( TreeItem*, bool on); 
		//@}
		
	private:		
		BPoint RePosChildren( TreeItem* item, BPoint where);
		
		// Guess what these do
		int selection_mode;
		bool boring;
		const BBitmap* collapsed_image;
		const BBitmap* exploded_image;
		TreeView* treeview;
		BScrollBar* vscroll;
		BScrollBar* hscroll;		
		TreeRoot* root;		
		TreeItem* current;
		float dataheight;
		float datawidth;			

	public:
	    /// Magic values
		/** A few default values, so you can see what I return when
		 you call ItemIndent or BoxIndent or accept default item height
		*/
		enum Interesting {
		    /// Amount by which each level is indented
			TL_DEFAULT_INDENT = 19,
			/// Offset of center of exploder box 
			TL_DEFAULT_BOX_INDENT = 10,  
			/// Height of each item
			TL_DEFAULT_ITEM_HEIGHT = 19 
		};
		
	    /// More magic values
		/** A few default values, so you can see what I return when
		 you call ItemIndent or BoxIndent or accept default item height.
		 This set is used when the list is set to boring mode.
		*/
		enum Boring {
			/// Indent value when in boring mode
			TL_DEFAULT_BORING_INDENT = 9,
			/// Offset of center of exploder box 
			TL_DEFAULT_BORING_BOX_INDENT = 4,
			/// Height of boring items
			TL_DEFAULT_BORING_ITEM_HEIGHT = 12 
		};

};
