#include "TreeList.h"

/* This Image is 16 pixels wide, 16 pixels high, and 8 bits deep */
static unsigned char default_bitmap_data[] = 
{ 
0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0xFA, 0xFA, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0xFA, 0xFA, 0xFA, 0xFA, 0xFA, 0x00, 0x00, 0xFF, 0xFF, 0xFF,
0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0x1F, 0xFA, 0xFA, 0xFA, 0xFA, 0x1F, 0x5D, 0x00, 0xFF, 0xFF, 0xFF, 
0xFF, 0xFF, 0xFF, 0xFF, 0x00, 0xF9, 0x1F, 0x1F, 0xFA, 0x1F, 0x5D, 0x5D, 0x00, 0xFF, 0xFF, 0xFF,
0xFF, 0xFF, 0xFF, 0x00, 0x00, 0xF9, 0xF9, 0xF9, 0x1F, 0x5D, 0x5D, 0x5D, 0x00, 0xFF, 0xFF, 0xFF,
0xFF, 0xFF, 0x00, 0x60, 0x01, 0xF9, 0xF9, 0xF9, 0xF9, 0x5D, 0x5D, 0x5D, 0x00, 0x00, 0xFF, 0xFF,
0xFF, 0x00, 0x60, 0x60, 0x01, 0xF9, 0xF9, 0xF9, 0xF9, 0x5D, 0x5D, 0x5D, 0x00, 0xA3, 0x00, 0x00,
0x00, 0x1F, 0x60, 0x60, 0x60, 0x01, 0xF9, 0xF9, 0xF9, 0x5D, 0x5D, 0x00, 0xA3, 0x1F, 0x2D, 0x00,
0x00, 0x86, 0x1F, 0x1F, 0x60, 0x1F, 0x00, 0x00, 0xF9, 0x5D, 0x00, 0xA3, 0x1F, 0x2D, 0x2D, 0x00,
0x00, 0x86, 0x86, 0x86, 0x1F, 0xD5, 0x27, 0x00, 0x00, 0x00, 0xA3, 0x1F, 0x2D, 0x2D, 0x2E, 0x00,
0x00, 0x86, 0x86, 0x86, 0x86, 0xD5, 0x28, 0x01, 0xCA, 0xCA, 0xA3, 0xA3, 0x2D, 0x2D, 0x2E, 0x00,
0x00, 0x86, 0x86, 0x86, 0x86, 0xD5, 0xD5, 0x00, 0xCA, 0xA3, 0xA3, 0xA3, 0x2D, 0x2D, 0x2D, 0x00,
0x00, 0x86, 0x86, 0x86, 0x86, 0xD5, 0xD5, 0x00, 0xA3, 0xA3, 0xA3, 0xA3, 0x2D, 0x2D, 0x2E, 0x01,
0xFF, 0x00, 0x00, 0x86, 0x86, 0xD5, 0xD5, 0x01, 0x00, 0x00, 0xA3, 0xA3, 0x2D, 0x2E, 0x00, 0x11, 
0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x11, 0x11, 0xFF, 0x00, 0x00, 0x00, 0x00, 0x11, 0x11,
};

BBitmap* TreeItem::CreateDimBitmap( const BBitmap* map)
{
	static bool initialized = false;
	static uchar colortable[256];

	// Check validity of image
	if (!map || (map->ColorSpace() != B_COLOR_8_BIT))
		return 0;
	
	if( !initialized)
	{
		color_map *systemColors = system_colors();
		for (int i = 0; i < 256; ++i)
		{		
			rgb_color aColor = systemColors->color_list[i];
			aColor.red = (long)aColor.red * 2 / 3;
			aColor.green = (long)aColor.green * 2 / 3;
			aColor.blue = (long)aColor.blue * 2 / 3;
			colortable[i] = index_for_color(aColor);
		}
		initialized = TRUE;
	}
	
	// Create the bitmap to fill...
	BBitmap* bitmap = new BBitmap( map->Bounds(), B_COLOR_8_BIT);
		
	uchar* src = (uchar*)map->Bits();
	uchar* dest = (uchar*)bitmap->Bits();
	
	long datasize = map->BitsLength();
	
	for (long i = 0; i < datasize; ++i)
	{
		uchar index = *src++;
		
		// Do not touch transparency
		if ( index != B_TRANSPARENT_8_BIT)
			*dest++ = colortable[index];
		else
			*dest++ = B_TRANSPARENT_8_BIT;
	}

	return bitmap;
}


TreeItem::TreeItem() :
		sibling(0),
		children(0),
		parent(0),
		flags(INVALID),
		position(0,0),
		label(0)
{
}

TreeItem::TreeItem( const char* name)
	:
		sibling(0),
		children(0),
		parent(0),
		flags(INVALID),
		position(0,0),
		label(0)
{
	SetLabel(name);
}

TreeItem::~TreeItem()
{
#ifdef LOGGING
	if ( label)
		LogPuts( label, LOGGER_INFORMATIONAL, "Being deleted");
	else
		LogPuts( "Unknown", LOGGER_INFORMATIONAL, "Being deleted");
#endif
		
	if ( parent)
	{
		// Make sure there are no dangling links
		parent->RemoveChild(this);
	}
	
	DeleteChildren();
	
	SetLabel(0);
}


TreeItem* TreeItem::Next() const
{
	// Find the next, do not go down if collapsed
	if ( isCollapsed() || (children == 0))
	{
		// Nobody to go down to...
		// look for a sibling
		const TreeItem* here = this;
		while ( here)
		{
			if ( here->Sibling())
				return here->Sibling();
				
			here = here->parent;
		} 
	}
	else
	{
		return children;
	}
	
	return 0;
}

TreeItem* TreeItem::Prev() const
{
	if (parent)
	{
		// See if I'm the first child!
		if ( parent->children == this)
			return parent;
			
		// Not the first child, find my previous sibling
		TreeItem* prevsib = parent->children;
		while( prevsib)
		{
			if ( prevsib->sibling == this)
			{
				// Found the previous sibling, now
				// go to the deepest visible child of that sibling,
				// otherwise use the previous sibling
				return prevsib->Last();
			}
			prevsib = prevsib->sibling;
		} 	
	}
	return 0;
}

TreeItem* TreeItem::Last() const
{
	if ( isCollapsed() ||  (children == 0))
		return (TreeItem*)this;
		
	TreeItem* child = children;
	while(child->sibling)
		child = child->sibling;

	return child->Last();
}

void TreeItem::DeleteChildren()
{
	while(children)
	{
		TreeItem* target = children;
		RemoveChild(target);
		
		// rely on desturctor to delete sub-children.
		delete target;
	}
}


float TreeItem::Height() const
{
	// Use this to get good spacing for standard miniicons
	TreeList* owner = Owner();
	
	if ( owner && owner->Boring())
		return TreeList::TL_DEFAULT_BORING_ITEM_HEIGHT;
	else 
		return TreeList::TL_DEFAULT_ITEM_HEIGHT;
}

const BBitmap* TreeItem::Icon() const
{
	static BBitmap* normal = 0;
	static BBitmap* selected = 0;
	
	if (!normal)
	{
		// Once only, create the bitmaps
		normal = new BBitmap( BRect(0,0,15,15), B_COLOR_8_BIT);
		::memcpy( normal->Bits(), default_bitmap_data, normal->BitsLength());
		selected = CreateDimBitmap(normal); 
	}
	
	return isSelected() ? selected : normal;
}

void TreeItem::SetLabel( const char* newlabel)
{
	if ( label)
		free(label);
	
	if ( newlabel)	
		label = strdup(newlabel);
	else
		label = 0;

	flags |= INVALID;
}

void TreeItem::AddSibling( TreeItem* item)
{
	item->sibling = sibling;
	item->parent = parent;
	sibling = item;
	
	if ( parent)
		parent->ChildAdded(item);	
}
		
void TreeItem::AddChild( TreeItem* item, int at)
{
	if ( children == 0 || at == 0)
	{
		item->sibling = children;
		children = item;
	}
	else
	{
		TreeItem* here = children;
		while( --at && here->sibling)
			here = here->sibling;
		item->sibling = here->sibling;
		here->sibling = item;		
	}
			
	item->parent = this;
	flags |= INVALID;
	
	ChildAdded(item);
}

// Always deep...
TreeItem* TreeItem::RemoveChild( TreeItem* item)
{
	if ( children != 0)
	{
		if ( children == item)
		{
			children = item->sibling;
			item->sibling = 0;
			item->parent = 0;
			
			if ( !isCollapsed())
				ChildRemoved(item);
			
			return item;
		}
		else
		{
			TreeItem* here = children;
			
			do
			{
				TreeItem* next = here->sibling;
				if (next == item)
				{
					here->sibling = item->sibling;
					item->sibling = 0;
					item->parent = 0;
					if ( !isCollapsed())
						ChildRemoved(item);
						
					return item;
				}
				
				if ( here->RemoveChild(item))
					return item;
					
				here = next;	
			} while(here);
		}
	}
	return 0;
}

// Pass addition notification up the tree...
void TreeItem::ChildAdded( TreeItem* who)
{
	if ( parent)
	{
		parent->ChildAdded( who);
	}
}

TreeList* TreeItem::Owner() const
{
	return parent ? parent->Owner() : 0; 
}


// Pass removal notification up the tree...
void TreeItem::ChildRemoved( TreeItem* who)
{
	if ( parent)
	{
		parent->ChildRemoved( who);
	}
}

int TreeItem::FindSelectedChildren( BList& results)
{
	int count = 0;
	
	TreeItem* here = Children();
	
	while( here)
	{
		if (here->isSelected())
		{
			count++;
			results.AddItem(here);
		}
		
		TreeItem* next = here->sibling;
		count += here->FindSelectedChildren(results);
		here = next;
	}
	
	return count;		
}

void TreeItem::DoForAllChildren( void (TreeItem::*mf)())
{
	TreeItem* here = Children();
	
	while( here)
	{
		(here->*mf)();
		here->DoForAllChildren(mf);
		here = here->sibling;
	}		
}

void TreeItem::DoForAllChildren( void (TreeItem::*mf)( bool), bool p)
{
	TreeItem* here = Children();
	
	while( here)
	{
		(here->*mf)(p);
		here->DoForAllChildren(mf, p);
		here = here->sibling;
	}		
}

void TreeItem::DoForAllChildren( void (TreeItem::*mf)(TreeList*), TreeList* p)
{
	TreeItem* here = Children();
	
	while( here)
	{
		(here->*mf)(p);
		here->DoForAllChildren(mf, p);
		here = here->sibling;
	}		
}

TreeItem* TreeItem::FindChild( bool (TreeItem::*mf)())
{
	TreeItem* here = Children();
	
	while( here)
	{
		if ( (here->*mf)())
			return here;
		
		TreeItem* found = here->FindChild(mf);
			
		if( found)
			return found;
			
		here = here->sibling;
	}
	
	return 0;		
}


void TreeItem::Collapse()
{
	if ( HasChildren() && !isCollapsed())
	{
		flags |= INVALID | COLLAPSED;
	}
}

void TreeItem::Explode()
{
	if ( HasChildren() && isCollapsed())
	{
		flags &= ~COLLAPSED;
	
		// Force all immediate children to be collapsed
		TreeItem* child = children;	
		while( child)
		{
			child->flags |= COLLAPSED;
			child = child->sibling;
		}
	}
}

bool TreeItem::HasChildren() const
{
	return (children != 0);
}		
		
void TreeItem::Select( bool select)
{
	if (isSelected())
	{
		// Currently selected, deselect it
		if (!select)
		{
			flags &= ~SELECTED;
			flags |= INVALID;
			Notify( TreeList::ITEM_DESELECTED);
		}
	}
	else
	{
		if (select)
		{
			// Not currently selected, select it
			flags |= (SELECTED | INVALID);
			Notify( TreeList::ITEM_SELECTED);
		}
	}
}

void TreeItem::Notify( int event)
{
	TreeList* owner = Owner();
	if (owner)
		owner->Notify(event, this);
}

void TreeItem::CurrencyChanged( bool)
{
	flags |= INVALID;
}

void TreeItem::SetPosition( BPoint newpos)
{
	position = newpos;
}

void TreeItem::Draw( TreeList* owner)
{
	if ( !isHidden())
	{
		if (owner)
		{
// Reference dot...	owner->TView()->StrokeLine( position, position);
			if ( !owner->Boring())
				DrawIcon( owner);
			DrawLabel( owner);
			flags &= ~INVALID;
		}
#ifdef LOGGING
		else
		{
			::LogPuts( "DrawItem", LOGGER_WARNING, "Draw on ownerless child!"); 	 		
		}
#endif
	}
}
		
// Conditional redraw of this item (should be called
// after item level changes)
void TreeItem::Invalidate( TreeList* owner)
{
	// Only of interest if I have an owner and I am not hidden
	if ( owner && isInvalid() && !isHidden())
	{
		BRect inv_rect( 0, 
		                position.y, 
		                4000.0, 
		                position.y + Height() - 1);	
		owner->TView()->Invalidate( inv_rect);
	}
}
		

void TreeItem::DrawLabel( TreeList* owner)
{
	if ( label)
	{
		BView* view = owner->TView();
		// Recalculate some of the geometry
		font_info info;
		view->GetFontInfo( &info);
	
		float text_offset = (Height() - info.ascent)/2.0 + info.ascent;
		
		BPoint location(position);
		float h = Height();

		if ( Icon() && !owner->Boring())
		{
			// Make sure I move further over for Icon
			location.x += owner->ItemIndent() + (h - 1)/2.0 + owner->BoxIndent() + 4;
		}
		else
		{
			// Iconless items move the text back a bit
			location.x += owner->ItemIndent() + owner->BoxIndent();
		}
		
		location.y += text_offset;
		view->DrawString( label, location);

		BRect text_rect(
			location.x,
			location.y - (info.ascent + info.leading),
			view->PenLocation().x,
			location.y + info.descent
		);

		if ( isSelected())
		{
			view->InvertRect( text_rect);
		}
		
	}
}

void TreeItem::DrawIcon( TreeList* owner)
{
	const BBitmap* icon = Icon();
	
	if (icon)
	{ 
		BView* view = owner->TView();

		BPoint location( position.x + owner->ItemIndent() + owner->BoxIndent(),
		                 position.y + ((Height() - 1) / 2));

		TreeList::DrawCenteredImage( view, location, icon);
	}		
}

int TreeItem::HitTest( TreeList* owner, BPoint where)
{
	// Perform simple reject
	if ( (where.y < position.y) 
	  || (where.y > (position.y + Height()))
	  || (where.x < position.x)
	   )
		return NOHIT;


	if ( HasChildren())
	{
		// float edge = (position.x + owner->ItemIndent()  + owner->BoxIndent()) - (Height() / 2.0);
		float edge = position.x + owner->ItemIndent();
		// Anything between position and indent - halfheight is considered the box...
		if ( (where.x > position.x) && (where.x < edge))
			return PLUSMINUS_BOX;
	}
	
	// Anything else must be a text box hit
	return LABEL_BOX;
} 

TreeItem* TreeItem::ItemAt( BPoint p)
{
	// Is point over my rectangle?
	if ( (p.y > position.y) && (p.y < (position.y + Height())))
		return this;

	if ( !isCollapsed() && HasChildren())
	{
		// Spin over children and hittest them
		TreeItem* child = children;
	
		while( child)
		{
			TreeItem* item = child->ItemAt( p);
			
			if ( item)
				return item;
			
			child = child->sibling;
		}	
	}
	
	return 0;
}		

bool TreeItem::isHidden() const
{
	TreeItem* aparent = parent;
	
	while( aparent)
	{
		if (aparent->isCollapsed())
			return TRUE;
			
		aparent = aparent->parent;
	}
	
	return FALSE; 	
}
