//	TextHandler.cpp

#include "DataHandler.h"
#include "Datatypes.h"
#include "DataStreams.h"
#include <InterfaceDefs.h>
#include <Debug.h>

#include "Logger.h"

char handlerName[] = "PCX";
long handlerVersion = 100;
char handlerInfo[] = "PCX Handler by Andy Philpotts";

#define PCX_FORMAT		'PCXx'
#define PCX_HEADER_SIZE 128
#define PCX_PALETTE_SIZE ((256*3) + 1)


class CharStream {
		enum { BUFFSIZE = 2000 };
		
	public:
		CharStream( DStream& stream);
	
		int GetChar() 
		{ 
			if (here < amount) 
				return buffer[here++];
			else
				return GetMore();
		}
		
		int GetMore();

	private:
		int here;
		int amount;
		bool end_reached;
		unsigned char buffer[BUFFSIZE];
		DStream& stream;
};


CharStream::CharStream( DStream& str) :
	here(0),
	amount(0),
	stream(str),
	end_reached(false)
{
	long rd = BUFFSIZE;
	long e = stream.Read( buffer, rd);
	
	if (e)
	{
		// Assume error means end of file
		end_reached = true;
		return;
	}
	
	if ( rd == BUFFSIZE)
	{
		amount = BUFFSIZE;
	}
	else
	{
		end_reached = true;
		amount = rd;
	}
		
}

int CharStream::GetMore()
{
	here = 0;
	amount = 0;
	
	if (end_reached)
		return -1;
		
	long rd = BUFFSIZE;
	long e = stream.Read( buffer, rd);
	
	if (e)
	{
		// Assume error means end of file
		end_reached = true;
		return -1;
	}
	
	if ( rd == BUFFSIZE)
	{
		amount = rd;
	}
	else
	{
		end_reached = true;
		amount = rd;
	}
	
	if (amount)
	{
		here = 1;
		return buffer[0];
	}
	else
	{
		return -1;
	}
}

// Access 16 bit integer at "offset" in "buffer"
#define GETINT( buffer, offset) (buffer[offset] + (buffer[offset+1]<<8)) 

Format inputFormats[] = {
	{	PCX_FORMAT,	DATA_BITMAP,
		0.3,	0.5,
		"image/pcx",
		"PCX (PC Paintbrush)"
	},
	{	0,	0,
		0,	0,
		"",
		""
	}
};

Format outputFormats[] = {
	{	0,	0,
		0,	0,
		"",
		""
	}
};

enum PCXType {
	PCX_INVALID,
	PCX_MONO,
	PCX_4COLOR,
	PCX_8COLOR,
	PCX_16COLOR,
	PCX_CGA,
	PCX_EGA,
	PCX_256COLOR,
	PCX_24BIT
};

static enum PCXType type1[4] = { PCX_MONO, PCX_4COLOR, PCX_8COLOR, PCX_16COLOR };
static enum PCXType type2[4] = { PCX_CGA, PCX_INVALID, PCX_INVALID, PCX_INVALID };
static enum PCXType type4[4] = { PCX_EGA, PCX_INVALID, PCX_INVALID, PCX_INVALID };
static enum PCXType type8[4] = { PCX_256COLOR, PCX_INVALID, PCX_24BIT, PCX_INVALID };

// "Standard" palettes
static rgb_color MONO_PALETTE[2] = {
	{ 0,0,0,0 },
	{ 255,255,255,0}
};

static rgb_color FOUR_COLOR_PALETTE[4] = {
	{ 0,0,0,0 },
	{ 255,0,0,0},
	{ 0,255,0,0},
	{ 0,0,255,0}
};

static rgb_color EIGHT_COLOR_PALETTE[8] = {
	{ 0,0,0,0 },
	{ 255,0,0,0},
	{ 0,255,0,0},
	{ 255,255,0,0},
	{ 0,0,255,0 },
	{ 255,0,255,0},
	{ 0,255,255,0},
	{ 255,255,255,0}
};

static rgb_color SIXTEEN_COLOR_PALETTE[16] = {
	{ 0,0,0,0},
	{ 127,0,0,0},
	{ 0,127,0,0},
	{ 127,127,0,0},
	{ 0,0,127,0 },
	{ 127,0,127,0},
	{ 0,127,127,0},
	{ 127,127,127,0},
	{ 0,0,0,0},
	{ 255,0,0,0},
	{ 0,255,0,0},
	{ 255,255,0,0},
	{ 0,0,255,0 },
	{ 255,0,255,0},
	{ 0,255,255,0},
	{ 255,255,255,0}
};
		
static int GetBITMRowBytes( long rowBytes, color_space colors)
{
	switch (colors) {
	case B_COLOR_8_BIT:
		/* rowBytes are OK */ break;
	case B_RGB_32_BIT:
		rowBytes = rowBytes*4; break;
	case B_RGB_16_BIT:
		rowBytes = rowBytes*2; break;
	case B_MONOCHROME_1_BIT:
		rowBytes = rowBytes/8; break;
	case B_GRAYSCALE_8_BIT:
		/* rowBytes are OK */ break;
	default:
		/* illegal format - no rowBytes */
		rowBytes = 0; break;
	}
	return rowBytes;
}



long Identify(
	DStream &			inSpec,
	const Format *		format,
	BMessage *			/* ioExtension */,
	DATAInfo &			outInfo,
	ulong				outType)
{
	StartLogging();
	Trace( "PCX Identify");
	
	if (outType && (outType != DATA_BITMAP))
		return DATA_NO_HANDLER;

	unsigned char	data[32];

	long size = 32;
	long e = inSpec.Read(data, size);
	if (!e && (size < 32))
		e = DATA_NO_HANDLER;
	if (e < 0)
		return e;

	// Check for identifying info in the first 32 bytes
	if ( data[0] != 10)				// ZSoft flag
		return DATA_NO_HANDLER;
		
	if (data[1] > 5)				// Version 0..5
		return DATA_NO_HANDLER;
		
	if (data[2] != 1)				// PCX Run-length encoding
		return DATA_NO_HANDLER;
		
	const Format * inFormat = format;
	format = &inputFormats[0];

	if (inFormat)
	{
		if (inFormat->type != format->type)
			/*	It's not what we expected	*/
			return DATA_NO_HANDLER;
	}
	
	outInfo.formatType = format->type;
	outInfo.formatGroup = format->t_cls;
	outInfo.formatQuality = format->quality;
	outInfo.formatCapability = format->capability;
	strcpy(outInfo.formatName, format->formatName);
	strcpy(outInfo.MIMEName, format->MIME);
	
	Trace( "PCX Identify successful");

	return B_OK;
}

// Read a single (unsigned) byte, return -1 on EOF or error
static int ReadByte( DStream& inStream)
{
	// Read a byte
	unsigned char ch;
	long rd = 1;
	long e = inStream.Read( &ch, rd);
	
	if ( e || (rd != 1))
	{
		// Assume error means end of file
		return -1;
	}
	
	return ch;	
}

// Fill a buffer with RLE encoded data
static int ReadRleBytes( unsigned char* buffer, CharStream& inStream, int maxcount)
{
	int needed = maxcount;
	while( needed)
	{
		int ch = inStream.GetChar();
	
		if ( ch == -1)
			return maxcount - needed;	
	
		// Check top two bits of character to see if this is a repeat count
		if ( (ch & 0xc0) == 0xc0)
		{
			// Set count and get next byte...
			int runlength = ch & 0x3f;
			ch = inStream.GetChar();
			if ( ch == -1)
				return maxcount - needed;
				
			if ( needed >= runlength)
			{
				// Transfer according to runlength, not finished yet
				::memset( buffer, ch, runlength);
				buffer += runlength;
				needed -= runlength;
			}
			else
			{
				// Overrun, fill and trash the rest
				::memset( buffer, ch, needed);
				return maxcount;	
			}				
		}
		else
		{
			// Normal case, not RLE	
			*buffer++ = ch;
			needed--;
		}
	}
	
	return maxcount;
}

static long MakePalette( DStream& inStream,
						 PCXType pcx_type,
                         unsigned char* rgb_ptr, 
                         int& palette_size, 
                         rgb_color*& palette, 
                         bool& custom_palette)
{
	long e;
	
	palette_size = 0;
	palette = 0;
	custom_palette = false;
	
	switch(pcx_type)
	{
		case PCX_MONO:
			palette_size = 2;
			palette = MONO_PALETTE;
			break;
		
		case PCX_4COLOR:
			palette_size = 4;
			palette = FOUR_COLOR_PALETTE;
			break;
				
		case PCX_8COLOR:
			palette_size = 8;
			palette = EIGHT_COLOR_PALETTE;
			break;
		
		case PCX_16COLOR:
			palette_size = 16;
			palette = SIXTEEN_COLOR_PALETTE;
			break;
			
		case PCX_EGA:
			custom_palette = true;
			palette_size = 16;
			palette = new rgb_color[palette_size];
			// Fill from header palette
			for(int i = 0; i<16; i++)
			{
				rgb_color color;
				
				color.red = *rgb_ptr++;
				color.green = *rgb_ptr++;
				color.blue = *rgb_ptr++;
				color.alpha = 0;
				palette[i] = color;

			}
			break;
			
		case PCX_256COLOR:
		{
			unsigned char pcx_palette[ PCX_PALETTE_SIZE];
			custom_palette = true;
			palette_size = 256;
			
			// Grab pcx palette from end of file, this means reading to end (bummer)
			long stream_size = inStream.Size();
			if ( stream_size < (PCX_HEADER_SIZE + PCX_PALETTE_SIZE))
				return DATA_NO_HANDLER;
				
			e = inStream.Seek( stream_size - PCX_PALETTE_SIZE);
			if (e)
				return e;
				
			long size = PCX_PALETTE_SIZE;
			e = inStream.Read( pcx_palette, size);
			if (size < PCX_PALETTE_SIZE)
				e = DATA_NO_HANDLER;
		
			if (e)
				return e;	

			// Move back to whence we came!
			e = inStream.Seek( PCX_HEADER_SIZE);
			if (e)
				return e;
			
			if ( pcx_palette[0] != 12)
				return DATA_NO_HANDLER;

			palette = new rgb_color[palette_size];
			
			rgb_ptr = &(pcx_palette[1]);
			for(int i = 0; i<256; i++)
			{
				rgb_color color;
				
				color.red = *rgb_ptr++;
				color.green = *rgb_ptr++;
				color.blue = *rgb_ptr++;
				color.alpha = 0;
				palette[i] = color;
			}
		}
		break;
		
		case PCX_24BIT:
			palette_size = 0;
			palette = 0;
			break;
	}
	
	return 0;	
}


static int GetBits( unsigned char* base, int index, int numbits)
{
	// Masks to extract suitable number of bits...
	static int masks[] = { 0, 1, 3, 7, 15, 31, 63, 127, 255 };
	
	int bitoffset = index * numbits;
	int x = base[bitoffset / 8];
	return (x >> (bitoffset % 8)) & masks[ numbits];
}

static long WriteByPalette( DStream& inStream,
                            DStream& outStream, 
                            rgb_color* palette,
                            int palette_size,
                            int width,
                            int height,
                            int bmp_bytes_per_line,
                            int bytes_per_plane,
                            int number_of_planes,
                            int bits_per_pixel)
{
	CharStream stream(inStream);
	
	long e = 0;
	int pcx_bytes_per_line = number_of_planes * bytes_per_plane;
	
	unsigned char*  bmp_data = (unsigned char*)malloc( bmp_bytes_per_line);
	unsigned char*  pcx_data = (unsigned char*)malloc( pcx_bytes_per_line);
		
	for ( int row = 0; row < height; row++)
	{
		unsigned char* pcx_ptr = pcx_data;
		unsigned char* bmp_ptr = bmp_data;
			
		int bytes_read = ReadRleBytes( pcx_data, stream, pcx_bytes_per_line);
		
		if ( bytes_read != pcx_bytes_per_line)
		{
			e = DATA_NO_HANDLER;
			break;
		}
			
		for( int col = 0; col < width; col++)
		{
			rgb_color color;
			int index = 0;
			
			for( int plane = 0; plane < number_of_planes; plane++)
				index = (index << bits_per_pixel) 
				       + GetBits( pcx_data + (plane*bytes_per_plane), col, bits_per_pixel);

			if (index < palette_size)			
				color = palette[index];
			else
				color = palette[0];
				
			// Buffer one lines worth of 24bit color, in bitmap order
			*bmp_ptr++ = color.blue; 		
			*bmp_ptr++ = color.green; 		
			*bmp_ptr++ = color.red; 		
			*bmp_ptr++ = 0; // alpha 		
		}
		e = outStream.Write(bmp_data, bmp_bytes_per_line);
		if (e)
			break;
			
	}
	free( pcx_data);
	free( bmp_data);
	return e;
}

// The 256 color type is simple to decode as a special case...
static long Write256( DStream& inStream,
                      DStream& outStream, 
                      rgb_color* palette,
                      int width,
                      int height,
                      int bmp_bytes_per_line,
                      int bytes_per_plane)
{
	CharStream stream( inStream);
	long e = 0;
	int pcx_bytes_per_line = bytes_per_plane;
	
	unsigned char*  bmp_data = (unsigned char*)malloc( bmp_bytes_per_line);
	unsigned char*  pcx_data = (unsigned char*)malloc( pcx_bytes_per_line);
		
	for ( int row = 0; row < height; row++)
	{
		unsigned char* pcx_ptr = pcx_data;
		unsigned char* bmp_ptr = bmp_data;
			
		int bytes_read = ReadRleBytes( pcx_data, stream, pcx_bytes_per_line);
		
		if ( bytes_read != pcx_bytes_per_line)
		{
			e = DATA_NO_HANDLER;
			break;
		}
			
		for( int col = 0; col < width; col++)
		{
			rgb_color color;
			int index = *pcx_ptr++;

			color = palette[index];
				
			// Buffer one lines worth of 24bit color, in bitmap order
			*bmp_ptr++ = color.blue; 		
			*bmp_ptr++ = color.green; 		
			*bmp_ptr++ = color.red; 		
			*bmp_ptr++ = 0; // alpha 		
		}
		
		e = outStream.Write(bmp_data, bmp_bytes_per_line);
		if (e)
			break;
			
	}
	free( pcx_data);
	free( bmp_data);
	return e;
}


static long Write24( DStream& inStream, 
                     DStream& outStream, 
                     int width,
                     int height,
                     int bmp_bytes_per_line,
                     int pcx_bytes_per_plane)
{
	CharStream stream( inStream);
	long e = 0;
	
	int pcx_bytes_per_line = 3 * pcx_bytes_per_plane;
	
	unsigned char*  bmp_data = (unsigned char*)malloc( bmp_bytes_per_line);
	unsigned char*  pcx_data = (unsigned char*)malloc( pcx_bytes_per_line);
		
	for ( int row = 0; row < height; row++)
	{
		unsigned char* pcx_ptr = pcx_data;
		unsigned char* bmp_ptr = bmp_data;
			
		int bytes_read = ReadRleBytes( pcx_data, stream, pcx_bytes_per_line);
		
		if ( bytes_read != pcx_bytes_per_line)
		{
			e = DATA_NO_HANDLER;
			break;
		}
			
		for( int col = 0; col < width; col++)
		{
			// Buffer one lines worth of 24bit color, in bitmap order
			*bmp_ptr++ = pcx_ptr[ 2 * pcx_bytes_per_plane]; 		
			*bmp_ptr++ = pcx_ptr[ pcx_bytes_per_plane]; 		
			*bmp_ptr++ = *pcx_ptr; 		
			*bmp_ptr++ = 0; // alpha
			pcx_ptr++; 		
		}
			
		e = outStream.Write(bmp_data, bmp_bytes_per_line);
		if (e)
			break;
			
	}
	free( pcx_data);
	free( bmp_data);
	
	return e;
}


long Translate(
	DStream &			inStream,
	const DATAInfo &	inInfo,
	BMessage *			/* ioExtension */,
	ulong				outFormat,
	DStream &			outStream)
{
	Trace( "PCX Translate");

	PCXType pcx_type;
	long e;
	/*	Check that we handle input and output types */

	if (inInfo.formatType != PCX_FORMAT) 
		return DATA_NO_HANDLER;

	if (outFormat == 0)
		outFormat = DATA_BITMAP;
		
	if (outFormat != DATA_BITMAP) 
		return DATA_NO_HANDLER;

	unsigned char hdr[PCX_HEADER_SIZE];		
	DATABitmap bmp;
	
	// Clear header to get predictable results
	::memset( &bmp, 0, sizeof(bmp));
	
	long size = PCX_HEADER_SIZE;
	e = inStream.Read(hdr, size);
	if (size < PCX_HEADER_SIZE)
		e = DATA_NO_HANDLER;
		
	if (e)
		return e;

	Trace( "PCX Header read OK");
		
	bmp.magic = DATA_BITMAP;

	int pcx_bytes_per_plane = GETINT( hdr, 66); 
	
	int pleft = GETINT( hdr, 4);
	int ptop = GETINT( hdr, 6);
	int pright = GETINT( hdr, 8);
	int pbottom = GETINT( hdr, 10);
	
	bmp.bounds.left = pleft;
	bmp.bounds.top = ptop;
	bmp.bounds.right = pright;
	bmp.bounds.bottom = pbottom;

	Trace2( "PCX Left = %d, Top = %d", pleft, ptop);
	Trace2( "PCX Right = %d, Bottom = %d", pright, pbottom);
	

	int width = pright - pleft + 1;
	int height = pbottom - ptop + 1;

	Trace2( "PCX Width = %d, Height = %d", width, height);
	
	// Figure out what type of PCX map we are dealing with...
	int number_of_planes = hdr[65];
	int bits_per_pixel = hdr[3];
	
	// Validate number of planes
	if ( (number_of_planes < 1) || (number_of_planes > 4))
		return DATA_NO_HANDLER;
		
	// Figure out the PCX type
	switch(bits_per_pixel)
	{
		case 1:
			pcx_type = type1[number_of_planes-1];
			break;
			
		case 2:
			pcx_type = type2[number_of_planes-1];
			break;
			
		case 4:
			pcx_type = type4[number_of_planes-1];
			break;
			
		case 8:
			pcx_type = type8[number_of_planes-1];
			break;
			
		default:
			return DATA_NO_HANDLER;
	} 

	Trace1( "PCX type = %d", pcx_type);
	Trace1( "PCX bytes per plane %d", pcx_bytes_per_plane);
	
	switch(pcx_type)
	{
		case PCX_MONO:
		case PCX_4COLOR:		
		case PCX_8COLOR:
		case PCX_16COLOR:
		case PCX_EGA:
		case PCX_256COLOR:
		case PCX_24BIT:
		    // Cuz' I'm lazy, always do 32 bit!
			bmp.colors = B_RGB_32_BIT;
			break;
		
		case PCX_CGA:           // Note that CGA is unsupported!
		case PCX_INVALID:
			return DATA_NO_HANDLER;
	}
	
	int rowBytes = GetBITMRowBytes( width, bmp.colors);
	bmp.rowBytes = (rowBytes+3)&~3;
	bmp.dataSize = bmp.rowBytes*height;
	e = outStream.Write(&bmp, sizeof(bmp));
	if (e)
		return e;

	Trace( "PCX Bitmap header written");
		
	// Thats the header done, now figure how to setup the palette...
	int palette_size;
	rgb_color* palette;
	bool custom_palette;
	
	e = MakePalette( inStream, pcx_type, &(hdr[16]), palette_size, palette, custom_palette);
	if (e)
		return e;

	
	switch(pcx_type)
	{
		case PCX_24BIT:
			{
				Trace( "PCX Writing 24 bit");
				e = Write24( inStream, outStream, width, height, bmp.rowBytes, pcx_bytes_per_plane);
			}
			break;
		
		case PCX_256COLOR:
			{
				Trace( "PCX Writing 256 palette");
				e = Write256( inStream, outStream, palette, width, height, bmp.rowBytes,
		    	              pcx_bytes_per_plane);
			}
			break;
		
		default:
			{
				Trace( "PCX Writing by palette");
				e = WriteByPalette( inStream, outStream, palette, palette_size, width, height, bmp.rowBytes,
		    	 	                pcx_bytes_per_plane, number_of_planes, bits_per_pixel);
			}
			break;
	}
	
	if (custom_palette)
		delete [] palette;
	
	if (!e)
		Trace( "PCX Translate done, and I'm happy");
	else
		Trace( "PCX Translate done, but I'm pissed");
		
	return e;							
}


