#ifndef LOGGER_H
#define LOGGER_H

// This is used to conditionally compile
// usages of this module.
#define LOGGING

///////////////////////////////////////////////////////////////////
// Severities are used to help filter messages, otherwise they
// have no affect on what happens
//
// Guidelines for usage are:
// CRITICAL - program will probably terminate due to this
// SEVERE - nasty but recoverable, such as file not found  
// WARNING - something maybe wrong, maybe running low of disk space
// INFORMATIONAL - I have started/stopped/changed mode, whatever!
///////////////////////////////////////////////////////////////////

enum LoggerSeverity {
	LOGGER_CRITICAL,
	LOGGER_SEVERE,
	LOGGER_WARNING,
	LOGGER_INFORMATIONAL
};

///////////////////////////////////////////////////////////////////
// If you want to get your messages through call this first
// This connects to the Logger program (if it is running).
//
// With the default parameter, if the connect fails (which will
// happen if Logger is not already running) then all subsequent
// messages will be disposed of (efficiently!).  Of course the
// aplication can try "StartLogging" again.
//
// With retry, if the connection fails then every message will
// retry the connect.  This will be a little slower, but means
// you can start (and restart!) the Logger program anytime.
//
// Only call this function from one thread at a time, it updates
// global memory!
//
///////////////////////////////////////////////////////////////////

void StartLogging( bool retry = FALSE);

///////////////////////////////////////////////////////////////////
// Log a message, if StartLogging() has not been called, or
// the Logger program has not been started message will be
// quietly lost.
//
// The "const char*" parameters point to NIL terminated strings
// which are copied before the function returns.
///////////////////////////////////////////////////////////////////

// Send a simple string
void LogPuts( 
	const char* src,    // Your application or module name here
    LoggerSeverity sev, // see above
    const char* txt	    // Any message, any length
);

// Send a formatted string
void LogPrintf(
	const char* src,    // Your application or module name here
    LoggerSeverity sev, // see above
    const char* format,	// sprintf format string
    ...					// variable args, as per sprintf
);   

// Logging for beginners...
#define Trace(x) LogPuts( "Trace", LOGGER_INFORMATIONAL, (x))
#define Trace1(x,a) LogPrintf( "Trace", LOGGER_INFORMATIONAL, (x), a)
#define Trace2(x,a,b) LogPrintf( "Trace", LOGGER_INFORMATIONAL, (x), (a), (b))
#define Trace3(x,a,b,c) LogPrintf( "Trace", LOGGER_INFORMATIONAL, (x), (a), (b), (c))

///////////////////////////////////////////////////////////////////
// Give logging a chance to close gracefully
///////////////////////////////////////////////////////////////////

void StopLogging();

#endif
