/*
 * java.lang.System.c
 *
 * Copyright (c) 1996 Systems Architecture Research Centre,
 *		   City University, London, UK.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@sarc.city.ac.uk>, February 1996.
 */

#define __jtypes_h
#include "config.h"
#include <stdio.h>
#include <assert.h>
#include <stdlib.h>
#include <string.h>
#if defined(HAVE_SYS_UTSNAME_H)
#include <sys/utsname.h>
#endif
#if defined(HAVE_SYS_TIME_H)
#include <sys/time.h>
#endif
#if defined(HAVE_PWD_H)
#include <pwd.h>
#endif
#if defined(HAVE_UNISTD_H)
#include <unistd.h>
#endif
#if defined(HAVE_DIR_H)
#include <dir.h>
#endif
#include "../../kaffevm/gtypes.h"
#include <native.h>
#include "system.h"
#include "defs.h"
#include "java.lang.stubs/System.h"
#include "../../kaffevm/itypes.h"
#include "../../kaffevm/constants.h"
#include "../../kaffevm/access.h"
#include "../../kaffevm/classMethod.h"

static char cwdpath[MAXPATHLEN];

IMPORT(struct itypes) types[];

/*
 * Copy one part of an array to another.
 */
void
java_lang_System_arraycopy(struct Hjava_lang_System* none, struct Hjava_lang_Object* src, jint srcpos, struct Hjava_lang_Object* dst, jint dstpos, jint len)
{
	char* in;
	char* out;
	int elemsz;

	/* Must be some array type */
	assert(dst->mtable->class->sig[0] == '[');
	elemsz = TYPE_SIZE_C(dst->mtable->class->sig[1]);

	len *= elemsz;
	srcpos *= elemsz;
	dstpos *= elemsz;

	in = &((char*)(src+1))[srcpos];
	out = &((char*)(dst+1))[dstpos];
	for (; len > 0; len--) {
		*out++ = *in++;
	}
}

/*
 * Initialise system properties to their defaults.
 */
struct Hjava_util_Properties*
java_lang_System_initProperties(struct Hjava_lang_System* none, struct Hjava_util_Properties* p)
{
	int r;
	char* jhome;
	char* cpath;
	char* dir;
#if defined(HAVE_SYS_UTSNAME_H)
	struct utsname system;
#endif
#if defined(HAVE_PWD_H)
	struct passwd* pw;
#endif

	/* Define PUT to be a call to the put method taking two strings
	 * as arguments.  call_java_method does all the majic looking up
	 * here.
	 */
#define	PUT(k,s) \
	do_execute_java_method(0, (void*)p, "put", \
	  "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;", \
	  0, false, \
	  makeJavaString(k, strlen(k)), \
	  makeJavaString(s, strlen(s)))

	/* Now use it to add the default properties:
	 * java.version		Java version number
	 * java.vendor          Java vendor specific string
	 * java.vendor.url      Java vendor URL
	 * java.home            Java installation directory
	 * java.class.version   Java class version number
	 * java.class.path      Java classpath
	 * os.name              Operating System Name
	 * os.arch              Operating System Architecture
	 * os.version           Operating System Version
	 * file.separator       File separator ("/" on Unix)
	 * path.separator       Path separator (":" on Unix)
	 * line.separator       Line separator ("\n" on Unix)
	 * user.name            User account name
	 * user.home            User home directory
	 * user.dir             User's current working directory
	 */

	PUT("java.version", kaffe_version);
	PUT("java.vendor", kaffe_vendor);
	PUT("java.vendor.url", kaffe_vendor_url);

	jhome = getenv(KAFFEHOME);
	if (jhome == 0) {
		jhome = ".";
	}
	PUT("java.home", jhome);

	PUT("java.class.version", kaffe_class_version);

	cpath = getenv(KAFFECLASSPATH);
	if (cpath == 0) {
		cpath = ".";
	}
	PUT("java.class.path", cpath);

	PUT("file.separator", file_seperator);
	PUT("path.separator", path_seperator);
	PUT("line.separator", line_seperator);

#if defined(HAVE_GETCWD)
	dir = getcwd(cwdpath, MAXPATHLEN);
#elif defined(HAVE_GETWD)
	dir = getwd(cwdpath);
#else
	dir = 0;	/* Cannot get current directory */
#endif
	if (dir == 0) {
		dir = ".";
	}
	PUT("user.dir", dir);
	free(dir);

#if defined(HAVE_SYS_UTSNAME_H)
	/* Setup system properties */
	r = uname(&system);
	assert(r >= 0);
	PUT("os.name", system.sysname);
	PUT("os.arch", system.machine);
	PUT("os.version", system.version);
#else
	PUT("os.name", "Unknown");
	PUT("os.arch", "Unknown");
	PUT("os.version", "Unknown");
#endif
#if defined(HAVE_PWD_H)
	/* Setup user properties */
	pw = getpwuid(getuid());
	assert(pw != 0);
	PUT("user.name", pw->pw_name);
	PUT("user.home", pw->pw_dir);
#else
	PUT("user.name", "Unknown");
	PUT("user.home", "Unknown");
#endif

	return (p);
}

/*
 * Return current time.
 */
jlong
java_lang_System_currentTimeMillis(struct Hjava_lang_System* none)
{
	jlong tme;
#if defined(HAVE_GETTIMEOFDAY)
	struct timeval tm;
	gettimeofday(&tm, 0);
	tme = (((jlong)tm.tv_sec * 1000LL) + ((jlong)tm.tv_usec / 1000LL));
#else
	tme.jl = 0;
	tme.jh = 0;
#endif
	return (tme);
}
